#ifndef __CMVISION_H__
#define __CMVISION_H__
/*=========================================================================
    CMVision.cc
  -------------------------------------------------------------------------
    API definition for the CMVision real time Color Machine Vision library
  -------------------------------------------------------------------------
    Copyright 1999               #### ### ### ## ## ## #### ##  ###  ##  ##
    James R. Bruce              ##    ####### ## ## ## ##   ## ## ## ######
    School of Computer Science  ##    ## # ## ## ## ##  ### ## ## ## ## ###
    Carnegie Mellon University   #### ##   ##  ###  ## #### ##  ###  ##  ##
  -------------------------------------------------------------------------
    This source code is distributed "as is" with absolutely no warranty.
    See LICENSE, which should be included with this distribution.
  =========================================================================*/

#include <string.h>
#include <stdio.h>

/*
 - Color segmentation
   - load / save
   - set new values
   - process frame
 - Connected Regions
   - RLE
   - merge
   - extract blobs
   - separate blobs
   - sort blobs
   - merge blobs
 - Blob merging
   - merge by area occupied

Options format: (# RGB merge name)
[Colors]
(00,00,00) 0.95 'Orange'
(00,00,00) 0.00 'Pink'
(00,00,00) 0.00 'Red'
(00,00,00) 0.00 'DarkBlue'
(00,00,00) 0.00 'Blue'

[Thresholds]
(<lo>:<hi>,<lo>:<hi>,<lo>:<hi>)
(<lo>:<hi>,<lo>:<hi>,<lo>:<hi>)
(<lo>:<hi>,<lo>:<hi>,<lo>:<hi>)
*/

#define CMV_COLOR_LEVELS  256
#define CMV_MAX_COLORS     32






#define CMV_MAX_RUNS     4096
#define CMV_MAX_REGIONS  5120
#define CMV_MIN_AREA        4

#define CMV_NONE ((unsigned)(-1))

#ifndef NULL
#define NULL (0)
#endif

/*
#define CMV_COLOR_X(p) ((p.y1+p.y2)/2)
#define CMV_COLOR_Y(p) (p.u)
#define CMV_COLOR_Z(p) (p.v)
*/

struct rgb{
  unsigned char red,green,blue;
};
#define YUV 1
#if YUV
struct yuv422{
  unsigned char y1,u,y2,v;
};
#else
typedef rgb yuv422;
#endif

class CMVision{
public:
  struct region{
    int color;          // id of the color
    int area;           // occupied area in pixels
    int x1,y1,x2,y2;    // bounding box (x1,y1) - (x2,y2)
    float cen_x,cen_y;  // centroid
    int sum_x,sum_y;    // temporaries for centroid calculation
    region *next;       // next region in list
  };

  struct rle{
    unsigned color;     // which color(s) this run represents
    int length;         // the length of the run (in pixels)
    int parent;         // run's parent in the connected components tree
  };

  struct color_info{
    rgb color;
    char *name;
    double merge;
    int y_low,y_high;
    int u_low,u_high;
    int v_low,v_high;
  };

protected:
  unsigned yclass[CMV_COLOR_LEVELS];
  unsigned uclass[CMV_COLOR_LEVELS];
  unsigned vclass[CMV_COLOR_LEVELS];

  region region_table[CMV_MAX_REGIONS];
  region *region_list[CMV_MAX_COLORS];
  int region_count[CMV_MAX_COLORS];

  rle rmap[CMV_MAX_RUNS];

  color_info colors[CMV_MAX_COLORS];
  int width,height;
  unsigned *map;

protected:
// Private functions
  void classifyFrame(yuv422 *img,unsigned *map);
  int  encodeRuns(rle *out,unsigned *map);
  void connectComponents(rle *map,int num);
  int  extractRegions(region *reg,rle *rmap,int num);
  int  separateRegions(region *reg,int num);
  region *sortRegionListByArea(region *list,int passes);
  void sortRegions(int max_area);

public:
  CMVision();
  ~CMVision() {close();}

  bool initialize(int nwidth,int nheight);
  bool loadOptions(char *filename);
  bool saveOptions(char *filename);
  void close();

  bool testClassify(rgb *out,yuv422 *image);
  bool getThreshold(int color,
         int &y_low,int &y_high,
         int &u_low,int &u_high,
         int &v_low,int &v_high);
  bool setThreshold(int color,
         int y_low,int y_high,
         int u_low,int u_high,
         int v_low,int v_high);
  char *getColorName(int color)
    {return(colors[color].name);}

  bool processFrame(yuv422 *image);
  int numRegions(int color_id);
  region *getRegions(int color_id);
};

#endif // __CMVISION__H_
