#include <stdlib.h>

#include "driver.hh"
#include "debug.h"
#include <parselanguage.h>

/* Magic numbers for VW control loop */
#define v_lin 7.0
#define t_lin 0.3
#define v_rot 10.0
#define t_rot 0.1

Driver::Driver()
  : wheels(0),
    started(0),
    drivespeed(0.2),
    turnspeed(1.5)
{
  curpos.x = 0;
  curpos.y = 0;
  curpos.phi =0;
}

Driver::~Driver()
{
  stop();
  if (0 != wheels)
    VWRelease(wheels);
}

bool
Driver::initialize(void)
{
  wheels = VWInit(VW_DRIVE, 1);
  if (0 != wheels)
    return start();
  else
    return false;
}

const PositionType &
Driver::getPosition(void)
{
  if (0 != wheels)
    VWGetPosition(wheels, &curpos);
  return curpos;
}

void
Driver::setPosition(const PositionType &pos)
{
  if (0 != wheels)
    VWSetPosition(wheels, pos.x, pos.y, pos.phi);
}

/****f* Driver::getPosition
 * DESCRIPTION
 *   Return current position in mm.
 ****/
const vec2d 
Driver::getPosition(radians *heading)
{
  PositionType pos = getPosition();
  *heading = pos.phi;
  return vec2d(pos.x*1000, pos.y*1000);
}

/****f* Driver::setPosition
 * DESCRIPTION
 *   Set current position in mm.
 ****/
void
Driver::setPosition(const vec2d &pos, radians heading)
{
  if (0 != wheels)
    VWSetPosition(wheels, pos.x/1000.0, pos.y/1000.0, heading);
}

meter
Driver::getRemaining(void)
{
  if (0 != wheels)
    return VWDriveRemain(wheels);
  else
    return -1;
}

bool
Driver::isStalled(void)
{
  if (0 == wheels)
    return false; /* No wheels, must be standing */
  return 1 == VWStalled(wheels);
}

bool
Driver::isStanding(void)
{
  if (0 == wheels)
    return true; /* No wheels, must be standing */
  return VWDriveDone(wheels);
}

void 
Driver::waitUntilStanding(void)
{
  if (0 != wheels)
    VWDriveWait(wheels);
}

void
Driver::turn(radians rotation)
{
  if (0 != wheels)
    {
      VWDriveTurn(wheels, rotation, turnspeed);
#if DEBUG
      LCDPrintf("turn %5.2f, %5.2f\n", rotation, turnspeed);
#endif
      mydebug("turn %5.2f, %5.2f", rotation, turnspeed);
    }
}

void
Driver::straight(meter distance)
{
  if (0 != wheels)
    {
      VWDriveStraight(wheels, distance, drivespeed);
#if DEBUG
      LCDPrintf("straight %5.2f, %5.2f\n", distance, drivespeed);
#endif
      mydebug("straight %5.2f, %5.2f", distance, drivespeed);
    }
}

void
Driver::curve(meter distance, radians rotation)
{
  if (0 != wheels)
    {
      VWDriveCurve(wheels, distance, rotation, drivespeed);
#if DEBUG
      LCDPrintf("curve %5.2f, %5.2f, $5.2f\n", distance, rotation, drivespeed);
#endif
      mydebug("curve %5.2f, %5.2f, $5.2f", distance, rotation, drivespeed);
    }
}

void
Driver::path(PositionType targets[], int numtargets)
{
  
}

bool
Driver::start(void)
{
  if (0 < started++)
    return true;
  
  if (0 != wheels &&
      0 == VWStartControl(wheels, v_lin, t_lin , v_rot, t_rot))
    {
      mydebug("Starting wheel controller");
      return true;
    }

  return false;
}

void
Driver::stop(void)
{
  started--;
  if (0 >= started && 0 != wheels)
    {
      VWStopControl(wheels);
      mydebug("Stopping wheel controller");
    }
}
