/*
 * Author: Petter Reinholdtsen <pere@td.org.uit.no>
 * Date:   2000-08-28
 */

#include "envstatus.hh"
#include "fieldmap.hh"
#include "predictor.hh"

static int world2pic(const Picture *pic, const int p);
static radians world2picHeading(const Picture *pic, radians r);
static vec2d world2picPos(const Picture *pic, const vec2d &worldpos);

/* Range of battery status graph */
#define POWER_MAX ((1<<10) - 1)
#define POWER_MIN (550)

/****f* soccer-pere/EnvStatus::drawRadioStatus
 * SYNOPSIS
 *   void EnvStatus::drawRadioStatus(Picture *pic, int color)
 * DESCRIPTION
 *   Draw current radio/language status (available, robots connected etc.)
 ****/
void
EnvStatus::drawRadioStatus(Picture *pic, int color)
{
  int leftstart = pic->width - 8 - 7;
  int topstart = pic->height - 19;
  if (0 != channel)
    {
      picproc_drawLine(pic,
		       leftstart    , topstart + 7,
		       leftstart + 7, topstart + 7,
		       color);
      picproc_drawLine(pic,
		       leftstart + 4, topstart + 7,
		       leftstart + 4, topstart,
		       color);
      RadioStatus s;
      if (0 == RADIOGetStatus(&s))
	{
	  for (BYTE b = 0; b < 6; b++)
	    if (s.active[b])
	      picproc_drawPixel(pic,
				leftstart + b, topstart + 8,
				color);
	  picproc_drawPixel(pic,
			    leftstart + s.master, topstart + 9,
			    color);
	}
    }
  else
    picproc_drawBox(pic, 
		    leftstart,     topstart,
		    leftstart + 7, topstart + 9,
		    color);
}


/****f* soccer-pere/EnvStatus::drawCompass
 * SYNOPSIS
 *   void EnvStatus::drawCompass(Picture *pic, int color)
 * DESCRIPTION
 *   If a compass was detected, display the current compass heading in
 *   the lower left corder of the picture.
 ****/
void
EnvStatus::drawCompass(Picture *pic, int color)
{
  const int radius = 8;
  const int bbottom = 9;
  picproc_drawCircle(pic, radius+1, pic->height - bbottom - radius -1, radius,
		     color);
  if (use_compass)
    picproc_drawArrow(pic, radius+1, pic->height - bbottom - radius -1,
		      radius, 4, world2picHeading(pic,compassval), color);
}

/****f* soccer-pere/EnvStatus::drawBattery
 * SYNOPSIS
 *   void EnvStatus::drawBattery(Picture *pic, int color)
 * DESCRIPTION
 *   Draw the current battery status as a bar on the right side of the
 *   picture.
 ****/
void
EnvStatus::drawBattery(Picture *pic, int color)
{
  const int btop = 8, bbottom = 10;
  int current = (pic->height - btop - bbottom) *
    (battery_status - POWER_MIN) / (POWER_MAX - POWER_MIN);
  picproc_drawBox(pic, pic->width-4, btop,
		  pic->width-2, pic->height-bbottom, color);
  picproc_drawLine(pic, pic->width-3, pic->height-current-bbottom,
		   pic->width-3, pic->height-bbottom-1, color);
}

/****f* soccer-pere/EnvStatus::drawGameStatus
 * SYNOPSIS
 *   void EnvStatus::drawGameStatus(Picture *pic, int color)
 * DESCRIPTION
 *   Draw the current game status as a symbol in the upper right
 *   corner.
 ****/
void
EnvStatus::drawGameStatus(Picture *pic, int color)
{
  int top = 0, bottom = 6, left = 120, right = 127;
  switch (gamestatus)
    {
    case pause:
      picproc_drawBox(pic,  left,  top, right, bottom, color);
      picproc_drawLine(pic, left+1,  top+1, right-1, bottom-1, color);
      picproc_drawLine(pic, right-1, top+1, left+1,  bottom-1, color);
      break;
    case pregame:
      picproc_drawBox(pic,  left,  top, right, bottom, color);
      break;
    case play:
      picproc_drawArea(pic, left,  top, right, bottom, color);
      break;
    }
}

/****f* soccer-pere/EnvStatus::drawField
 * SYNOPSIS
 *   void EnvStatus::drawField(Picture *pic, int color)
 * DESCRIPTION
 *   Draw the edges of the soccer field in the picture.
 ****/
void
EnvStatus::drawField(Picture *pic, int color)
{
  unsigned int wallcount;
  const segment2d *walls = FieldMap::getMap(&wallcount);

  for (size_t i = 0; i < wallcount; i++)
    {
      vec2d p1 = world2picPos(pic, walls[i].start);
      vec2d p2 = world2picPos(pic, walls[i].end);
      picproc_drawLine(pic, p1.x, p1.y, p2.x, p2.y, color);
    }
}

/****f* soccer-pere/EnvStatus::drawPSD
 * SYNOPSIS
 *   void EnvStatus::drawPSD(Picture *pic, const vec2d &mypos,
 *                           radians myheading, int color)
 * DESCRIPTION
 *   Draw the PSD readings in the picture using the given color.  Draw
 *   dots where the distance reading indicated a reflection.
 ****/
void
EnvStatus::drawPSD(Picture *pic, const vec2d &mypos,
		   radians myheading, int color)
{
#if !defined(__mc68000__)
  return;
#endif

  for (int i = 0; i < NUM_EDGES; i++)
    {
      if (edges[i].cached && edges[i].timestamp > lastlocate &&
	  PSD_OUT_OF_RANGE != edges[i].dist)
	{
	  vec2d end = world2picPos(pic, edges[i].wpos);
	  picproc_drawPixel(pic, end.x,  end.y,  color);
	}
      
    }
}

void
EnvStatus::drawFriends(Picture *pic, int color)
{
  const int wradius = world2pic(pic, ROBOT_RADIUS);
  for (unsigned int id = 0; id < 6; id++)
    {
      if (id != my_global_id)
	{
	  ObjectStatus *obj = robots[id].getStatus(OSGetCount());
	  vec2d wpos = world2picPos(pic, obj->pos);
	  picproc_drawCircle(pic, wpos.x, wpos.y, wradius, color);
	}
    }
}
/****f* soccer-pere/EnvStatus::draw
 * SYNOPSIS
 *   void EnvStatus::draw(Picture *pic)
 * DESCRIPTION
 *   Draw the current enviroment status into pic.
 ****/
void
EnvStatus::draw(Picture *pic)
{
  int white = 0xffffffff;
  picture_clear(pic);

  drawField(pic, white);
  drawBattery(pic, white);
  drawCompass(pic, white);
  drawRadioStatus(pic, white);

  /* Draw my position and heading */
  radians heading;
  int wradius = world2pic(pic, ROBOT_RADIUS);
  vec2d mypos = getMyCurrentPos(&heading);
  vec2d wmypos = world2picPos(pic, mypos);
  picproc_drawCircle(pic, wmypos.x, wmypos.y, wradius, white);

  /* My heading */
  picproc_drawArrow(pic, wmypos.x, wmypos.y, wradius*3, 1,
		    world2picHeading(pic,heading), white);

  /* Camera heading */
  picproc_drawArrow(pic, wmypos.x, wmypos.y, wradius*2, 1,
		    world2picHeading(pic,heading+pancamera.getHeading()),
		    white);
  

  /* Draw where I'm going */
  vec2d dest = world2picPos(pic, destpoint);
  picproc_drawLine(pic, wmypos.x, wmypos.y, dest.x, dest.y, white);

  drawFriends(pic, white);

  /* Draw current PSD reflections */
  drawPSD(pic, mypos, heading, white);

  drawGameStatus(pic, white);

  /* Draw ball position */
  if (isBallKnown())
  {
    vec2d ballpos = world2picPos(pic, getBallCurrentPos());
    picproc_drawCircle(pic, ballpos.x, ballpos.y, 1, white);
  }
}

int
world2pic(const Picture *pic, const int p)
{
  double scale = ( (double) pic->height-16) / FIELD_WIDTH_X;
  return (int) (p * scale);
}

/****f* soccer-pere/world2picHeading
 * SYNOPSIS
 *   radians world2picHeading(const Picture *pic, const radians r)
 * DESCRIPTION
 *   Convert read-world heading to picture heading
 ****/
radians
world2picHeading(const Picture *pic, radians r)
{
  return r-M_PI_2;
}

/****f* soccer-pere/world2picPos
 * SYNOPSIS
 *   vec2d world2picPos(const Picture *pic, const vec2d &worldpos)
 * DESCRIPTION
 *   Convert read-world position to picture position
 ****/
vec2d
world2picPos(const Picture *pic, const vec2d &worldpos)
{
  double scale = ( (double) pic->height-16) / FIELD_WIDTH_X;

  return vec2d((int)(pic->width/2.0 + worldpos.y*scale), 
	       (int)(pic->height-8 - (FIELD_WIDTH_X/2.0 - worldpos.x)*scale));
}
