/*
 * Author: Petter Reinholdtsen <pere@td.org.uit.no>
 * Date:   2000-07-29
 */

#include "envstatus.hh"

#include <picproc.h>
#include <camera.h>

#include <string.h>
#include <limits.h>

#include "debug.h"
#include "soccerfield.h"
#include "fieldmap.hh"
#include "blocktimer.hh"
#include "angles.h"
#include "hdtcamera.h"
#include "hdtpsdpos.h"
#include "soccer.hh"
#include "miscmacro.h"
#include "input.h"
#include <assert.h>

/* Which servo is panning the camera? */
#define CAMSERVO SERVO11

/* Analog/dialog convert channel with battery status */
#define AD_CHANNEL_POWER (1)

#define POWER_LIMIT 900

/* The PSDs and their heading */
EnvStatus::psdinfo_t *EnvStatus::psdinfo = NULL;

EnvStatus::EnvStatus(int myid,
		     lang_channel _channel,
		     Camera *_camera,
		     Driver *_driver)
  : my_global_id(myid),
    doLocalization(false),
    edge(0),
    use_vision(false),
    use_compass(false),
    compassoffset(0.0),
    destpoint(vec2d(0,0)),
    visinfo(0),
    gamestatus(pause),
    goal_at_top(true),
    lastupdate(0),
    lastlocate(0),
    lastlocatetry(0),
    battery_status(0),
    channel(_channel),
    camera(_camera),
    driver(_driver)
{
  memset(&ballinfo, 0, sizeof(ballinfo));
  memset(&edges[0], 0, sizeof(edges));

  if (NULL == psdinfo)
    {
      static EnvStatus::psdinfo_t psdrobotinfo[] =
      {
	{PSD_LEFT,       {0.0, 0.0,  M_PI_2}, vec2d(0,0)},
	{PSD_FRONT,      {0.0, 0.0,       0}, vec2d(0,0)},
	{PSD_FRONT2,     {0.0, 0.0,       0}, vec2d(0,0)},
	{PSD_RIGHT,      {0.0, 0.0, -M_PI_2}, vec2d(0,0)},
	{PSD_FRONTLEFT,  {0.0, 0.0,  M_PI_4}, vec2d(0,0)},
	{PSD_FRONTRIGHT, {0.0, 0.0, -M_PI_4}, vec2d(0,0)},
	{PSD_BACK,       {0.0, 0.0,  M_PI  }, vec2d(0,0)}
      };
      psdinfo = &psdrobotinfo[0];
    }

  Picture *snapshot = camera->capture();

  initPSD();
  initPanCamera(snapshot);
  initCompass();

  if (0 != camera)
    {
      LCDClear();
      LCDPrintf("Use vision?");
      LCDMenu("NO", " ", " ", "YES");
      if (KEY4 == inputGet())
	use_vision = true;
    }
  else
    {
      LCDClear();
      LCDPrintf("Camera missing!");
      OSSleep(100);
    }

  if (use_vision)
    {
      mydebug("playball: lcd init");
      picture_init(&lcd, 128,64,pix_bitmap, 0, &lcd_data[0], sizeof(lcd_data));
      /* Initialize libvision */
      mydebug("playball: snapshot init");
      mydebug("playball: VIS_Init_var");
      visinfo = VIS_Init_var(&lcd, snapshot);
      if (0 == visinfo)
	{
	  use_vision = false;
	  mydebug("libvision init failed");
	}
      else
	{
	  mydebug("playball: VIS_Init_calibration");
	  VIS_Init_calibration(visinfo, Camera_capture, (void*)camera);
	}
    }

  for (int i=0; i< MAXEYE; i++)
    timeoffset[i] = INT_MAX;
  updateTimeoffset();

  update();
}

bool
EnvStatus::initialize(void)
{
  if (use_vision)
    {
      mydebug("playball: VIS_Init_color_classes");
      VIS_Init_color_classes(visinfo);
      mydebug("playball: VIS_Init_distance");
      Picture *snapshot = camera->capture();
      VIS_Init_distance(visinfo, snapshot);

#if 0
      /* clear to test if someone is pissing in memory */
      memset(&visinfo->Dist_tables->Dist_dist[0][0][0], 0,
	     sizeof(visinfo->Dist_tables->Dist_dist));
      memset(&visinfo->Dist_tables->Dist_coord[0][0][0][0], 0,
	     sizeof(visinfo->Dist_tables->Dist_coord));
#endif
    }
  return true;
}

EnvStatus::~EnvStatus()
{
  if (use_compass)
    {
      mydebug("Release compass");
      COMPASSStop();
      COMPASSRelease();
    }

  mydebug("Release PSD");
  PSDStop();
  PSDRelease();

  mydebug("Release vision library");
  if (visinfo)
    VIS_Release_var(visinfo);
}

/****f* soccer-pere/EnvStatus::initCompass
 * SYNOPSIS
 *   void EnvStatus::initCompass(void)
 * DESCRIPTION
 *   Initialize and calibrate compass.  Calibration is done by turning
 *   360 degrees on the spot.
 ****/
void
EnvStatus::initCompass(void)
{
  /* Initialize compass */
  if (0 != COMPASSInit(COMPASS))
    {
      mydebug("Compass init failed");
      return;
    }

  if (0 != COMPASSStart(TRUE))
    {
      mydebug("Compass start failed");
      return;
    }
  else
    {
      LCDClear();
      LCDPrintf("Compass calibration\n\n");
      LCDPrintf("Turing 360 deg\n\n");
      LCDPrintf("Put robot down\n");

      COMPASSStop();

      OSSleep(200);

      driver->start();
      COMPASSCalibrate(1);
      driver->turn(M_PI);
      driver->waitUntilStanding();

      /* Give the compass time to settle down */
      OSSleep(100);
      COMPASSCalibrate(1);

      driver->turn(M_PI);
      driver->waitUntilStanding();
      driver->stop();

      LCDClear();
      LCDPrintf("Compass init\n\n");
      LCDPrintf("done");
      OSSleep(100);

      if (0 != COMPASSStart(TRUE))
	{
	  mydebug("Compass start 2 failed");
	  return;
	}
      else
	{
	  int start = OSGetCount();
	  while (start + 100 > OSGetCount() && !COMPASSCheck())
	    ;
	  if (start + 100 <= OSGetCount())
	    use_compass = true;
	  else
	    mydebug("compass check timed out");
	}
    }
}
void
EnvStatus::initPanCamera(Picture *pic)
{
  BYTE center = 255/2;
  radians rotation = deg2rad(160.0);
  radians view = deg2rad(50.0);
  DeviceSemantics servo = CAMSERVO;
  camera_pan_type *cpdata = (camera_pan_type*)HDTFindEntry(CAMERA, CAMERAPAN);
  if (NULL != cpdata && 0 == cpdata->version)
    {
      servo  = cpdata->servo;
      center = cpdata->center;
      rotation  = cpdata->rotation;
      view   = cpdata->view;
      mydebug("found pancam HDT %d %d %5.2f %5.2f",
	      servo, center, rad2deg(rotation), rad2deg(view));
    }
  
  if (pancamera.initialize(servo))
    {
      if (use_vision)
	center = visinfo->CAM_offset->alpha;

      pancamera.setCenter(center);
      pancamera.setRotation(rotation);

      pancamera.setHeading(0.0);
    }
  pixel_angle = view / pic->width;
}

/****f* soccer-pere/EnvStatus::initPSD
 * DESCRIPTION
 *   Initialize all PSD, find distance range and start continuous
 *   sensoring.
 ****/
void
EnvStatus::initPSD(void)
{
  DeviceSemantics mask = 0;
  mydebug("playball: Init PSD");

  for (int i=0; i < NUM_PSD; i++)
    {
      psdval[i].dist = PSD_OUT_OF_RANGE;
      psdval[i].min = 0;
      psdval[i].max = 500;

      psdval[i].handle = PSDInit(psdinfo[i].semantic);
      if (0 != psdval[i].handle)
	{
	  mask |= psdval[i].handle;
	  getPSDRange(psdinfo[i].semantic, &psdval[i].min, &psdval[i].max);
	  psdpos_type *ppt =
	    (psdpos_type*)HDTFindEntry(PSDPOS, psdinfo[i].semantic);
	  if (NULL != ppt)
	    psdinfo[i].hdtpos = ppt->pos;
	  psdinfo[i].pos.x = (int)(psdinfo[i].hdtpos.x * 1000);
	  psdinfo[i].pos.y = (int)(psdinfo[i].hdtpos.y * 1000);

	  mydebug("PSD %d range %d-%d mm pos=(%dx%d)",
		  i, psdval[i].min, psdval[i].max,
		  (int)(psdinfo[i].hdtpos.x*1000),
		  (int)(psdinfo[i].hdtpos.y*1000));
	}
      else
	mydebug("PSDInit(%d) for psd %d failed", psdinfo[i].semantic, i);
    }
  if (0 != PSDStart(mask, 1))
    mydebug("PSDStart(0x%x, 1) failed", (int)mask);
}

/****f* soccer-pere/EnvStatus::updateTimeoffset
 * SYNOPSIS
 *   void EnvStatus::updateTimeoffset(void)
 * DESCRIPTION
 *   Send ping packets to all the robots on the net, to initialize
 *   time syncronization.  The time difference is calculated when the
 *   pong packets are received.
 ****/
void
EnvStatus::updateTimeoffset(void)
{
#if 1
  return;
#else
  /* XXX This seems to do bad things to the program, disable! */
  int count = 0;
  RadioStatus curstatus;
  RADIOGetStatus(&curstatus);
  for (int i = 0; i < MAXEYE; i++)
    {
      if (curstatus.active[i] && i != OSMachineID())
	{
	  langEventPing event;
	  event.timestamp = OSGetCount();
	  event.type = langTypePing;
	  event.to = i;
	  lang_send(channel, (langEvent*)&event);
	  count++;
	}
    }
#endif
}

/****f* soccer-pere/EnvStatus::lang_handler_cb
 * SYNOPSIS
 *   static int EnvStatus::lang_handler_cb(langEvent *event, void *ref)
 * DESCRIPTION
 *   Callback to receive language events.  Cast ref to (EnvStatus *)
 *   and pass the event to  the real handler EnvStatus::lang_handler.
 * SEE ALSO
 *   EnvStatus::lang_handler()
 ****/
int
EnvStatus::lang_handler_cb(langEvent *event, void *ref)
{
  EnvStatus *s = (EnvStatus *)ref;
  return s->lang_handler(event);
}

/****f* soccer-pere/EnvStatus::lang_handler
 * DESCRIPTION
 *   Receive and handle all language events.
 ****/
int
EnvStatus::lang_handler(langEvent *event)
{
  int when;
  if (0 == event)
    return -1;

  if (timeoffset[event->any.from] == INT_MAX)
    updateTimeoffset();

  when = event->any.timestamp + timeoffset[event->any.from];

  switch (event->type)
    {
    case langTypePong:
      {
	langEventPong *e = (langEventPong *)event;
	int now = OSGetCount();
	int diff = now + e->time;
	if (diff >= 1000) /* More then 10 second old ping packet */
	  {
	    mydebug("pong %d old from %d", diff, e->from);
	    break;
	  }
	if (0 < e->from && e->from < MAXEYE)
	  timeoffset[e->from] = e->timestamp - (diff)/2;
	else
	  mydebug("Bogus from %d", e->from);
      }
      break;
    case langTypeCommand:
      {
	langEventCommand *cmd = (langEventCommand *)event;
	switch (cmd->cmd)
	  {
	    /* XXX We should get pregame messages as well */
	  case langCmdStart:
	    setGameStatus(play);
	    break;
	  case langCmdWait:
	    setGameStatus(pause);
	    break;
	  }
	break;
      }
    case langTypePlayMode:
      {
	langEventPlayMode *mode = (langEventPlayMode *)event;
	switch (mode->mode)
	  {
	  case langModePlayBlue:
	    goal_at_top = true;
	    break;
	  case langModePlayYellow:
	    goal_at_top = false;
	    break;

	  case langModeKickOffBlue:
	  case langModeKickOffYellow:
	    setGameStatus(play);
	    break;

	  case langModeBeforeKickOff:
	  case langModeHalfTime:
	  case langModeTimeUp:
	    setGameStatus(pause);
	    break;

	  case langModeGoalBlue:
	  case langModeGoalYellow:
	    setGameStatus(pause);
	    break; /* XXX Not really handled at the moment */
	  }
	break;
      }
    case langTypeTrack:
      {
        langEventTrack *track = (langEventTrack *)event;
        mydebug("Track: %d %c%c %1d %4dx%5d/%3d", track->timestamp,
		track->type, track->name, track->num, track->x, track->y,
		track->heading, track->velocity);
        switch(track->name)
          {
          case langNameBall:
	    /* XXX Ignore heading and speed info! */
	    setBallPos(when, track->from, track->x, track->y);
            break;
          case langNameFriend:
	    if (track->num == my_global_id && 0 == track->from)
	      { /* Trust overhead tracking system */
		vec2d pos;
		radians heading;
		pos.x = track->x;
		pos.y = track->y;
		heading = lang_angle2rad(track->heading);
		driver->setPosition(pos, heading);
	      }
	    if (track->num < 6)
	      {
		vec2d pos(track->x, track->y);
		robots[track->num].newPosition(&pos, when);
	      }
            break;
	  case langNameEnemy: /* Ignore these for now */
	    break;
          }
        break;
      }
    default:
      mydebug("Got unhandled event %d", event->type);
      break;
    }
  return 0;
}

void
EnvStatus::setGameStatus(GameStatus s)
{
  gamestatus = s;
  if (pause == s)
    driver->straight(0.0); /* Stop driving */
};
/****f* soccer-pere/dummyBallPos
 * SYNOPSIS
 *   langEvent *dummyBallPos(long now)
 * DESCRIPTION
 *   Place ball on a circle around the center point.  Only used in the
 *   simulator to test the drive routines.
 ****/
langEvent *
dummyBallPos(long now)
{
  static langEventTrack track;
  static long lasttime = now;
  const int circleradius = 7 * ROBOT_RADIUS;
  if (lasttime + 1000 < now)
    {
      track.timestamp = now;
      track.type = langTypeTrack;
      track.name = langNameBall;
      track.num = 0;

      track.x = (int) (circleradius * sin(now/M_PI));
      track.y = (int) (circleradius * cos(now/M_PI));
      track.heading = 0;
      track.velocity = 0;;

      lasttime = now;
      return (langEvent *)&track;
    }
  else
    return 0;
}

/****f* soccer-pere/EnvStatus::updateCameraView
 * SYNOPSIS
 *   void EnvStatus::updateCameraView(int timestamp)
 * DESCRIPTION
 *   Fetch one snapshot from the camera, and process it to extract the
 *   visible objects.  Store the extracted information.
 *****/
void
EnvStatus::updateCameraView(int timestamp)
{
  /* This will block until a new frame is ready */
  Picture *snapshot = camera->capture();

  if (0 != snapshot)
    {
      VIS_color_region *region = VIS_Find_classes(visinfo, snapshot);
      if (VIS_BALL == region[0].color)
	{
	  VIS_distance distance;
	  /* found ball, update position. */
	  if (0 == VIS_Get_position(visinfo, &distance, VIS_BALL))
	    {
	      if (distance.d_row < 0 || 1000 < distance.dist)
		{
		  /* Behind me, as seen from the camera - yeah right */
		  mydebug("Saw: ball (%dx%d)-(%dx%d): (%3dx%3d)%d",
			  region[0].coord.left, region[0].coord.top,
			  region[0].coord.right, region[0].coord.bot,
			  distance.d_col, distance.d_row, distance.dist);
		}
	      else
		{
		  /* distance.d_row, distance.d_col, distance.dist */
		  vec2d lballpos(distance.d_row, -distance.d_col);

		  LCDSetPrintf(1,2,"(%3dx%3d) ", distance.d_col,
			       distance.d_row);

		  radians robotangle;
		  radians camheading = pancamera.getHeading();
		  vec2d mypos = getMyCurrentPos(&robotangle);

		  radians rotangle = camheading + robotangle;
		  vec2d ballpos = mypos + lballpos.rotate(rotangle);

		  mydebug("Saw ball from (%dx%d) %5.1f + %5.1f at (%dx%d) - "
			  "rotating %5.1f -> (%dx%d)",
			  mypos.x, mypos.y, rad2deg(robotangle),
			  rad2deg(camheading),
			  distance.d_col, distance.d_row, rad2deg(rotangle),
			  ballpos.x, ballpos.y);

		  if (!pancamera.isRotating() && driver->isStanding())
		    setBallPos(timestamp, OSMachineID(), ballpos.x, ballpos.y);
		  else
		    mydebug("Ignoring new ball pos");
		}
	    }
	  else
	    { /* distance tables not calibrated, calculate angle to ball */
	      int center = (region[VIS_BALL].coord.right +
			    region[VIS_BALL].coord.left)>>1;
	      ball_angle = pixel_angle * (snapshot->width>>1 - center);
	    }
	}
    }
}

/****f* soccer-pere/EnvStatus::updatePSD
 * SYNOPSIS
 *   void EnvStatus::updatePSD(int timestamp)
 * DESCRIPTION
 *   Collect PSD distance measurement, and store the values.
 *****/
void
EnvStatus::updatePSD(const vec2d &mypos, radians myheading, int timestamp)
{
  static int lastPSDtime = 0;
  int availablePSDtime;
  int i;

  availablePSDtime = PSDGet(0);
  if (availablePSDtime == lastPSDtime)
    return;
  lastPSDtime = availablePSDtime;

  for (i = 0; i < NUM_PSD; i++)
    {
      psdval[i].dist = PSDGet(psdval[i].handle);

      edges[edge].timestamp = timestamp;
      edges[edge].dist = psdval[i].dist;
      edges[edge].origin = mypos + psdinfo[i].pos.rotate(myheading);
      edges[edge].min = psdval[i].min;
      edges[edge].max = psdval[i].max;

      radians heading = rad2rad(myheading + psdinfo[i].hdtpos.phi);
      edges[edge].heading = heading;

      vec2d reading;
      if (PSD_OUT_OF_RANGE != psdval[i].dist)
	reading = vec2d(psdval[i].dist,0);
      else
	reading = vec2d(psdval[i].max,0);
      edges[edge].wpos = edges[edge].origin + reading.rotate(heading);
      edges[edge].cached = true;

      edge = (edge + 1) % NUM_EDGES;
    }
}

/****f* soccer-pere/EnvStatus::uploadPSD
 * SYNOPSIS
 *   void EnvStatus::uploadPSD(void)
 * DESCRIPTION
 *   Upload the current PSD readings on the serial port.
 *****/
void
EnvStatus::uploadPSD(void)
{
  radians myheading;
  vec2d mypos = getMyCurrentPos(&myheading);

  LCDClear();
  LCDPrintf("Upload PSD val\n");
  LCDMenu("Send", " ", " ", "END");
  if (KEY4 == inputWait(ANYKEY))
    return;

  mydebug("mypos %d %d %f", mypos.x, mypos.y, myheading);

  for (int i = 0; i < NUM_EDGES; i++)
    {
      mydebug("psdval %5d %d (%5d,%5d) %f %4d %4d %5d %5d",
	      edges[i].timestamp,
	      edges[i].dist,
	      edges[i].origin.x, edges[i].origin.y,
	      edges[i].heading,
	      edges[i].min, edges[i].max,
	      edges[i].wpos.x, edges[i].wpos.y);
    }
}

void
EnvStatus::updateCompass(void)
{
  int val = COMPASSGet();
  if (-1 == val)
    return; /* Ignore */
  compassval = rad2rad(deg2rad((float)val) + compassoffset);
}

void
EnvStatus::setCompassHeading(radians heading)
{
  compassoffset = 0.0;
  
  updateCompass();
  compassoffset = rad2rad(heading - compassval);
}

void
EnvStatus::setMyPosition(const PositionType &pos)
{
  driver->setPosition(pos);
  lastlocate = lastupdate;
}

/****f* soccer-pere/EnvStatus::update
 * SYNOPSIS
 *   void EnvStatus::update(void)
 * DESCRIPTION
 *   Collect information about the current environmet status, and
 *   merge information into a consistent world status.
 *****/
void
EnvStatus::update(void)
{
  /* Process radio messages */
  if (0 < lang_messageWaiting(channel))
    lang_receive(channel, lang_handler_cb, this);

  long now = OSGetCount();

  /* Where am I */
  radians myheading;
  vec2d mypos = getMyCurrentPos(&myheading);

  updateBattery(now, POWER_LIMIT);

  /* Check distance sensors */
  if (PSDCheck())
    updatePSD(mypos, myheading, now);

  if (use_compass)
    updateCompass();

  /*
   * Average current VW heading with compass reading, giving compass
   * reading 1/10 weight.
   */
  if (driver->isStanding())
    {
      vec2d mv = vec2d(10000, 0).rotate(myheading);
      vec2d cv = vec2d(1000, 0).rotate(compassval);
      vec2d nv = mv + cv;
      myheading = nv.heading();
      driver->setPosition(mypos, myheading);
    }

  /*
   * Do position correction and self localization only when the ball
   * position is unknown.
   */

  /* every 60 second selflocate and then every 10 second until successfull */
  if (doLocalization && !isBallKnown() && 
      lastlocate    + 6000 < now &&
      lastlocatetry + 1000 < now)
    {
      bool islocatedok = false;
      vec2d delta;
      radians deltaphi;
      unsigned int wallcount;
      const segment2d *walls = FieldMap::getMap(&wallcount);
      islocatedok = selflocate(lastlocate,
			       walls, wallcount, &edges[0], NUM_EDGES,
			       mypos, myheading, &delta, &deltaphi);
      lastlocatetry = now;
      if (islocatedok)
	{
	  mydebug("Moving from (%dx%d) %5.2f", mypos.x, mypos.y,
		  rad2deg(myheading));
	  driver->setPosition(mypos + delta, myheading + deltaphi);
	  mypos = driver->getPosition(&myheading);
	  mydebug("Moving to (%dx%d) %5.2f", mypos.x, mypos.y,
		  rad2deg(myheading));
	  lastlocate = now;
	}
    }

  if (use_vision)
    updateCameraView(now);

  sendMyPosition(now);

#if 0
#if !defined(__mc68000__)
  /* Send dummy ball postition to test program in simulator */
  lang_handler(dummyBallPos(now));
#endif
#endif
  lang_sendFlush(channel);

  lastupdate = now;
}

/****f* soccer-pere/RobotControl::sendPowerStatus
 * SYNOPSIS
 *   void sendPowerStatus(lang_channel ch, int timestamp, short int level)
 * DESCRIPTION
 *   Broadcast the current battery power status.
 ****/
void
EnvStatus::sendPowerStatus(long timestamp, short int level)
{
  static long laststamp = 0;
  if (0 == channel)
    return;

  if (laststamp + 6000 > timestamp) /* Only send every 60 second */
    return;

  langEventPowerStatus event;
  event.timestamp = timestamp;
  event.type = langTypePowerStatus;
  event.to = BROADCAST;
  event.level = level;
  lang_send(channel, (langEvent *)&event);
  laststamp = timestamp;
}

/****f* soccer-pere/RobotControl::sendMyPosition
 * SYNOPSIS
 *   void EnvStatus::sendMyPosition(long timestamp)
 * DESCRIPTION
 *   Broadcast the current position and heading
 ****/
void
EnvStatus::sendMyPosition(long timestamp)
{
  static long lastbroadcast = 0;
  if (0 == channel)
    return;

  if (lastbroadcast + 500 > timestamp) /* Only send every 5 second */
    return;

  langEventTrack event;
  event.timestamp = timestamp;
  event.type = langTypeTrack;
  event.name = langNameFriend;
  event.num = my_global_id;
  event.to = BROADCAST;

  radians myheading;
  vec2d mypos = getMyCurrentPos(&myheading);
  event.x = mypos.x;
  event.y = mypos.y;
  event.heading = lang_rad2angle(myheading);
  event.velocity = 0;

  lang_send(channel, (langEvent *)&event);
  lastbroadcast = timestamp;
}

/****f* soccer-pere/EnvStatus::updateBattery
 * SYNOPSIS
 *   void EnvStatus::updateBattery(int limit)
 * DESCRIPTION
 *   Get current battery status, and broadcast it if it is too low.
 ****/
void
EnvStatus::updateBattery(long timestamp, int limit)
{
  battery_status = (short int)OSGetAD(AD_CHANNEL_POWER); /* 10 bit value */

  /* If we are running low on battery, notify everyone about our status. */
  if (battery_status < limit)
    sendPowerStatus(timestamp, battery_status);
}

void
EnvStatus::setBallPos(int when, BYTE from, int x, int y)
{
  ballinfo.x = y;
  ballinfo.y = y;
  balltime = when;

  if (from == OSMachineID())
    { /* From me - I must have seen it */
      /* Send updated ball and own position */
      langEventTrack event;
      event.to = BROADCAST;
      event.timestamp = when;
      event.type = langTypeTrack;

      /* myself */
      radians heading;
      vec2d mypos = getMyCurrentPos(&heading);

      event.name = langNameFriend;
      event.num = my_global_id;
      event.x = mypos.x;
      event.y = mypos.y;
      event.heading = lang_rad2angle(heading);
      event.velocity = 0; /* unknown */
      lang_send(channel, (langEvent*)&event);

      /* Ball */
      event.name = langNameBall;
      event.num = 0;
      event.x = x;
      event.y = y;
      event.heading = 0; /* unknown */
      event.velocity = 0; /* unknown */
      lang_send(channel, (langEvent*)&event);
    }
}

vec2d
EnvStatus::getMyCurrentPos(radians *heading)
{
  assert(NULL != driver);
  return driver->getPosition(heading);
}

/****f* soccer-pere/EnvStatus::isBallKnown
 * SYNOPSIS
 *   bool EnvStatus::isBallKnown(void)
 * DESCRIPTION
 *   Check if the position of the ball is known.
 * RETURN VALUE
 *   Return true if the ball position is less tne 10 seconds all, and
 *   false othervise.
 * SEE ALSO
 *   EnvStatus::getBallCurrentPos()
 ****/
bool
EnvStatus::isBallKnown(void)
{
  if (balltime + 1000 > OSGetCount())
    return true;
  else
    return false;
}

vec2d
EnvStatus::getBallCurrentPos(void)
{
  return vec2d(ballinfo.x, ballinfo.y);
}

vec2d
EnvStatus::getTheirGoalCenter(void)
{
  if (goal_at_top)
    return FieldMap::getBlueGoalCenter();
  else
    return FieldMap::getYellowGoalCenter();
}

vec2d
EnvStatus::getTheirGoalLeft(void)
{
  if (goal_at_top)
    return vec2d(-FIELD_GOAL_WIDTH/2, FIELD_WIDTH_Y/2);
  else
    return vec2d(FIELD_GOAL_WIDTH/2, -FIELD_WIDTH_Y/2);
}

vec2d
EnvStatus::getTheirGoalRight(void)
{
  if (goal_at_top)
    return vec2d(FIELD_GOAL_WIDTH/2, FIELD_WIDTH_Y/2);
  else
    return vec2d(-FIELD_GOAL_WIDTH/2, -FIELD_WIDTH_Y/2);
}

vec2d
EnvStatus::getOurGoalCenter(void)
{
  if (goal_at_top)
    return FieldMap::getYellowGoalCenter();
  else
    return FieldMap::getBlueGoalCenter();
}

void
EnvStatus::cameraSearch(void)
{ /* Pan from side to side when using vision */
  if (use_vision)
    {
      if (!pancamera.isRotating())
	{
	  int val = pancamera.getServo();
	  pancamera.setServo(val+25 % 255);
	}
    }
  else
    cameraTrack(VIS_BALL);
}

void
EnvStatus::cameraTrack(int object)
{
  if (VIS_BALL==object)
    {
      radians myheading;
      vec2d mypos         = getMyCurrentPos(&myheading);

      vec2d ballpos       = getBallCurrentPos();
      vec2d toball = ballpos - mypos;         /* From me to the ball */

      pancamera.setHeading(toball.heading() - myheading);
    }
}


/****f* soccer-pere/EnvStatus::setDestPoint
 * SYNOPSIS
 *   void EnvStatus::setDestPoint(const vec2d &v)
 * DESCRIPTION
 *   Update the position we are heading for, to make it possible to
 *   draw a line there.
 ****/
void
EnvStatus::setDestPoint(const vec2d &v)
{
  destpoint = v;
}

#ifdef TEST
#include <stdio.h>
#include <unistd.h>
#include "eyebotcamera.hh"

int
test_drawPolygon()
{
  int white = 0xffffffff;
  Picture *pic = picture_new(128,64,pix_bitmap);
  picture_clear(pic);
  
  drawPolygon(pic, fieldborder, ARRAYSIZE(fieldborder), white);

  FILE *fp = fopen("field.ppm", "w");
  if (NULL != fp)
    {
      picproc_pnmEncode(pic, (picproc_writer)write, fileno(fp));
      fclose(fp);
    }

  picture_delete(pic);
  pic = NULL;
  return 0;
}

int
main()
{
  int retval = 0;
  mydebug_init();
  lang_channel channel = lang_init();
  Driver driver;
  EyebotCamera camera;
  if (!camera.initialize(imagecolumns, imagerows, pix_rgb24) ||
      0 == channel)
    {
      mydebug("Init failed: channel=0x%x\n", channel);
      retval = 1;
    }
  else
    {
      /* EnvStatus status(0, channel, &camera, &driver); */
      BYTE screendata[128*64/8];
      Picture screen;
      picture_init(&screen, 128, 64, pix_bitmap, 0,
		   &screendata[0], sizeof(screendata));
      /* status.draw(&screen); */
      LCDPutImage(screen.data);
      inputWait(ANYKEY);
    }

  if (0 != channel)
    lang_release(channel);
  mydebug_release();

  test_drawPolygon();

  return retval;
}
#endif
