/*
 * Author: Petter Reinholdtsen <pere@td.org.uit.no>
 * Date:   2000-07-29
 */

#ifndef _envstatus_h
#define _envstatus_h

#include <eyebot.h>

#include <parselanguage.h>
#include <picture.h>
#include <vision.h>
#include <camera.hh>
#include "vec2d.hh"
#include "driver.hh"
#include "predictor.hh"
#include "panning.hh"
#include "segment2d.hh"
#include "psdbeam.h"
#include "evidencemap.hh"

/****c* soccer-pere/EnvStatus
 * DESCRIPTION
 *   Collector class for all global status information.  Where is the
 *   ball, where am I, where are the others, where should I go, what
 *   do I sense.
 ****/
class EnvStatus
{
public:
  /* PSD handles and last readings. */
  enum psdid_t
  {
    PSDLeft = 0,
    PSDFront,
    PSDRight,
    PSDFrontLeft,
    PSDFrontRight,
    PSDBack,
    NUM_PSD
  };
  enum { NUM_EDGES = NUM_PSD * 50 };

private:
  int my_global_id;
  bool doLocalization;

  struct psdinfo_t
  {
    DeviceSemantics semantic;
    PositionType hdtpos;      /* fetched from psdpos HDT entry */
    vec2d pos;                /* Recalculated to mm */
  };
  static psdinfo_t *psdinfo;
  typedef struct {
    PSDHandle handle;
    int dist;
    int min;
    int max;
  } psdval_t;
  psdval_t psdval[NUM_PSD];

  /* The last psd readings */
  int edge;
  psdbeam_t edges[NUM_EDGES];

  bool use_vision;

  bool use_compass;
  radians compassval;  /* Last compass reading including offset */
  radians compassoffset; /* Rotation of compass, relative to global heading */

  vec2d destpoint; /* Where are we heading */

  langEventTrack ballinfo;
  int balltime; /* when did we learn the ball position */
  int timeoffset[MAXEYE]; /* How do our clock relate to their clock */
  Predictor robots[12]; /* 0-5 = friends, 6-12 = foe */
#if 0
  myvector<langEventTrack> lastinfo;
#endif

  /* libvision info */
  Picture lcd;
  BYTE lcd_data[128*64/8];
  VIS_structs *visinfo;

#if defined(USE_EVIDENCEMAP)
  EvidenceMap emap;
#endif

  void initPSD(void);
  void initPanCamera(Picture *pic);
  void initCompass(void);
  void updateTimeoffset(void);
  void updateCameraView(int timestamp);
  void updatePSD(const vec2d &mypos, radians myheading, int timestamp);
  void updateBattery(long timestamp, int limit);
  void updateCompass(void);
  void sendPowerStatus(long timestamp, short int level);
  void sendMyPosition(long timestamp);
  void setBallPos(int when, BYTE from, int x, int y);
  static int lang_handler_cb(langEvent *event, void *ref);

  VIS_color_region *vis_region;
  radians pixel_angle;
  radians ball_angle;
public:
  int lang_handler(langEvent *event);
  enum GameStatus {
    pause,   /* Do not move */
    pregame, /* Position before kick-off */
    play     /* Start playing */
  };
  GameStatus gamestatus;

  bool goal_at_top;

  long lastupdate; /* Timestamp of last call to update() */
  long lastlocate;
  long lastlocatetry;

  short int battery_status;

  lang_channel channel;
  Camera *camera;
  Panning pancamera;
  Driver *driver;

  EnvStatus(int myid, lang_channel _channel, Camera *_camera, Driver *_driver);
  ~EnvStatus();

  bool initialize(void);

  void update(void);
  void draw(Picture *pic);

  void setGameStatus(GameStatus s);
  void setCompassHeading(radians heading);
  void setMyPosition(const PositionType &pos);

  vec2d getMyCurrentPos(radians *heading = 0);
  bool isBallKnown(void);
  vec2d getBallCurrentPos(void);

  VIS_color_region *getColorRegions(void) { return vis_region; }

  vec2d getTheirGoalCenter(void);
  vec2d getTheirGoalLeft(void);
  vec2d getTheirGoalRight(void);

  vec2d getOurGoalCenter(void);

  void setDestPoint(const vec2d &v);

  int getPSDVal(psdid_t psd) { return psdval[(unsigned int)psd].dist; };

  void cameraSearch(void);
  void cameraTrack(int object);

  void uploadPSD(void);
private:
  void drawBattery(Picture *pic, int color);
  void drawCompass(Picture *pic, int color);
  void drawField(Picture *pic, int color);
  void drawPSD(Picture *pic, const vec2d &mypos, radians myheading, int color);
  void drawGameStatus(Picture *pic, int color);
  void drawRadioStatus(Picture *pic, int color);
  void drawFriends(Picture *pic, int color);
};

#endif /* _envstatus_h */
