/*
 * Author: Petter Reinholdtsen <pere@td.org.uit.no>
 * Date:   2000-09-19
 */

#include "evidencemap.hh"
#include "blocktimer.hh"
#include "angles.h"
#include "miscmacro.h"

#include <picproc.h>

#include <stdio.h>
#include <unistd.h>
#include <assert.h>

/*
 * Draw soccer field
 */
void
EvidenceMap::drawInitialMap(void)
{
  TIMEBLOCK();
  int center = mapcols/2;
  int upper = FIELD_WIDTH_Y/(2*MAPRESOLUTION);
  int left  = FIELD_WIDTH_X/(2*MAPRESOLUTION);

  for (int row = center-upper; row < center+upper; row++)
    for (int column = center-left; column < center+left; column++)
      ltmap[row][column] = empty;
  
  for (int row = center-upper; row <= center+upper; row++)
    {
      ltmap[row][center-left] = occupied;
      ltmap[row][center+left] = occupied;
    }
  for (int column = center-left; column <= center+left; column++)
    {
      ltmap[center-upper][column] = occupied;
      ltmap[center+upper][column] = occupied;
    }
}

EvidenceMap::EvidenceMap(void)
{
  TIMEBLOCK();
  for (unsigned int i = 0; i < mapcols*mapcols; i++)
    ltmap[0][i] = unknown;
  for (stcurrent = 0; stcurrent < stmapcount; stcurrent++)
    clearSTMap();
  stcurrent = 0;
  drawInitialMap();
}

void
EvidenceMap::clearSTMap(void)
{
  TIMEBLOCK();
  for (unsigned int i = 0; i < mapcols*mapcols; i++)
    stmap[stcurrent][0][i] = unknown;
}

void
EvidenceMap::addSTMapCellValue(unsigned int stmapid, int x, int y,
			       cell_t value)
{
  cell_t old = stmap[stmapid][y][x];
  assert(sizeof(cell_t) < sizeof(long));
  long newval = old + value; /* Make sure we do not overflow int */
  newval /=2;
#if 0
  mydebug("update %d (%dx%d) %d %d = %d", stmapid, x, y, old, value, newval);
#endif
  assert(0 <= y && y < mapcols && 0 <= x && x < mapcols);
  stmap[stmapid][y][x] = newval;
}

/****f* EvidenceMap::addPSDLine
 * DESCRIPTION
 *   Update evidence map along the line from (c1,r1) to (c2,r2).
 ****/
void
EvidenceMap::addPSDLine(unsigned int stmapid, int c1, int r1,
			int c2, int r2, bool m)
{
  int x,y, deltax,deltay, xchange, ychange;
  int i, error;

  x=c1;      /* starting point */
  y=r1;
  error=0;
  deltax = c2-c1;  /* difference */
  deltay = r2-r1;

  if (deltax < 0)
    {
      xchange = -1;
      deltax = -deltax;
    }
  else
    xchange = 1;
  if (deltay < 0)
    {
      ychange = -1;
      deltay = -deltay;
    }
  else
    ychange = 1;

  if (deltax < deltay)
    {
      for (i=0; i < deltay+1; i++)
        {
	  addSTMapCellValue(stmapid, x,y, empty);
          y     += ychange;
          error += deltax;
          if (error > deltay)
            {
              x += xchange;
              error -= deltay;
            }
        }
    }
  else
    {
      for (i=0; i < deltax+1; i++)
        {
	  addSTMapCellValue(stmapid, x,y, empty);
          x     += xchange;
          error += deltay;
          if (error > deltax)
            {
              y += ychange;
              error -= deltax;
            }
        }
    }

  if (m)
    addSTMapCellValue(stmapid, c2,r2, occupied);
}

/****f* soccer-pere/EvidenceMap::addPSDReading
 * DESCRIPTION
 *   Add one PSD reading to the evidence maps.
 ****/
void
EvidenceMap::addPSDReading(const psdbeam_t &reading)
{
  TIMEBLOCK();
  
  const int center = mapcols/2;

  if (!reading.cached)
    {
      mydebug("missing cached wpos");
      return;
    }

  int startrow    = center + reading.origin.y/MAPRESOLUTION;
  int startcolumn = center + reading.origin.x/MAPRESOLUTION;
  int endrow      = center + reading.wpos.y/MAPRESOLUTION;
  int endcolumn   = center + reading.wpos.x/MAPRESOLUTION;

  for (unsigned int stmapid = 0; stmapid < stmapcount; stmapid++)
    addPSDLine(stmapid, startcolumn, startrow, endcolumn, endrow,
	       PSD_OUT_OF_RANGE != reading.dist);
}

/****f* soccer-pere/EvidenceMap::compareCells
 * DESCRIPTION
 *   Compare two map cells, and return 1 if they match, and 0
 *   othervise.
 ****/
int
EvidenceMap::compareCells(const cell_t c1, const cell_t c2)
{
  int retval = 0;
  if (c1 > unknown && c2 > unknown)
    retval = 1;
  if (c1 < unknown && c2 < unknown)
    retval = 1;

#if 0
  mydebug("cmp cells %u %u = %d", c1, c2, retval);
#endif
  
  return retval;
}

/****f* soccer-pere/EvidenceMap::compare
 * DESCRIPTION
 *   Compare this long term map with nmap using given translation,
 *   returning a MatchScore.
 ****/
int
EvidenceMap::compare(const map_t &nmap, int tcolumn, int trow, int trotation)
{
  int sum = 0;
  for (unsigned int row = 0; row < mapcols; row++)
    for (unsigned int column = 0; column < mapcols; column++)
      {
	vec2d source = vec2d(column+tcolumn, row+trow).rotate(deg2rad(trotation));
#if 0
	mydebug("cmp source = (%dx%d)", source.x, source.y);
#endif
	if (0 <= source.x && source.x < (int)mapcols &&
	    0 <= source.y && source.y < (int)mapcols)
	  sum += compareCells(ltmap[row][column], nmap[source.y][source.x]);
      }
  return sum;
}

class vec3d
{
public:
  int x, y, z;
  vec3d(int _x = 0, int _y = 0, int _z = 0) : x(_x), y(_y), z(_z) {};
  vec3d operator+(const vec3d &v) const { return vec3d(x+v.x, y+v.y, z+v.z); };
  vec3d operator*(int i) const          { return vec3d(x*i, y*i, z*i); };
  vec3d operator/(int i) const          { return vec3d(x/i, y/i, z/i); };
};

void
EvidenceMap::saveMap(const char *filename, const map_t &nmap)
{
  Picture pic;
  assert(sizeof(cell_t) == 1);

  picture_init(&pic, mapcols, mapcols, pix_grey, 1,
	       (unsigned char*)&nmap[0][0], sizeof(nmap));

  FILE *fp = fopen(filename, "w");
  if (NULL != fp)
    {
      picproc_pnmEncode(&pic, (picproc_writer)write, fileno(fp));
      fclose(fp);
    }
}

bool
EvidenceMap::matchMatureMap(vec2d *delta, radians *phidelta)
{
  TIMEBLOCK();
  int score[5][5][5];

  vec3d match;
  int maxscore = 0;

  int maturemapid = (stcurrent + stmapcount - 1) % stmapcount;

  saveMap("evidencemap.ppm", stmap[maturemapid]);

  /* Center of mass */
  for (int row = -2; row <= 2; row++)
    for (int column = -2; column <= 2; column++)
      for (int rotation = -4; rotation <= 4; rotation += 2)
	{
	  int v = compare(stmap[maturemapid], column, row, rotation);
	  maxscore = MAX(maxscore, v);
#if 0
	  mydebug(" match %d %d %d = %d", row, column, rotation, v);
#endif

	  match = match +
	    vec3d(column, row, rotation) * v / ARRAYSIZE(stmap[maturemapid]);

	  score[row+2][column+2][rotation/2 + 2] = v;
	}

  mydebug("total: %d %d %d (max score %d)", match.x, match.y, match.z,
	  maxscore);

  if (0 != delta)
    *delta = vec2d(match.x,match.y);
  if (0 != phidelta)
    *phidelta = deg2rad(match.z);
  return true;
}

#ifdef TEST
int
main()
{
  EvidenceMap emap;

  emap.matchMatureMap(0,0);
  emap.saveMap("evidencemap-lt.ppm", emap.getLongTermMap());
  return 0;
}
#endif
