/*
 * Author: Petter Reinholdtsen <pere@td.org.uit.no>
 * Date:   2000-08-21
 */

#include "fieldmap.hh"
#include "soccerfield.h"
#include "triangle2d.hh"
#include "miscmacro.h"

#include <assert.h>

void
FieldMap::makeSegments(segment2d walls[], unsigned int wallcount,
		       const vec2d corners[], unsigned int cornercount)
{
  assert(wallcount > cornercount);
  for (unsigned int i = 0; i < cornercount; i++)
    {
      unsigned int last = (i-1 + cornercount) % cornercount;
      assert(last < cornercount);
      vec2d diff = corners[i] - corners[last];
      float lheading = diff.heading();
      vec2d lstart;
      vec2d lend;
      assert(i<wallcount);
      walls[i].length   = diff.length();
      if (fabs(lheading) > M_PI_2)
	{
	  lstart = corners[i];
	  lend   = corners[last];
	  lheading = rad2rad(lheading + M_PI);
	}
      else
	{
	  lstart = corners[last];
	  lend   = corners[i];
	}
      walls[i].heading  = lheading;
      walls[i].start = lstart;
      walls[i].end = lend;
    }
}
/****f* soccer-pere/FieldMap::isOnField
 * DESCRIPTION
 *   Check if the given position is inside the soccer field
 ****/
bool
FieldMap::isOnField(vec2d pos)
{
  /*
   * If a line from the center of the field to the given position
   * intersect any of the walls, the point is on the outside. of the
   * soccer field
   */

  unsigned int wallcount;
  const segment2d *walls = getMap(&wallcount);
  segment2d line(vec2d(0,0), pos);

  for (unsigned int i=0; i<wallcount; i++)
    if (line.intersects(walls[i]))
      return false;
  return true;
}

const vec2d *
FieldMap::getCorners(unsigned int *cornercount)
{
  /* Polygon surrounding the soccer field */
  static const vec2d corners[] =
  {
    vec2d( FIELD_WIDTH_X/2,     FIELD_WIDTH_Y/2), /* top right */
    vec2d( FIELD_WIDTH_X/2,    -FIELD_WIDTH_Y/2), /* bottom right */
    vec2d( FIELD_GOAL_WIDTH/2, -FIELD_WIDTH_Y/2), /* bottom goal right front */
    vec2d( FIELD_GOAL_WIDTH/2, -FIELD_WIDTH_Y/2 - FIELD_GOAL_DEPTH), /* yellow */
    vec2d(-FIELD_GOAL_WIDTH/2, -FIELD_WIDTH_Y/2 - FIELD_GOAL_DEPTH),
    vec2d(-FIELD_GOAL_WIDTH/2, -FIELD_WIDTH_Y/2), /* bottom goal left front */
    vec2d(-FIELD_WIDTH_X/2,    -FIELD_WIDTH_Y/2), /* bottom left */
    vec2d(-FIELD_WIDTH_X/2,     FIELD_WIDTH_Y/2), /* top left */
    vec2d(-FIELD_GOAL_WIDTH/2,  FIELD_WIDTH_Y/2), /* top goal left front */
    vec2d(-FIELD_GOAL_WIDTH/2,  FIELD_WIDTH_Y/2 + FIELD_GOAL_DEPTH), /* blue */
    vec2d( FIELD_GOAL_WIDTH/2,  FIELD_WIDTH_Y/2 + FIELD_GOAL_DEPTH),
    vec2d( FIELD_GOAL_WIDTH/2,  FIELD_WIDTH_Y/2)  /* top goal right front */
  };

  *cornercount = ARRAYSIZE(corners);
  return &corners[0];
}

const segment2d *
FieldMap::getMap(unsigned int *segments)
{
  static segment2d walls[15];
  static unsigned int wallcount = 0;
  static bool firsttime = true;

  if (firsttime) /* generate walls from corners */
    {
      unsigned int cornercount;
      const vec2d *corners = getCorners(&cornercount);
      makeSegments(walls, ARRAYSIZE(walls), corners, cornercount);
      wallcount = cornercount;

      firsttime = false;
    }

  *segments = wallcount;
  return &walls[0];
}

vec2d &
FieldMap::getBlueGoalCenter(void)
{
  static vec2d center(0, FIELD_WIDTH_Y/2);
  return center;
}

vec2d &
FieldMap::getYellowGoalCenter(void)
{
  static vec2d center(0, -FIELD_WIDTH_Y/2);
  return center;
}
