/*
 * Author: Petter Reinholdtsen <pere@td.org.uit.no>
 * Date:   2000-07-01
 *
 * Try to find our global (overhead tracking) id by comparing our
 * movement with the movement of the tracked friendly robots.
 */

#include <eyebot.h>
#include <stdlib.h>
#include <math.h>
#include <language.h>
#include <parselanguage.h>
#include "picproc.h"
#include "driver.hh"
#include "input.h"

#include "debug.h"

typedef struct {
  PositionType startpos;
  PositionType curpos;
  bool valid; /* true if this robot could be me */
} robot_info_t;

typedef struct {
  PositionType myStartpos;
  PositionType myCurpos;
  int ballcount;

  int validleft;  /* Number of 'valid' robots left in robot */
  int robotcount; /* Number of elemnts in robots */
  robot_info_t robots[MAXEYE];
} env_info_t;

static void
init_env(env_info_t *info, Driver *drive)
{
  info->ballcount = 0;
  info->validleft = MAXEYE;
  info->robotcount = MAXEYE;

  info->myStartpos = drive->getPosition();

  for (int i = 0; i < MAXEYE; i++)
    {
      robot_info_t *robot = &info->robots[i];
      robot->valid = true;
      robot->curpos.x = 0;
      robot->curpos.y = 0;
      robot->curpos.phi = 0;
      robot->startpos = robot->curpos;
    }
}

static int
my_lang_handler(langEvent *event, void *ref)
{
  env_info_t *info = (env_info_t *)ref;
  switch (event->type)
    {
    case langTypeTrack:
      {
	langEventTrack *ev = (langEventTrack *)event;
	switch (ev->name)
	  {
	  case langNameFriend:
	    mydebug("Got %d pos", ev->num);
	    info->robots[ev->num].curpos.x = ev->x/1000.0;
	    info->robots[ev->num].curpos.y = ev->y/1000.0;
	    info->robots[ev->num].curpos.phi = lang_angle2rad(ev->heading);
	    break;
	  case langNameBall:
	    info->ballcount++;
	    mydebug("Got ball #%d", info->ballcount);
	    break;
	  }
	break;
      }
    default: /* Ignore */
      break;
    }
  return 0;
}

static double
square(double val)
{
  return val*val;
}

static bool
is_same_movement(const PositionType *mp1, const PositionType *mp2, 
		 const PositionType *rp1, const PositionType *rp2)
{
  const meter distlimit  = 0.10;
  const radians rotlimit = M_PI_4;

  meter distdiff =
    square(mp2->x - mp1->x) + square(mp2->y - mp1->y) -
    square(rp2->x - rp1->x) + square(rp2->y - rp1->y);

  radians rotdiff = (mp2->phi - mp1->phi) - (rp2->phi - rp1->phi);
    
  mydebug("Dd: %f\nRd: %f", distdiff, rotdiff);

  if (distlimit > fabs(distdiff) && rotlimit > fabs(rotdiff))
    return true;

  return false;
}

/*
 * Compare the given movement with the difference in position and
 * heading as given by environment, changing 'valid' to false if there
 * are large differences.
 */
static int
compare_movement(env_info_t *env, Driver *drive)
{
  env->myCurpos = drive->getPosition();
  for (int id = 0; id < env->robotcount; id++)
    {
      if (!is_same_movement(&env->myStartpos, &env->myCurpos,
			    &env->robots[id].startpos,
			    &env->robots[id].curpos))
	{
	  env->robots[id].valid = false;
	  env->validleft--;
	}
    }
  return 0;
}

static int
update_startpos(env_info_t *env, Driver *drive)
{
  mydebug("update_startpos");
  env->myStartpos = drive->getPosition();
  for (int id = 0; id < env->robotcount; id++)
    env->robots[id].startpos = env->robots[id].curpos;
  return 0;
}

static int
draw_env(Picture *pic, env_info_t *env)
{
  mydebug("draw_env(%x, %x)", pic, env);
  int white = 0x00ffffff;
  picture_coord row;
  picture_clear(pic);

  LCDSetPrintf(1, 11, "#%3d", env->ballcount);
  LCDSetPrintf(2, 11, "#%3d", env->validleft);

  int step = pic->width / env->robotcount;
  int radius = step/2 - 1;

  if (1 > radius)
    radius = 1;

  row = pic->height - 8 - 6 * radius;

  mydebug("Drawing %d robots", env->robotcount);

  /* Draw all potential robots */
  for (int i = 0; i < env->robotcount; i++)
    {
      if (env->robots[i].valid)
	{
	  int x = step/2 + i * step;
	  picproc_drawCircle(pic, x, row, radius, white);
	  picproc_drawArrow(pic, x, row, radius, radius/2,
			    env->robots[i].curpos.phi -
			    env->robots[i].startpos.phi, white);
	}
    }

  mydebug("draw me");

  /* Draw me */
  picproc_drawCircle(pic, pic->width/2, row+radius*3, radius, white);
  picproc_drawArrow(pic,  pic->width/2, row+radius*3, radius, radius/2,
		    env->myCurpos.phi - env->myStartpos.phi, white);
  mydebug("drawing done");
  return 0;
}

static int
display_env(Picture *pic, env_info_t *env)
{
  draw_env(pic, env);
  LCDPutImage(pic->data);
  return 0;
}
/****i* soccer-pere/get_valid_id
 * DESCRIPTION
 *   Find and return the id of the first valid robot in the
 *   enviroment.  Return -1 if no one is found.
 ****/
static int
get_valid_id(env_info_t *env)
{
  for (int id = 0; id < env->robotcount; id++)
    if (env->robots[id].valid)
      return id;
  return -1;
}
int
setGlobalID(int current)
{
  int id = current;
  int key;
  int max = 6;

  LCDClear();
  LCDMenu("-", "+", "ABRT", " OK");

  LCDPrintf("Specify robots\n"
            "global id.\n");
  LCDSetPrintf(3, 5, "%2d", id);
  while (KEY4 != (key = inputRead()))
    {
      switch (key)
        {
        case KEY1:
          id = (id + max - 1) % max;
          LCDSetPrintf(3, 5, "%2d", id);
          break;
        case KEY2:
          id = (id + max + 1) % max;
	  LCDSetPrintf(3, 5, "%2d", id);
          break;
        case KEY3:
          return current;
          break;
        }
    }
  return id;
}

/****f* soccer-pere/findGlobalId
 * SYNOPSIS
 *  int findGlobalId(lang_channel ch, Driver *drive)
 * DESCRIPTION
 *   Find and return the id of the first valid robot in the
 *   enviroment.
 * RETURN VALUE
 *   0 to MAXEYE with the matching global id or -1 if matching failed.
 ****/
int
findGlobalId(lang_channel ch, Driver *drive)
{
  int defaultid = OSMachineID();
  env_info_t env;
  int now;
  int retval = -1;
  Picture lcdpic;
  BYTE lcdpic_data[128*64/8];

  if (0 == ch)
    {
      LCDPrintf("lang channel missing!\n");
      return setGlobalID(defaultid);
    }

  LCDClear();
  LCDPutString("Autodetect\nglobal id?");
  LCDMenu("YES", " ", " ", "NO");
  switch (inputGet())
    {
    case KEY4:
      return setGlobalID(defaultid);
      break;
    default:
      break;
    }

  picture_init(&lcdpic, 128,64,pix_bitmap, 0,
	       &lcdpic_data[0], sizeof(lcdpic_data));

  LCDClear();
  LCDPrintf("Detect global id\n");
  LCDMenu(" ", " ", " ", "END");

  /* Flush message queue */
  mydebug("flushing message queue");
  while (lang_messageWaiting(ch))
    lang_receive(ch, my_lang_handler, &env);

  mydebug("init env");
  init_env(&env, drive);

  mydebug("init env done");

  /* Receive info for 15 second, or until we got 2 ballpos packets */
  now = OSGetCount();
  while (KEY4 != inputRead() && now + 1500 > OSGetCount() && env.ballcount < 2)
    {
      mydebug("waiting for packages, time %d ball #%d",
	    OSGetCount(), env.ballcount);
      if (lang_messageWaiting(ch))
	lang_receive(ch, my_lang_handler, &env);

      env.myCurpos = drive->getPosition();
      display_env(&lcdpic, &env);
    }

  if (2 > env.ballcount)
    {
      LCDPrintf("Timeout #%d\n", env.ballcount);
      OSSleep(200);
      return -1;
    }

  env.ballcount=0;
  update_startpos(&env, drive);

  LCDSetPrintf(3,0,"turning\n");

  drive->turn(M_PI_2);

  while (KEY4 != inputRead() && 1 < env.validleft)
    {
      if (lang_messageWaiting(ch))
	lang_receive(ch, my_lang_handler, &env);

      env.myCurpos = drive->getPosition();
      display_env(&lcdpic, &env);

      if (drive->isStanding() && 2 <= env.ballcount)
	{
	  radians turn = M_PI_2;

	  env.ballcount = 0;  /* Wait for stable and updated position */
	  while (KEY4 != inputRead() && 1 > env.ballcount)
	    {
	      if (lang_messageWaiting(ch))
		lang_receive(ch,my_lang_handler, &env);
	      display_env(&lcdpic, &env);
	    }

	  compare_movement(&env, drive);
	  update_startpos(&env, drive);

	  if (rand() < RAND_MAX/2) /* Turn the other way half the time */
	    turn = -turn;

	  LCDSetPrintf(3, 0, "Turning %4.2f", turn);
	  drive->turn(turn);
	}
    }

  if (1 == env.validleft)
    {
      retval = get_valid_id(&env);
      LCDPrintf("My id: %d\n", retval);
    }
  else
    LCDPrintf("No ID found!");

  OSSleep(400);
  return retval;
}

/**
 * Test the functions to make sure they work as expected
 */
int
test_global_id_find(lang_channel ch, Driver *drive)
{
  PositionType p1 = {0, 0.00, 0};
  PositionType p2 = {0, 0.00, M_PI_2};
  PositionType p3 = {0, 0.00, -M_PI_2};
  PositionType p4 = {0, 0.05, M_PI/8};

  if (is_same_movement(&p1, &p2, &p3, &p4))
    LCDPrintf("Same movement\n");
  else
    LCDPrintf("Different movement\n");
  inputWait(ANYKEY);
  return 0;
}
