/*
 * Author: Petter Reinholdtsen <pere@td.org.uit.no>
 * Date:   2000-02-04
 *
 * Use panning camera and goal detection to determine robot position
 * using triangulation.
 */

#include <eyebot.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>

#include "picture.h"
#include "picproc.h"
#include "camera.hh"
#include "camera.h"
#include <vision.h>

#include "scrollmenu.hh"
#include "driver.hh"
#include "panning.hh"
#include "angles.h"
#include "input.h"

enum {
  /* How many degres do the camera pan */
  CAMERA_PAN_ANGLE = 180,

  CAMERA_PAN_CENTER = 184, /* in the range 0 - 255 */

  /* Horizontal view angle */
  CAMERA_HVIEW_ANGLE = 47,

  /* Which row holds the horizont? */
  HORIZONT_ROW = 24,

  /* Limit this until camera cable is long enough */
  SERVO_MAX = 240
};

/*
 * Invert cross on LCD with senter in row, column, with a range*2 wide
 * block in the senter of the cross.
 */

static void
draw_target_marker(int row, int column, int range)
{
  int r, c;
  for (r = 0; r < imagerows-2-6; r++)
    LCDSetPixel(r,column, 2);
  for (c = 0; c < imagecolumns-2; c++)
    LCDSetPixel(row,c, 2);

  for (r = row - range; r < row + range+1; r++)
    for (c = column - range; c < column + range+1; c++)
      {
        LCDSetPixel(r,c, 2);
      }
}

static int
get_color(const char *descr)
{
  int curcol = -1, lastcol = 0;
  int color = 0;
  colimage cimg;
  int key;

  LCDClear();
  LCDMenu("Set", "+", "-", "End");
  LCDSetPos(0,10);
  LCDPrintf("%s", descr);

  while (KEY4 != (key = KEYRead()))
    {
      if (curcol != lastcol)
        {
          LCDSetPos(1,10);
          LCDPrintf("%3d", curcol);
          lastcol = curcol;
        }
      CAMGetColFrame(&cimg, 0);
      LCDPutColorGraphic(&cimg);
#if 0
      draw_target_marker((imagerows-2)/2, (imagecolumns-2)/2, 3);

      color = hue_median(cimg, (imagerows-2)/2, (imagecolumns-2)/2, 3);
      LCDSetPos(2,10);
      LCDPrintf("%3d", color);
#endif

      switch (key)
        {
        case KEY1:
          curcol = color;
          break;
        case KEY2:
          curcol++;
          break;
        case KEY3:
          curcol--;
          break;
        default:
          break;
        }
    }

  return 0;
}

/* Keep all vision library structs together */
typedef struct {
  VIS_structs         *info;
  colimage            pic;
  Camera *            framegrabber;
  Panning             pancamera;
  Driver *            driver;
} vis_info_t;


typedef struct {
  int start;
  int end;
  int color;
} pan_info;


static void
region_print(char code, int imgcount, VIS_color_region *region)
{
  LCDPrintf("%c%d:%2dx%2d->%2dx%2d\n", code, imgcount,
	    region->coord.left, region->coord.top,
	    region->coord.right, region->coord.bot);
}

static void
find_move_to_center(vis_info_t *vinfo)
{
  const int num_images = 1+CAMERA_PAN_ANGLE/CAMERA_HVIEW_ANGLE;
  pan_info info[num_images];
  int servo_step = SERVO_MAX / (num_images-1);
  int i;

  if (!vinfo->pancamera.initialize(SERVO11))
    {
      LCDPutString("No camera servo!\n");
      return;
    }
  vinfo->pancamera.setRotation(deg2rad(CAMERA_PAN_ANGLE));
  if (vinfo->info)
    vinfo->pancamera.setCenter(vinfo->info->CAM_offset->alpha);
  else
    vinfo->pancamera.setCenter(deg2rad(CAMERA_PAN_CENTER));

  /* Take panorama snapshots over the complete view range */
  LCDClear();
  LCDPrintf("Panorama/%d/%d\n",num_images,servo_step);

  for (i=0; i < num_images; i++)
    {
      VIS_color_region *region; /* ball : 0 - ygoal : 1 - bgoal : 2 */
      Picture *snapshot;

      vinfo->pancamera.setServo(i * servo_step);
      while (vinfo->pancamera.isRotating())
	; /* Give the servo time to settle in */
      snapshot = vinfo->framegrabber->capture();

      region = VIS_Find_classes(vinfo->info, snapshot);
      if (VIS_BALL == region[0].color) /* orange region */
	region_print('O', i, &region[0]);

      if (VIS_YGOAL == region[1].color) /* yellow region */
	region_print('Y', i, &region[1]);

      if (VIS_BGOAL == region[2].color) /* Blue region */
	region_print('B', i, &region[2]);
    }

  LCDMenu("END", "END", "END", "END");
  inputWait(ANYKEY);
}

static void
vis_setup(vis_info_t *vinfo)
{
  Picture *snapshot = vinfo->framegrabber->capture();
  VIS_Init_calibration(vinfo->info, Camera_capture,
		       (void*)vinfo->framegrabber);
  VIS_Init_color_classes(vinfo->info);
  VIS_Init_distance(vinfo->info, snapshot);
}

static void
calibrate_cb(scrollmenu *menu, void *data, int direction)
{
  vis_setup((vis_info_t*)data);
}

static void
find_move_to_center_cb(scrollmenu *menu, void *data, int direction)
{
  find_move_to_center((vis_info_t*)data);
}

int
pan_pos_test(Camera *framegrabber, Driver *driver)
{
  scrollmenu menu;
  vis_info_t vinfo;
  vinfo.framegrabber = framegrabber;
  vinfo.driver = driver;
  Picture lcd;
  BYTE lcddata[128*64/8];
  picture_init(&lcd, 128, 64, pix_bitmap, 0, &lcddata[0], sizeof(lcddata));

  vinfo.info = VIS_Init_var(&lcd, vinfo.framegrabber->capture());

  menu.initialize("Pan-Pos test");
  menu.addEntry("Calibrate", calibrate_cb, &vinfo);
  menu.addEntry("Start",     find_move_to_center_cb,   &vinfo);

  menu.exec();

  VIS_Release_var(vinfo.info);
  return 0;
}
