/*
 * Author: Petter Reinholdtsen <pere@td.org.uit.no>
 * Date:   2000-08-09
 */

#include "panning.hh"
#include "debug.h"
#include "angles.h"
#include "hdtcamera.h"

const int Panning::ticperturn = 150; /* in 1/100 second */

/****f* soccer-pere/Panning::initialize
 * SYNOPSIS
 *   bool Panning::initialize(DeviceSemantics servoid)
 * DESCRIPTION
 *   Initialize the panning object, controlling the given servo.
 ****/
bool
Panning::initialize(DeviceSemantics servoid)
{
  handle = SERVOInit(servoid);
  if (0 != handle)
    {
      setCenter(255/2);
      setRotation(M_PI);
      setHeading(0.0);
      return true;
    }
  else
    return false;
}

Panning::~Panning()
{
  mydebug("Release camera servo");
  if (0 != handle)
    SERVORelease(handle);
}

/****f* soccer-pere/Panning::setCenter
 * SYNOPSIS
 *   void Panning::setCenter(BYTE v);
 * DESCRIPTION
 *   Set servo "center setting, ie. when the platform is at angle 0.0.
 ****/
void
Panning::setCenter(BYTE v)
{
  center = v;
}

/****f* soccer-pere/Panning::setRotation
 * SYNOPSIS
 *   void Panning::setRotation(radians r);
 * DESCRIPTION
 *   Set servo rotation range in radians.
 ****/
void
Panning::setRotation(radians r)
{
  radperval = r/255;
}

/****f* soccer-pere/Panning::setServo
 * SYNOPSIS
 *   void Panning::setServo(BYTE val)
 * DESCRIPTION
 *   Set a new servo value.
 ****/
void
Panning::setServo(BYTE val)
{
  /* do nothing if there are no change */
  if (val == servoval)
    return;

  /*
   *  Calculate how long it will take before the camera is stable,
   *  assuming 1 second for a complete turn.
   */
  int delay = ticperturn * abs(val - servoval) / 255;
  ready = OSGetCount() + delay;

  servoval = val;
  SERVOSet(handle, servoval);
}

/****f* soccer-pere/Panning::getServo
 * SYNOPSIS
 *   BYTE Panning::getServo(void);
 * DESCRIPTION
 *   Get the current servo value.
 ****/
BYTE
Panning::getServo(void)
{
  return servoval;
}

/****f* soccer-pere/Panning::setHeading
 * SYNOPSIS
 *   bool Panning::setHeading(radians r)
 * DESCRIPTION
 *   Set a new camera heading as close as possible to the heading
 *   specified.
 * RETURN VALUE
 *   Returns true if it is possible to look in the requested
 *   direction, and false if not.
 * BUGS
 *   XXX Currently assumes 180 degree turn radius.
 * SEE ALSO
 *   Panning::getHeading()
 ****/
bool
Panning::setHeading(radians r)
{
  bool retval = true;
  r = rad2rad(r);
  int val = center + (int)(r / radperval);

  if (0 > val)
    {
      retval = false;
      val = 0;
    }
  if (255 < val)
    {
      retval = false;
      val = 255;
    }

  /* Do nothing if there are no change */
  if (val == servoval)
    return retval;

  mydebug("Camera set angle %d=%5.2f cval=%d rpv=%5.2f",
	  val, rad2deg(r), servoval, radperval);

  setServo(val);

  return retval;
}

/****f* soccer-pere/Panning::getHeading
 * SYNOPSIS
 *   radians Panning::getHeading(void)
 * DESCRIPTION
 *   Get the current camera heading angle, based on the current camera
 *   servo setting.  0 radians is looking forward.
 * NOTE
 *   XXX Currently assumes servo rotates left to right, should assume
 *   servo rotates right(0) to left(255).
 * SEE ALSO
 *   Panning::setHeading()
 ****/
radians
Panning::getHeading(void)
{
  radians r = (servoval - center) * radperval;
  return r;
}

/****f* soccer-pere/Panning::isRotating
 * SYNOPSIS
 *   bool Panning::isRotating(void)
 * DESCRIPTION
 *   Check if panning platform is still rotating, or if the image is
 *   expected to be stable.
 * RETURN VALUE
 *   true if the platform is no longer rotating.
 * SEE ALSO
 *   Panning::setHeading(), Panning::setServo()
 ****/
bool
Panning::isRotating(void)
{
  return OSGetCount() < ready;
}

/* Test the panning class */
void
test_panning(void)
{
  int key;
  Panning pan;
  if (!pan.initialize(SERVO11))
    {
      mydebug("pan init failed");
      return;
    }

  camera_pan_type *cpdata = (camera_pan_type*)HDTFindEntry(CAMERA, CAMERAPAN);
  if (NULL != cpdata && 0 == cpdata->version)
    {
      pan.setServo(cpdata->servo);
      pan.setCenter(cpdata->center);
      pan.setRotation(cpdata->rotation);
      mydebug("found pancam HDT %d %d %5.2f",
	      cpdata->servo, cpdata->center, rad2deg(cpdata->rotation));
    }

  LCDClear();
  LCDPrintf("Test panning\n");
  LCDPrintf("255=left,0=right\n");
  LCDMenu("+", "-", "CNTR", "END");
  while (KEY4 != (key = KEYRead()))
    {
      BYTE val = pan.getServo();
      LCDSetPrintf(3, 0, "Servo: %5d",   val);
      LCDSetPrintf(4, 0, "Angle: %5.1f deg", rad2deg(pan.getHeading()));
      LCDSetPrintf(5, 0, "Rotating: %c", pan.isRotating()? 'y' : 'n');
      switch (key)
	{
	case KEY1:
	  val += 8;
	  pan.setServo(val);
	  break;
	case KEY2:
	  val -= 8;
	  pan.setServo(val);
	  break;
	case KEY3:
	  pan.setHeading(0.0);
	  break;
	}
    }
}
