/*
 * Author: Petter Reinholdtsen <pere@td.org.uit.no>
 * Date:   2000-07-14
 *
 * RoboCup 2000 Eyebot soccer player control program.
 */

#include <string.h>
#include <stdio.h>

#include <irtv.h>
#if defined(__mc68000__)
#include <IRnokia_vcn620.h>
#endif

#include "picproc.h"
#include "parselanguage.h"

#include "robotcontrol.hh"
#include "soccer.hh"
#include "debug.h"
#include "soccerfield.h"
#include "vec2d.hh"
#include "vision.h"
#include "camera.h"
#include "envstatus.hh"
#include "angles.h"
#include "input.h"
#include "scrollmenu.hh"

/* If the heading difference is less then this, start driving. */
const radians ANGLE_LIMIT = M_PI/32;

/* If I am this close to the goal position, turn to the end heading */
const int MOVE_LIMIT = 30; /* mm */

/****f* soccer-pere/square
 * SYNOPSIS
 *   template <class intype> intype square(intype t)
 * DESCRIPTION
 *   Returns the square (t*t) of t.
 ****/
template <class intype>
intype square(intype t)
{
  return t*t;
}

/****f* soccer-pere/signof
 * SYNOPSIS
 *   template <class intype> int signof(intype value)
 * DESCRIPTION
 *   Return the sig of the parameter.
 * RETURN VALUE
 *   -1 if value is < 0, else +1.
 ****/
template <class intype>
int signof(intype value)
{
  if (value < 0)
    return -1;
  else
    return 1;
}


/****f* soccer-pere/RobotControl::driveTo
 * SYNOPSIS
 *   void RobotConrol::driveTo(vec2d pos, radians endheading)
 * DESCRIPTION
 *   Drive shortest path to the given position, and make sure we end
 *   up facing (or tailing) the given heading when we arrive.
 *
 *   If game status is 'pause', ignore drive request.
 *****/
void
RobotControl::driveTo(const vec2d &pos, radians endheading)
{
  /* Make it possible to draw line to where we are going */
  status->setDestPoint(pos);

  Driver *driver = status->driver;

#if CURVE
  radians myheading;
  vec2d mypos = status->getMyCurrentPos(&myheading);
  vec2d posdiff = pos - mypos;
  int dist = posdiff.length();
  radians turn = rad2rad(- myheading + endheading
				   + 2 * posdiff.heading());
  
  mydebug("Drive from (%dx%d) to (%dx%d) d=%d h=%5.1f v=%5.1f e=%5.1f t=%5.1f",
	  mypos.x, mypos.y, pos.x, pos.y, dist,
	  rad2deg(myheading), rad2deg(posdiff.heading()), rad2deg(endheading),
	  rad2deg(turn));

  if (EnvStatus::pause == status->status)
    return;

  /* If it is faster to back off, do so */
  if (turn < -M_PI_2 || turn > M_PI_2)
    {
      turn = rad2rad(turn+M_PI);
      dist = -dist;
    }

  if (abs(dist) > MOVE_LIMIT && fabs(turn) < ANGLE_LIMIT)
    {
      radians rot = 2 * (endheading + posdiff.heading());
      double curve = dist * rot * sin(rot);
      mydebug("  curve %d mm %5.1f rot", (int)curve, rad2deg(rot));
      driver->curve(curve/1000.0, rot);
    }
  else
    {
      mydebug("  turn %f", turn);
      if (fabs(turn) > ANGLE_LIMIT)
	driver->turn(turn);
    }
#else
  radians myheading;
  vec2d mypos = status->getMyCurrentPos(&myheading);
  vec2d posdiff = pos - mypos;
  int dist = posdiff.length();
  radians turn;

  if (dist > MOVE_LIMIT)
    turn = rad2rad(posdiff.heading() - myheading);
  else
    turn = rad2rad(endheading - myheading);

  /* If it is faster to back off, do so */
  if (turn < -M_PI_2 || turn > M_PI_2)
    {
      turn = rad2rad(turn+M_PI);
      dist = -dist;
    }

  mydebug("Drive from (%dx%d) to (%dx%d) d=%d h=%5.1f v=%5.1f e=%5.1f",
          mypos.x, mypos.y, pos.x, pos.y, dist,
	  rad2deg(myheading), rad2deg(posdiff.heading()), rad2deg(endheading));

  if (EnvStatus::pause == status->gamestatus)
    return;

  if (abs(dist) > MOVE_LIMIT && fabs(turn) < ANGLE_LIMIT)
    {
      driver->straight(dist/1000.0);
    }
  else
    {
      if (fabs(turn) > ANGLE_LIMIT)
	driver->turn(turn);
    }
#endif
}

bool
RobotControl::decide(void)
{
  return true;
}

class Player : protected RobotControl
{
public:
  Player(EnvStatus *_status) : RobotControl(_status) {};
  bool decide(void);

  void setPlayMode(int mode);
  void menu(void);
private:
  bool trackBall(void);
  bool gotoBallUseCompass(void);
  void driveRandomFollowWalls();
};

bool
Player::decide(void)
{
  const int timestep = 200; /* n/100 seconds between decisions */

  if (status->isBallKnown())
    status->cameraTrack(VIS_BALL);
  else
    status->cameraSearch();

  if (lastdecision + timestep > status->lastupdate)
    return true;
  lastdecision = status->lastupdate;

  if (false)
    return trackBall();
  else
    return gotoBallUseCompass();
}

/****f* soccer-pere/driveRandomFollowWalls
 * DESCRIPTION
 *   Drive randomly, follow walls
 ****/
void
Player::driveRandomFollowWalls()
{
  mydebug("drive random");
  const int distance = 200; /* mm */
  const meter drivedistance = distance * 2.0 / (3 * 1000.0);
  int psdfront      = status->getPSDVal(EnvStatus::PSDFront);
  int psdfrontleft  = status->getPSDVal(EnvStatus::PSDFrontLeft);
  int psdfrontright = status->getPSDVal(EnvStatus::PSDFrontRight);
  int psdleft       = status->getPSDVal(EnvStatus::PSDLeft);
  int psdright      = status->getPSDVal(EnvStatus::PSDRight);

  bool pause = EnvStatus::pause == status->gamestatus;

  if (status->driver->isStalled())
    {
      status->driver->straight(-0.10);
      return;
    }

  if (psdfront < 200 || psdfrontleft < 200 || psdfrontright < 200)
    { /* turn away */
      if (psdleft > psdright)
	{
	  if (!pause)
	    status->driver->turn(M_PI_2);
	}
      else
	{
	  if (!pause)
	    status->driver->turn(-M_PI_2);
	}
    }
  else
    {
      if (psdleft == PSD_OUT_OF_RANGE && psdright == PSD_OUT_OF_RANGE)
	{
	  if (!pause)
	    status->driver->straight(drivedistance); /* keep going */
	}
      else 
	{
	  if (PSD_OUT_OF_RANGE == psdleft)
	    psdleft = distance;
	  if (PSD_OUT_OF_RANGE == psdright)
	    psdright = distance;

	  int width         = psdright + psdleft;
	  radians direction = M_PI*(psdleft - psdright)/(4*width);
	  mydebug("rand narrow %d .. %5.2f", width, direction);
	  if (!pause)
	    status->driver->curve(drivedistance, direction);
	}
    }
}

bool
Player::gotoBallUseCompass(void)
{
  if (status->driver->isStalled())
    {
      status->driver->straight(-0.10);
      return true;
    }

  if (status->isBallKnown())
    { /* Drive to ball */
      mydebug("go to ball");
      radians myheading;
      vec2d mypos   = status->getMyCurrentPos(&myheading);

      vec2d ballpos = status->getBallCurrentPos();
      vec2d toball  = ballpos - mypos;         /* From me to the ball */
      driveTo(ballpos, toball.heading());
    }
  else
    driveRandomFollowWalls();
  return true;
}

/****f* soccer-pere/Player::trackBall
 * SYNOPSIS
 *   bool Player::trackBall(void)
 * DESCRIPTION
 *   Based on the current status, decide on an action.
 * RETURN VALUE
 *   'true' if we still are running, 'false' if the game is over.
 ****/
bool
Player::trackBall(void)
{
  /* Where are we going now */
  vec2d respos;
  radians resheading;

  radians myheading;
  vec2d mypos         = status->getMyCurrentPos(&myheading);

  vec2d ballpos       = status->getBallCurrentPos();
  vec2d toball = ballpos - mypos;         /* From me to the ball */

  vec2d ourgoalcenter = status->getOurGoalCenter();
  
  vec2d theirgoalcenter = status->getTheirGoalCenter();
  vec2d theirgoalleft   = status->getTheirGoalLeft();
  vec2d theirgoalright  = status->getTheirGoalRight();

  vec2d tomygoal = ourgoalcenter - mypos; /* from me to my goal center */


  if (0 > tomygoal.dot(toball))
    { /* I'm between my goal and the ball */
      if (toball.heading() - theirgoalleft.heading() < 0 &&
	  toball.heading() - theirgoalright.heading() > 0)
	{
	  LCDSetString(0,0, "Push    ");
	  vec2d fromball = theirgoalcenter-ballpos;
	  respos = ballpos;
	  resheading = fromball.heading();
	}
      else
	{ /* Line up to push the ball first */
	  LCDSetString(0,0, "Attack  ");
	  vec2d fromball = theirgoalcenter-ballpos;
	  vec2d halfthere = toball/2;

	  vec2d behind = fromball * halfthere.length() / fromball.length();

	  respos = ballpos - behind;
	  resheading = fromball.heading();
	}
      LCDSetChar(1,0,' ');
    }
  else
    { /* I'm on the wrong side of the ball */
      if (signof(ballpos.y) == signof(ourgoalcenter.y))
	{ /* The ball is on our side */
	  /*
	   * If ball is on our side of the field, place yourself between the
	   * ball and our goal center.
	   */
	  LCDSetString(0,0, "Defend  ");
	  vec2d fromgoal = (ballpos - ourgoalcenter)/2;
	  respos = ourgoalcenter + fromgoal;
	  resheading = fromgoal.heading();
	}
      else
	{
	  /*
	   * go to point behind the ball, facing ball and opponent goal
	   */
	  LCDSetString(0,0, "Position");
	  vec2d fromball = theirgoalcenter-ballpos;
	  respos = ballpos - fromball * 3 * ROBOT_RADIUS / fromball.length();
	  resheading = fromball.heading();
	}

      /*
       * If I would hit the ball while position myself, find point
       * outside ball.
       */
      vec2d posdiff = respos - mypos; /* From me to where I'm going */
      vec2d ballcross = posdiff.project(toball) - toball;
      mydebug("Ball cross (%dx%d)->(%dx%d)=(%dx%d)",
	      toball.x, toball.y,
	      posdiff.x, posdiff.y,
	      ballcross.x, ballcross.y);
      int lsquare = ballcross.lengthSquare();
      if (0 == lsquare)
	{
	  mydebug("Line on ball, pix any way around it!");
	  vec2d rotvec(posdiff.y * 2 * ROBOT_RADIUS,
		       posdiff.x * 2 * ROBOT_RADIUS);
	  respos = ballpos + rotvec / posdiff.length();
	}
      else if (ballcross.lengthSquare() < square(2*ROBOT_RADIUS))
	{
	  mydebug("Moving around ball");
	  LCDSetChar(1,0,'m');
	  respos = ballpos + ballcross * 2 * ROBOT_RADIUS / ballcross.length();
	}
      else
	LCDSetChar(1,0,' ');
      
    }

  driveTo(respos, resheading);
  return true;
}

void
Player::setPlayMode(int mode)
{
  langEventPlayMode event;
  event.timestamp = OSGetCount();
  event.type = langTypePlayMode;
  event.to = BROADCAST;
  event.mode = mode;
  /* Tell the others */
  lang_send(status->channel, (langEvent *)&event);
  lang_sendFlush(status->channel);
  /* and myself */
  status->lang_handler((langEvent *)&event);
}

void
Player::menu(void)
{
  int key;
  LCDClear();
  LCDMenu("Side", "Left", "Righ", "END");
  while (KEY4 != (key = inputRead()))
    {
      LCDSetPrintf(0,0, "Side: %s",
		   status->goal_at_top ? "Blue  " : "Yellow");
      PositionType pos = status->driver->getPosition();
      LCDSetPrintf(1,0, "Pos: %4dx%4d",
		   (int)(pos.x*1000), (int)(pos.y*1000));
      LCDSetPrintf(2, 0, "Phi: %5.2f", rad2deg(pos.phi));
      switch (key)
	{
	case KEY1:
	  status->goal_at_top = !status->goal_at_top;
	  break;
	case KEY2:
	  if (status->goal_at_top)
	    {
	      PositionType startpos = {-0.29,-1.140,M_PI_2};
	      status->setMyPosition(startpos);
	    }
	  else
	    {
	      PositionType startpos = { 0.29, 1.140,-M_PI_2};
	      status->setMyPosition(startpos);
	    }
	  break;
	case KEY3:
	  if (status->goal_at_top)
	    {
	      PositionType startpos = { 0.29,-1.140,M_PI_2};
	      status->setMyPosition(startpos);
	    }
	  else
	    {
	      PositionType startpos = {-0.29, 1.140,-M_PI_2};
	      status->setMyPosition(startpos);
	    }
	  break;
	}
    }
  LCDClear();
  return;
}

void
print_status(EnvStatus *status)
{
  if (status->goal_at_top)
    LCDSetString(0, 8, "->");
  else
    LCDSetString(0, 8, "<-");
}

int
play_ball(lang_channel channel, Camera *camera, Driver *driver)
{
  enum {
    display_none,
    display_field,
    display_psdhist
  } display = display_field;
  int id;
  Picture screen;
  PositionType startpos = {-0.29,-1.140,M_PI_2};
  BYTE screendata[128*64/8];
  bool running = true;
  bool fpsstate = false;

  if (NULL == camera)
    mydebug("no camera!");
  else
    fpsstate = camera->framerateOn(true);

#if 0
  id = findGlobalId(channel, driver);
#else
  id = OSMachineID();
#endif
  if (id == -1)
    running = false;

  EnvStatus status(id, channel, camera, driver);
  Player control(&status);

  picture_init(&screen,128,64,pix_bitmap,0,&screendata[0],sizeof(screendata));

  status.setMyPosition(startpos);
  status.setCompassHeading(startpos.phi);
  driver->start();

  if (!status.initialize())
    mydebug("status.initialize failed");

  LCDClear();
  LCDMenu("Menu", " Soc", "cer ", "END");
  while (running)
    {
      status.update();
      running = control.decide();

      if (EnvStatus::play == status.gamestatus)
	display = display_none;

      switch (display)
	{
	case display_none:
	  LCDClear();
	  print_status(&status);
	  break;
	case display_field:
	  status.draw(&screen);
#if !defined(__mc68000__)
	  {
	    FILE *fp = fopen("soccer-lcd.ppm", "w");
	    picproc_pnmEncode(&screen, (picproc_writer)write, fileno(fp));
	    fclose(fp);
	  }
#endif
	  LCDPutImage(screen.data);
	  print_status(&status);
	  break;
	case display_psdhist:
	  picture_clear(&screen);
	  LCDPutImage(screen.data);
	  break;
	}

      switch (inputRead())
	{
	case KEY1:
	  control.menu();
	  LCDMenu("Menu", " Soc", "cer ", "END");
	  break;
	case KEY2:
	  if (EnvStatus::pause == status.gamestatus)
	    {
	      status.setGameStatus(EnvStatus::pregame);
	      OSSleep(500);
	    }
	  else
	    {
	      if (status.goal_at_top)
		control.setPlayMode(langModeKickOffYellow);
	      else
		control.setPlayMode(langModeKickOffBlue);
	      display = display_none;
	      OSSleep(500);
	    }
	  break;
	case KEY3:
	  status.setGameStatus(EnvStatus::pause);
	  display = display_field;
	  LCDMenu("Menu", " Soc", "cer ", "END");
	  break;
	case KEY4:
	  running = false;
	  break;

#if defined(__mc68000__)
	case RC_0: /* Do not display anything */
	  display = display_none;
	  break;
	case RC_1: /* show soccer field */
	case RC_2:
	  display = display_field;
	  LCDMenu("Menu", " Soc", "cer ", "END");
	  break;
	case RC_3: /* show localization PSD angle histogram */
	  display = display_psdhist;
	  LCDMenu("Menu", " Soc", "cer ", "END");
	  break;
	case RC_RECORD:
	  status.uploadPSD();
	  LCDClear();
	  LCDMenu("Menu", " Soc", "cer ", "END");
	  break;
	case RC_FF:
	  control.setPlayMode(langModePlayBlue);
	  break;
	case RC_RW:
	  control.setPlayMode(langModePlayYellow);
	  break;
	case RC_PLAY: /* Kick off for the other team */
	  if (status.goal_at_top)
	    control.setPlayMode(langModeKickOffYellow);
	  else
	    control.setPlayMode(langModeKickOffBlue);
	  display = display_none;
	  break;
	case RC_OK: /* Kick off for our team, I'm kicking. */
	  if (status.goal_at_top)
	    control.setPlayMode(langModeKickOffBlue);
	  else
	    control.setPlayMode(langModeKickOffYellow);
	  display = display_none;
	  driver->straight(0.40);
	  break;
	case RC_STOP:
	  control.setPlayMode(langModeHalfTime);
	  display = display_field;
	  break;
#endif
	}
    }

  if (NULL != camera)
    camera->framerateOn(fpsstate);

  driver->stop();
  
  return 0;
}
