/*
 * Author: Petter Reinholdtsen <pere@td.org.uit.no>
 * Date:   2000-08-29
 *
 * Display robot and ball positions as broadcasted on the language channel.
 */ 
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>

#include <stdio.h>
#include <string.h>
#include <iostream>
#include <fstream>
#include <strstream>

#include "posviewer.hh"
#include "soccerfield.h"
#include "fieldmap.hh"
#include "language.h"

// default constructor 
PosViewer::PosViewer(Display *d, int s) : 
  XWindow(d, s, "CIIPSGlory Position Viewer ", 0,0, 640, 400)
{
  colorWhite = WhitePixel(display, screen_num);
  colorRed   = rgb_to_color(255, 0, 0);
  colorGreen = rgb_to_color(0, 150, 0);
  colorBlue  = rgb_to_color(0, 0, 255);
  colorBlack = BlackPixel(display, screen_num);

  border = 20;
  scale = (width-4.0*border) / FIELD_WIDTH_Y;

  for (unsigned int j = 0; j<3; j++)
    for (unsigned int i = 0; i<POSITIONS; i++)
      position[j][i].timestamp = 0;
}

PosViewer::~PosViewer()
{
}

int
PosViewer::getNameIndex(int name)
{
  int n = -1;
  switch (name)
    {
    case langNameBall:
      n = 0;
      break;
    case langNameFriend:
      n = 1;
      break;
    case langNameEnemy:
      n = 2;
      break;
    }
  return n;
}

void
PosViewer::setPosition(unsigned char from, int name, unsigned char id,
		       int timestamp, const vec2d &pos, float heading)
{
  printf("Robot %d told %c #%d was position at (%dx%d) %5.2f at %d\n",
	 from, name, id, pos.x, pos.y, rad2deg(heading), timestamp);
  int idx = getNameIndex(name);
  if (idx < 0)
    return;
  if (id < POSITIONS)
    {
      position[idx][id].timestamp = timestamp;
      position[idx][id].from = from;
      position[idx][id].pos = pos;
      position[idx][id].heading = heading;
    }
}

vec2d
PosViewer::mappos(const vec2d &p)
{
  vec2d npos((int)(width/2.0 + p.y*scale), 
	     (int)(height - border - (FIELD_WIDTH_X/2.0 - p.x)*scale));
  return npos;
}

// draw static images here, like the CMVision logo
void PosViewer::paint()
{ /* draw soccer field */
  fillRect(0,0,width-1, height-1, colorGreen);

  unsigned int wallcount;
  const segment2d *walls = FieldMap::getMap(&wallcount);
  for (unsigned int i = 0; i<wallcount; i++)
    {
      vec2d wstart = mappos(walls[i].start);
      vec2d wend = mappos(walls[i].end);
      drawLine(wstart.x, wstart.y, wend.x, wend.y, colorWhite);
    }
}

void PosViewer::update()
{

  /*
   * Reduce the refresh rate to avoid flickering.  This should be done
   * with double buffering instead.
   */
  static int lastupdate = 0;
  int now = time(NULL);
  if (now < lastupdate + 1)
    return;
  lastupdate = now;

  paint();

  /* draw robot positions */
  int radies[] = {21, ROBOT_RADIUS, ROBOT_RADIUS};
  for (unsigned int j = 0; j < 3; j++)
    {
      int radius = (int)(radies[j]*scale);
      for (unsigned int i = 0; i<POSITIONS; i++)
	{
	  if (0 != position[j][i].timestamp)
	    {
	      double heading = position[j][i].heading - M_PI_2;
	      vec2d wpos = mappos(position[j][i].pos);
	      drawArc(wpos.x-radius, wpos.y-radius, 2*radius, 2*radius,
		      0, 64*360, colorWhite);
	      drawLine(wpos.x,wpos.y,
		       wpos.x+radius*cos(heading),
		       wpos.y-radius*sin(heading), colorWhite);
	      
	      /* Who told us so ... */
	      vec2d teller = mappos(position[1][position[j][i].from].pos);
	      drawLine(wpos.x,wpos.y, teller.x, teller.y, colorBlue);
	    }
	}
    }
}


// draw the static stuff before we display the window
void PosViewer::showWindow(){
  XWindow::showWindow();
}


// handle expose event so static stuff can be redrawn
bool PosViewer::handleEvent(XEvent &report){
  switch(report.type){
  case Expose:
    paint();
    return true;
  default:
    return false;
  }
}

