/*
 * Author: Petter Reinholdtsen <pere@td.org.uit.no>
 * Date:   2000-07-10
 *
 * Scrolldown menu for the Eyebot.
 */

#include <eyebot.h>

#include "scrollmenu.hh"
#include "miscmacro.h"
#include "input.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#define NOPROC ((entryProc)-1)

/**** soccer-pere/scrollmenu
 * DESCRIPTION
 *   Keep the menu list as a doubly linked circular list.
 ****/
void
scrollmenu::setTitle(const char *_title)
{
  if (0 == _title)
    _title = "[no title]";
  strncpy(title, _title, sizeof(title));
  title[sizeof(title)-1] = '\0';
}

/**** soccer-pere/scrollmenu::scrollmenu
 * SYNOPSIS
 *   scrollmenu::scrollmenu(const char *_title)
 * DESCRIPTION
 *   Create new scroll menu with the given title.
 ****/
scrollmenu::scrollmenu(void)
  : bottom(0), current(0), screentop(0), screenbottom(0), menulines(0)
{
  for (int i = 0; i < maxentries; i++)
    entries[i].proc = NOPROC;
}

bool
scrollmenu::initialize(const char *_title)
{
  menulines = lcdtextheigth-2;
  setTitle(_title);

  /*
   * First entry ends up at the end of the menu.  At the start the
   * ring got only one element
   */
  bottom = newEntry("Exit", 0, 0, 0, 0);
  if (0 == bottom)
    return false;

  bottom->up = bottom;
  bottom->down = bottom;

  current = bottom->down;
  screentop = bottom->down;

  return true;
}

scrollmenu::~scrollmenu()
{
  menuEntry *me = bottom->down;
  bottom->down = 0;
  while (0 != me)
    {
      menuEntry *down = me->down;
      deleteEntry(me);
      me = down;
    }  
  bottom = 0;
}

bool
scrollmenu::addEntry(const char *entry, entryProc proc, void *data)
{
  menuEntry *me;

  if (strlen(entry) > lcdtextwidth-1)
    {
      LCDPrintf("Menu entry \"%s\" too long\n", entry);
      return false;
    }

  me = newEntry(entry, proc, data, bottom->up, bottom);
  if (0 == me)
    return false;

  bottom->up->down = me;
  bottom->up = me;

  current = bottom->down;
  screentop = bottom->down;

  return true;
}

/****f* soccer-pere/scrollmenu::removeEntryCall
 * DESCRIPTION
 *   Remove the first menu entry with proc as the activation function.
 * NOTE
 *   Unimplemented at the moment.
 ****/
void
scrollmenu::removeEntryCall(entryProc proc, void *data)
{
  proc = 0;
  data = 0;
  LCDPutString("Unimplemented\n");
}

/****f* soccer-pere/scrollmenu::removeEntry
 * DESCRIPTION
 *   Remove the first menu entry with the given text.
 * NOTE
 *   Unimplemented at the moment.
 ****/
void
scrollmenu::removeEntry(const char *text, void *data)
{
  text = 0;
  data = 0;
  LCDPutString("Unimplemented\n");
}

void
scrollmenu::setEntry(menuEntry *me, const char *entry)
{
  strncpy(me->entry, entry, sizeof(me->entry));
  me->entry[sizeof(me->entry)-1] = '\0';
  me->entrylen = strlen(me->entry);
}

bool
scrollmenu::changeMenuText(entryProc proc, void *data, const char *entry)
{
  menuEntry *me;

  for (me = bottom->down; me->down != bottom->down; me = me->down)
    {
      if (me->proc == proc && me->data == data)
	break;
    }
  if (me->down == bottom->down)
    return false;
  setEntry(me, entry);
  return true;
}

scrollmenu::menuEntry *
scrollmenu::newEntry(const char *entry, entryProc proc, void *data,
		    menuEntry *up, menuEntry *down)
{
  menuEntry *me = 0;
#if !defined(use_malloc)
  for (int i = 0; i < maxentries; i++)
    {
      if (NOPROC == entries[i].proc)
	{
	  me = &entries[i];
	  break;
	}
    }
#else
  me = new menuEntry();
#endif
  if (0 == me)
    return 0;

  setEntry(me, entry);

  me->proc = proc;
  me->data = data;
  me->up = up;
  me->down = down;
  return me;
}

/****f* soccer-pere/scrollmenu::deleteEntry
 * DESCRIPTION
 *   Free all resouces allocated by newEntry
 ****/
void
scrollmenu::deleteEntry(menuEntry *me)
{
#if !defined(use_malloc)
  me->proc = NOPROC;
#else
  delete me;
#endif
}

void
scrollmenu::display(menuEntry *current)
{
  int lines = menulines;
  LCDClear();
  LCDMode(NONSCROLLING);
  LCDPrintf("%s\n", title);
  for (menuEntry *me = screentop;
       0 != me && lines--;
       me = me->down)
    {
      LCDPrintf("%c%s",
		current == me ? '*' : ' ',
		me->entry);
      if (me->entrylen < lcdtextwidth-1)
	LCDPrintf("\n");

      screenbottom = me;
      if (bottom->down == me->down) /* We are at the end */
	break;
    }

  LCDMenu("/\\", "\\/", " <", " >");
}

void
scrollmenu::show(void)
{
  display(current);
}

void
scrollmenu::exec(void)
{
  display(current);
  while (0 != current)
    {
      switch (inputRead())
	{
	case KEY1:

	  if (screentop == current)
	    {
	      if (bottom->down != screentop)
		screentop = current->up;
	      else
		/*
		 * Scroll down until last entry is at the bottom of the screen
		 */
		while (screenbottom->down != bottom->down)
		  {
		    screentop = screentop->down;
		    screenbottom = screenbottom->down;
		  }
	    }

	  current = current->up;

	  display(current);
	  break;
	case KEY2:
	  if (screenbottom == current)
	    {
	      if (current->down != bottom->down)
		screentop = screentop->down;
	      else
		screentop = bottom->down;
	    }

	  current = current->down;
	  display(current);
	  break;
	case KEY3:
	  if (0 != current->proc)
	    current->proc(this, current->data, -1);

	  if (bottom->down == current->down) /* Exit was choosen */
	    current = 0;

	  display(current);
	  break;
	case KEY4:
	  if (0 != current->proc)
	    current->proc(this, current->data, 1);

	  if (bottom->down == current->down) /* Exit was choosen */
	    current = 0;

	  display(current);
	  break;
	}
    }
}

#ifdef TEST

struct value {
  char *text;
  int val;
};

void
updateEntry(scrollmenu *menu, void *data, int direction)
{
  struct value *v = (struct value*)data;
  v->val += direction;

  char buf[17];
#if !defined(__mc68000__)
  /* Missing in RoBIOS */
  snprintf(buf, sizeof(buf), "%-13s%2d", v->text, v->val);
#else
  sprintf(buf, "%-13s%2d", v->text, v->val);
#endif
  menu->changeMenuText(updateEntry, data, buf);
}

int
main()
{
  scrollmenu menu;

  if (!menu.initialize("Main menu"))
    {
      LCDPrintf("Menu init failed\n");
      return -1;
    }

  struct value values[] =
  {
    {"Test bottom", 1},
    {"Test middle1", 2},
    {"Test middle2", 2},
    {"Test middle3", 2},
    {"Test middle4", 2},
    {"Test middle5", 2},
    {"Test top",    3}
  };

  for (unsigned int i = 0; i < ARRAYSIZE(values); i++)
    {
      menu.addEntry(values[i].text, updateEntry, &values[i]);
      updateEntry(&menu, &values[i], 0);
    }
  menu.exec();
  free(malloc(1));
  return 0;
}
#endif
