/*
 * Author: Petter Reinholdtsen <pere@td.org.uit.no>
 * Date:   2000-08-01
 */

#include <stdio.h>
#include "angles.h"
#include "segment2d.hh"
#include "debug.h"
#include "envstatus.hh"
#include "blocktimer.hh"
#include "debug.h"
#include "soccerfield.h"
#include "fieldmap.hh"
#include "picproc.h"
#include "miscmacro.h"
#include "soccer.hh"

#include <assert.h>

#define HISTOGRAMSIZE 16
#define RESOLUTION 20
#define MAXDIST 150
#define filter 1
/* Low limit for histogram max after filtering */
#define HISTMAXLIMIT 5

/* low limit on # of points after removing duplicates */
#define VALUELIMIT 20

typedef struct {
  /* into the edges array */
  unsigned int index;
  unsigned int nindex; /* index of diffphi point */

  /* polar coordinates, relative to current robot position */
  radians phi;
  int d;

  /* Original coordinates, with reduced resolution */
  vec2d wpos;

  radians diffphi;
  unsigned int hindex; /* histogram index */
  unsigned int hval;   /* How many in my histgram class */
} translated_t;

#if !defined(__mc68000__)
/****f* soccer-pere/drawHistogram
 * DESCRIPTION
 *   Draw a scaled histgram in picture using color.
 ****/
static void
drawHistogram(Picture *pic, unsigned int histogram[], unsigned int size,
	      unsigned int maxval, unsigned int color)
{
  int prevw = 0;
  int prevh = 0;

  int woffset = pic->width/10;
  int hoffset = pic->height - pic->height/10;

  assert((int)size < pic->width);

  for (unsigned int i = 0; i < size; i++)
    {
      int val = histogram[i];
      int w = woffset + pic->width  *   i * 8 / (size*10);
      int h = hoffset - pic->height * val * 8 / (maxval*10);
      if (prevw != 0)
	picproc_drawLine(pic, prevw, prevh, w, h, color);
      prevw = w;
      prevh = h;

      if (0 == (i % 10))
	picproc_drawLine(pic, w, hoffset, w, hoffset+10, color);
      else if (0 == (i % 5))
	picproc_drawLine(pic, w, hoffset, w, hoffset+5, color);
    }
}
#endif

/****f* soccer-pere/cmp_segment_length
 * DESCRIPTION
 *   qsort() sorting function.  Compare two segment2d element lengths.
 ****/
static int
cmp_segment_length(const void *ap, const void *bp)
{
  segment2d *a = (segment2d*)ap, *b = (segment2d*)bp;
  return b->length - a->length;
}

static int
cmp_translated_phi(const void *ap, const void *bp)
{
  translated_t *a = (translated_t*)ap, *b = (translated_t*)bp;
  return (int)((a->phi - b->phi)*1024);
}

static int
cmp_translated_hval_phi(const void *ap, const void *bp)
{
  translated_t *a = (translated_t*)ap, *b = (translated_t*)bp;
  int hvaldiff = b->hval - a->hval;
  if (0 == hvaldiff)
    {
      radians phidiff = a->phi - b->phi;
      return (int)((phidiff)*1024);
    }
  else
    return hvaldiff;
}

/****f* soccer-pere/removeDuplicates
 * DESCRIPTION
 *   Given a sorted array with values elements in there, move all the
 *   unique elements to the beginning of the list, keeping their
 *   order, and returning the number of unique elements.
 ****/
static unsigned int
removeDuplicates(translated_t translated[], unsigned int values)
{
  TIMEBLOCK();
  unsigned int count = 0;
  for (unsigned int i = 0; i < values; i++)
    if (0 == count || translated[count].wpos != translated[i].wpos)
      {
	count++;
	memcpy(&translated[count], &translated[i], sizeof(translated[0]));
      }
  mydebug(" removed %d duplicates, %d left", values - count, count);
  return count;
}

/****f* soccer-pere/rad2histval
 * DESCRIPTION
 *   Map / convert angle to histogram value.  Lines with the same
 *   angle and lines 90 degrees on each other are mapped into the same
 *   histogram value.
 ****/
static unsigned int
rad2histval(radians r, int size)
{
  /*
   * Make sure 90 degree peak (when everything is normal) is in the
   * center of the histogram
   */
  r += M_PI_4;

  /* Group rotated angles together */
  if (r < 0.0)
    r += M_PI;

  /* group angles 90 degrees on each other together */
  if (r > M_PI_2)
    r -= M_PI_2;

  return (unsigned int) (r*size/(M_PI_2)) % size;
}

/****f* soccer-pere/histval2rad
 * DESCRIPTION
 *   Map a histgram value to the smallest one of the four radians
 *   angles mapping into this value.  The four angles are rad, rad+90,
 *   rad+180, rad-90, rad-180.
 ****/
static radians
histval2rad(int hval, int size)
{
  return rad2rad(hval * M_PI_2 / size - M_PI_4);
}

/****f* soccer-pere/findHistogramPeak
 * DESCRIPTION
 *   Find peaks in histogram with values higher then threshold.
 *   Return center value (angle) in r.
 ****/
static bool
findHistogramPeak(unsigned int histogram[],
		  unsigned int size, unsigned int threshold, 
		  radians *r)
{
  TIMEBLOCK();
  int runs = 0;
  bool high = false;
  bool top[size];
  radians angle = 0.0;


  /* Find length and center of 'high' sequences */
  int start = 0;

  for (unsigned int i = 0; i < size; i++)
    {
      unsigned int v = histogram[i];
      if (v > threshold)
	top[i] = true;
      else
	 top[i] = false;

      if (top[i])
	{
	  if (top[i] != high)
	    {
	      start = i;
	      high = true;
	    }
	}
      else
	{
	  if (top[i] != high)
	    {
	      int length = i-start;
	      int center = start + i >> 1;
	      angle = histval2rad(center, size);
	      mydebug("run: length=%d, center=%2d (%5.2f)", length, center,
		      rad2deg(angle));
	      high = false;
	      runs++;
	    }
	}
    }

  if (1 == runs)
    {
      if (NULL != r)
	*r = angle;
      return true;
    }
  else
    return false;
}

static int
filterHistogram(unsigned int histogram[], unsigned int size)
{
  TIMEBLOCK();
  int hmax = 0;
  unsigned int h2[size];
  for (unsigned int i = 0; i < size; i++)
    {
#if filter
      unsigned int prev1 = (i + size - 1) % size;
      unsigned int next1 = (i + 1) % size;
      int val = (histogram[prev1] + histogram[i] + histogram[next1]);
#else
      int val = histogram[i];
#endif
      h2[i] = val;
      hmax = MAX(hmax, val);
    }
  memcpy(&histogram[0], &h2[0], sizeof(h2));
  return hmax;
}

/****f* soccer-pere/lineDistance
 * DESCRIPTION
 *   Return the distance between two parallell segments.
 ****/
bool
lineDistance(const segment2d l1, const segment2d l2, vec2d *diff)
{
  const int anglelimit = 5;
  int xdiff = 0, ydiff = 0;
  mydebug(" segment: (%dx%d)->(%dx%d) heading=%5.2f",
	  l2.start.x, l2.start.y, l2.end.x, l2.end.y, rad2deg(l2.heading));

  mydebug(" map    : (%dx%d)->(%dx%d) heading=%5.2f",
	  l1.start.x,l1.start.y,l1.end.x, l1.end.y, rad2deg(l1.heading));

  if (fabs(l2.heading) < deg2rad(anglelimit))
    {
      ydiff = (l1.start.y + l1.end.y - l2.start.y - l2.end.y)/2;

      if (l1.start.x > l2.start.x)
	xdiff = l1.start.x - l2.start.x;

      if (l1.end.x < l2.end.x)
	xdiff = l1.end.x - l2.end.x;

      mydebug("ydiff = %d", ydiff);
    }
  else if (fabs(l2.heading) > deg2rad(90-anglelimit))
    {
      xdiff = (l1.start.x + l1.end.x - l2.start.x - l2.end.x)/2;

      if (l1.start.y > l2.start.y)
	ydiff = l1.start.y - l2.start.y;

      if (l1.end.y < l2.end.y)
	ydiff = l1.end.y - l2.end.y;

      mydebug("xdiff = %d",xdiff);
    }
  else
    {
      mydebug("Go crasy!");
      return false;
    }
  diff->x = xdiff;
  diff->y = ydiff;
  return true;
}

typedef struct {
  radians deltaphi;
  vec2d delta;
} pose_t;

static int
cmp_poses_delta(const void *ap, const void *bp)
{
  pose_t *a = (pose_t*)ap, *b = (pose_t*)bp;
  int distdiff = abs(a->delta.length()) - abs(b->delta.length());
  if (0 != distdiff)
    return distdiff;
  else
    {
      int anglediff = (int)(10*rad2deg(fabs(a->deltaphi) - fabs(b->deltaphi)));
      return anglediff;
    }
}

/****f* soccer-pere/locateSegmentsInMap
 * DESCRIPTION
 *   Compare the segments with the lines in the map, and return the
 *   translation and rotation with the best match - and the least
 *   translation and rotation.  The segments must be sorted on length
 *   with the longest segments first.
 * RETURN VALUE
 *   true if match was found, false if lines do not match map.
 ****/
static bool
locateSegmentsInMap(const segment2d map[], int mapcount,
		    const segment2d segments[], int segmentcount,
		    vec2d curpos, radians curheading,
		    vec2d *delta, radians *deltaphi)
{
  TIMEBLOCK();
  radians rotations[] = {0, M_PI};
  pose_t poses[26];
  unsigned int numposes = 0;

  /* Should only accept lines almost 90 degrees on each other */

  /* Ignore the shorter lines for now */
  if (1 < segmentcount)
    segmentcount = 1;

  for (int sid = 0; sid < segmentcount; sid++)
    for (int mid = 0; mid < mapcount; mid++)
      {
	if (segments[sid].length > map[mid].length)
	  continue; /* map segment to short, skip to the next */

	radians diffphi = map[mid].heading - segments[sid].heading;
	for (unsigned int i = 0; i < ARRAYSIZE(rotations); i++)
	  {
	    radians ldeltaphi = rad2rad(diffphi + rotations[i]);
	    mydebug("assume %d = %d -> %5.2f rot",sid,mid,rad2deg(ldeltaphi));

	    vec2d ldelta;
	    bool ok = lineDistance(map[mid], segments[sid].rotate(ldeltaphi),
				   &ldelta);
	    if (ok)
	      { /* Make sure the robot stays inside the soccer field.. :-) */
		vec2d newpos = curpos + ldelta;
		if (!FieldMap::isOnField(newpos))
		  ok = false;
  	      }

	    if (ok)
	      {
	    if (segmentcount == 1)
	      {
		/* store possible pose */
		if (numposes < ARRAYSIZE(poses))
		  {
		    poses[numposes].delta = ldelta;
		    poses[numposes].deltaphi = ldeltaphi;
		    numposes++;
		  }
		else
		  mydebug("To many poses, skipping one");
	      }
	    else
	      {
		/*
		 * We have more then one line, try one of the others to
		 * get a better match
		 */
#if 0
		vec2d delta = mapSegmentToSegment(map[mid],
						  segments[sid].rotate(ldeltaphi));
#endif
	      }
	      }
	  }
      }

  if (0 < numposes)
    {
      qsort(&poses[0], numposes, sizeof(poses[0]), cmp_poses_delta);
      if (NULL != delta)
	*delta    = poses[0].delta;
      if (NULL != deltaphi)
	*deltaphi = poses[0].deltaphi;
      mydebug("Move robot (%dx%d) rotate %5.2f",
	      poses[0].delta.x, poses[0].delta.y, rad2deg(poses[0].deltaphi));
      return true;
    }
  else
    return false;
}

#if !defined(__mc68000__)
#define picpos(pic,x,y) (pic->width/2  + (x)/20), (pic->height/2 - (y)/20)
#define picvec2dpos(pic, v) picpos(pic, v.x, v.y)
static void
drawSegments(Picture *pic, const segment2d *walls, unsigned int wallcount,
	    int color)
{
  for (unsigned int i = 0; i < wallcount; i++)
    picproc_drawLine(pic,
		     picvec2dpos(pic, walls[i].start),
		     picvec2dpos(pic, walls[i].end), color);
}
#endif

bool
isEdgeUseless(const psdbeam_t &edge, int timelimit)
{
  return (!edge.cached || edge.timestamp <= timelimit ||
	  PSD_OUT_OF_RANGE == edge.dist);
}

bool
selflocate(int timelimit, const segment2d walls[], int wallcount,
	   psdbeam_t edges[], unsigned int edgecount,
	   const vec2d &curpos, radians curheading,
	   vec2d *translation, radians *rotation)
{
  TIMEBLOCK();
  bool retval = false;
  translated_t translated[EnvStatus::NUM_EDGES];
  unsigned int values = 0;
  unsigned int histogram[HISTOGRAMSIZE];
  segment2d segments[10];
  unsigned int segmentcount = 0;

  memset(&histogram[0], 0, sizeof(histogram));

  /*
   * Convert to polar coordinates with reduced resolution,
   * and sort the points around the clock */
  for (unsigned int i = 0; i < edgecount; i++)
    {
      /* Skip uninitialized, out of range or too old values */
      if (isEdgeUseless(edges[i], timelimit))
	continue;
      vec2d diff = (edges[i].wpos - curpos);
      assert(values < ARRAYSIZE(translated));

      translated[values].index = i;

      radians r = diff.heading();
      if (r < 0.0)
	r += 2*M_PI; /* Make sure the angles are positive */
      translated[values].phi = r;
      translated[values].d   = diff.lengthSquare();

#if 1 != RESOLUTION
      translated[values].wpos = (edges[i].wpos/RESOLUTION) * RESOLUTION;
#else
      translated[values].wpos = edges[i].wpos;
#endif
      values++;
    }
  qsort(&translated[0], values, sizeof(translated[0]), cmp_translated_phi);

  values = removeDuplicates(translated, values);

  if (VALUELIMIT > values)
    {
      mydebug("#psd values %d below %d", values, VALUELIMIT);
      return false;
    }

  /*
   * Calculate angle from one point to the nest in the list, store
   * result in histogram
   */
  for (unsigned int i = 0; i < values; i++)
    {
      unsigned int next = (i + 1) % values;
      vec2d diff = translated[i].wpos - translated[next].wpos;

      /* Ignore point diff if the distance is to long */
      if (MAXDIST*MAXDIST < diff.lengthSquare())
	{
	  translated[i].index = 0;
	  translated[i].nindex = 0;
	  translated[i].hindex = 0;
	  mydebug("%i skipped %6.2f (%2dx%2d)",
		  i, rad2deg(translated[i].phi),
		  edges[translated[i].index].wpos.x,
		  edges[translated[i].index].wpos.y);
	  continue;
	}

      radians r = diff.heading();

      translated[i].diffphi = r;

      unsigned int hindex = rad2histval(r, HISTOGRAMSIZE);
      assert(hindex < HISTOGRAMSIZE);
      translated[i].hindex = hindex;
      histogram[hindex] ++;

      translated[i].nindex = translated[next].index;

      mydebug("%2d %6.2f (%2dx%2d)-(%2dx%2d) diffphi = %6.2f (%2d)",
	      i, rad2deg(translated[i].phi),
	      edges[translated[i].index].wpos.x,
	      edges[translated[i].index].wpos.y,
	      edges[translated[next].index].wpos.x,
	      edges[translated[next].index].wpos.y,
	      rad2deg(r), hindex);
    }

  /*
   * Find peaks in histogram, try to map them to walls
   */

  /* Filter histogram, average neightbours */
  unsigned int hmax = filterHistogram(histogram, HISTOGRAMSIZE);
  mydebug("hmax=%d after filtering", hmax);

  if (HISTMAXLIMIT > hmax)
    {
      mydebug("hmax %d below limit %d", hmax, HISTMAXLIMIT);
      return false;
    }

  /* Sort out the high values by hval and phi*/
  for (unsigned int i = 0; i < values; i++)
    translated[i].hval = histogram[translated[i].hindex];
  qsort(&translated[0], values, sizeof(translated[0]),
	cmp_translated_hval_phi);

  vec2d delta;
  radians deltaphi = 0.0;
  if (findHistogramPeak(histogram, HISTOGRAMSIZE, 2*hmax/3, &deltaphi))
    {
      /* Hm, what should we do now ... */
    }

  radians maxlineheading = 0.0;
  int maxlinelen = 0;

  unsigned int start = 0;
  for (unsigned int i = 1; i < values; i++)
    {
      vec2d diff = translated[i].wpos - translated[start].wpos;
      if (rad2histval(diff.heading(),
		      HISTOGRAMSIZE) != translated[start].hindex)
	{
	  if (start != i-1 &&
	      translated[i-1].hindex == translated[start].hindex)
	    {
	      vec2d line = translated[i-1].wpos - translated[start].wpos;
	      int length = line.length();
	      maxlinelen = MAX(maxlinelen, length);
	      radians heading = line.heading();
	      if (length == maxlinelen)
		{
		  maxlineheading = heading;
		}

	      if (length > 200) /* 20 cm min limit */
		{
		  if (segmentcount < ARRAYSIZE(segments))
		    {
		      vec2d lstart;
		      vec2d lend;
		      if (fabs(heading) > M_PI_2)
			{ /* rotate line 180 degrees */
			  lend     = translated[start].wpos;
			  lstart   = translated[i-1].wpos;
			  heading = rad2rad(heading + M_PI);
			}
		      else
			{
			  lstart = translated[start].wpos;
			  lend   = translated[i-1].wpos;
			}
		      segments[segmentcount].start   = lstart;
		      segments[segmentcount].end     = lend;
		      segments[segmentcount].length  = length;
		      segments[segmentcount].heading = heading;
		      segmentcount++;
		    }
		  else
		    mydebug("to many segments");
		  mydebug("Long line from (%dx%d) to (%dx%d) h=%5.2f l=%d",
			  translated[start].wpos.x, translated[start].wpos.y,
			  translated[i-1].wpos.x, translated[i-1].wpos.y,
			  rad2deg(heading), length);
		}
	    }
	  start = i;
	}
    }

  qsort(&segments[0], segmentcount, sizeof(segments[0]), cmp_segment_length);
  retval = locateSegmentsInMap(walls, wallcount, segments, segmentcount,
				curpos, curheading, &delta, &deltaphi);
  if (retval)
    {
      mydebug("rotating %6.2f, translating (%dx%d)",
	      rad2deg(deltaphi), delta.x, delta.y);
      for (unsigned int i = 1; i < values; i++)
	{
	  translated[i].wpos = delta + translated[i].wpos.rotate(deltaphi);
	}
    }
#if !defined(__mc68000__)
  {
    int white = 0xffffffff;
    int grey  = 0xa0a0a0a0;
    int red   = 0x00ff0000;
    int blue  = 0x000000ff;
    int green = 0x0000ff00;
    int brown = 0x00ff00ff;
    Picture *pic = picture_new(300,300, pix_rgb24);

    drawHistogram(pic, histogram, HISTOGRAMSIZE, hmax, grey);

    /* Draw current robot position */
    picproc_drawCircle(pic, picvec2dpos(pic, curpos), 3, red);
    picproc_drawPixel(pic, picvec2dpos(pic, curpos), red);

    for (unsigned int i = 0; i < edgecount; i++)
      {
	if (isEdgeUseless(edges[i], timelimit))
	  continue;
	picproc_drawPixel(pic, picvec2dpos(pic, edges[i].wpos), white);
      }
    for (unsigned int i = 0; translated[i].hval > hmax/2 && i < values; i++)
      {
#if 0
	picproc_drawPixel(pic,
			  picvec2dpos(pic, edges[translated[i].index].wpos),
			  blue);
	picproc_drawLine(pic,
			 picvec2dpos(pic, edges[translated[i].index].wpos),
			 picvec2dpos(pic, edges[translated[i].nindex].wpos),
			 green);
#endif
      }

    drawSegments(pic, walls, wallcount, green);


    for (unsigned int i = 0; i < values; i++)
      {
	picproc_drawPixel(pic, picvec2dpos(pic, translated[i].wpos), brown);
      }

    drawSegments(pic, segments, segmentcount, brown);

    FILE *fp = fopen("self-walls.ppm", "w");
    picproc_pnmEncode(pic, (picproc_writer)write, fileno(fp));
    fclose(fp);

    picture_delete(pic);
  }
#endif

  if (retval)
    {
      vec2d newpos = curpos + delta;
      mydebug("Suggesting new position (%dx%d) %5.2f",
	      newpos.x, newpos.y, rad2deg(curheading + deltaphi));
      if (NULL != translation)
	*translation = delta;
      if (NULL != rotation)
	*rotation = deltaphi;
    }
  return retval;
}

#if 0
#define TEST
#endif
#ifdef TEST
static bool
loadEdges(char *filename, psdbeam_t edges[], int size,
	  vec2d *pos, radians *heading)
{
  char buffer[1024];
  FILE *fp;
  int count = 0;
  if (NULL == filename)
    return false;

  fp = fopen(filename, "r");
  if (NULL == fp)
    return false;
  while (NULL != fgets(&buffer[0], sizeof(buffer), fp))
    {
      if (0 == strncmp(&buffer[0], "mypos ", 6))
	{
	  sscanf(&buffer[0], "mypos %d %d %f ", &pos->x, &pos->y, heading);
	}
      if (0 == strncmp(&buffer[0], "psdval ", 6) && count < size)
	{
	  sscanf(&buffer[0], "psdval %d %d (%d,%d) %f %d %d %d %d ",
		 &edges[count].timestamp,
		 &edges[count].dist,
		 &edges[count].origin.x, &edges[count].origin.y,
		 &edges[count].heading,
		 &edges[count].min, &edges[count].max,
		 &edges[count].wpos.x, &edges[count].wpos.y);
	  if (0 != edges[count].wpos.x && 0 != edges[count].wpos.y)
	    edges[count].cached = true;
	  count++;
	}
    }
  fclose(fp);
  return true;
}

int
main(int argc, char *argv[])
{
  vec2d mypos;
  radians myheading;
  psdbeam_t edges[EnvStatus::NUM_EDGES];
  memset(&edges[0], 0, sizeof(edges));
  unsigned int wallcount;
  const segment2d *walls = FieldMap::getMap(&wallcount);

  if (argc == 2 &&
      loadEdges(argv[1], edges, ARRAYSIZE(edges), &mypos, &myheading))
    {
      selflocate(0, walls, wallcount, edges, ARRAYSIZE(edges),
		 mypos, myheading, 0, 0);
    }
  else
    printf("Failed to load edges file\n");
  return 0;
}
#endif /* TEST */
