/****c* soccer-pere/vec2d
 * DESCRIPTION
 *   2D integer vector class.
 * AUTHOR
 *   Petter Reinholdtsen <pere@td.org.uit.no>, 2000-07-17
 ****/

#ifndef _vec2d_hh
#define _vec2d_hh

#include <math.h>

class vec2d
{
public:
  int x,y;
  vec2d(int _x = 0, int _y = 0) : x(_x), y(_y) {};

  vec2d operator+(const vec2d &v) const { return vec2d(x+v.x, y+v.y); };
  vec2d operator-(const vec2d &v) const { return vec2d(x-v.x, y-v.y); };
  vec2d operator-(void) const           { return vec2d(-x, -y); };
  vec2d operator/(int i) const          { return vec2d(x/i, y/i); };
  vec2d operator*(int i) const          { return vec2d(x*i, y*i); };

  bool operator==(const vec2d &v) const { return x == v.x && y == v.y; };
  bool operator!=(const vec2d &v) const { return x != v.x || y != v.y; };

/****f* soccer-pere/vec2d::dot
 * SYNOPSIS
 *   int vec2d::dot(const vec2d &v)
 * DESCRIPTION
 *   Return the dot (scalar) product of the two vectors (this and v).
 ****/
  int dot(const vec2d &v) const         { return x * v.x + y * v.y;};

  /* Scalar product */
  int operator*(const vec2d &v) const   { return dot(v); };

/****f* soccer-pere/vec2d::project
 * SYNOPSIS
 *   vec2d vec2d::project(const vec2d &v)
 * DESCRIPTION
 *   Return the projection of v onto this vector.
 ****/
  vec2d project(const vec2d &v) const
    {
      return operator*(dot(v)) / lengthSquare();
    };

/****f* soccer-pere/vec2d::rotate
 * SYNOPSIS
 *   vec2d vec2d::rotate(float rad)
 * DESCRIPTION
 *   Return a new vector with the same length as the current vector,
 *   but rotated r radians.
 ****/
  vec2d rotate(float rad) const
  {
    return vec2d(x * cos(rad) - y * sin(rad), x * sin(rad) + y * cos(rad));
  }

/****f* soccer-pere/vec2d::heading
 * SYNOPSIS
 *   float vec2d::heading(void) const
 * DESCRIPTION
 *   Gives the heading (direction) of the 2d vector in radians.
 ****/
  float heading(void) const           { return atan2((double)y,(double)x); };
  
/****f* soccer-pere/vec2d::length
 * SYNOPSIS
 *   int vec2d::length(void) const
 * DESCRIPTION
 *   Gives the length of the vector.
 ****/
  int length(void) const                { return (int)sqrt(lengthSquare()); };

/****f* soccer-pere/vec2d::lengthSquare
 * SYNOPSIS
 *   int vec2d::lengthSquare(void) const
 * DESCRIPTION
 *   Returns the squared length of the vector.
 ****/
  int lengthSquare(void) const          { return x*x + y*y; };
};

#endif /* _vec2d_hh */
