/*
 * C API to interface with camera capture class.
 */

#include "camera.hh"

#ifdef HAVE_V4L2
#include "v4l2camera.hh"
#endif

#ifdef HAVE_EYEBOT
#include "eyebotcamera.hh"
#endif

#include "dummycamera.hh"

#include "camera.h"

#include <string.h>
#include <stdlib.h>

Camera *
Camera::newCamera(unsigned int nwidth, unsigned int nheight,
		  pixel_format nfmt)
{
  char *type = 0;
  Camera * retval = 0;

#ifndef __mc68000__
  /* This hangs the program when using RoBIOS 3v1d. */
  type = getenv("CAMERA");
#endif

#ifdef HAVE_V4L2
  if ((0 != type && 0 == strcasecmp("v4l2", type)) || 0 == retval)
    {
      retval = new V4L2Camera();
      if (0 != retval && false == retval->initialize(nwidth, nheight, nfmt))
	{
	  delete retval;
	  retval = 0;
	}
    }
#endif

#ifdef HAVE_EYEBOT
  if ((0 != type && 0 == strcasecmp("eyebot", type)) || 0 == retval)
    {
      retval = new EyebotCamera();
      if (0 != retval && false == retval->initialize(nwidth, nheight, nfmt))
	{
	  delete retval;
	  retval = 0;
	}
    }
#endif

#ifdef HAVE_DUMMYCAMERA
  if ((0 != type && 0 == strcasecmp("dummy", type)) || 0 == retval)
    {
      retval = new DummyCamera();
      if (0 != retval && false == retval->initialize(nwidth, nheight, nfmt))
	{
	  delete retval;
	  retval = 0;
	}
    }
#endif

  return retval;
}

/****f* libcamera/Camera_newCamera
 * SYNOPSIS
 *   camref Camera_newCamera(unsigned int nwidth, unsigned int nheight,
 *                           pixel_format nfmt)
 * DESCRIPTION
 *   C wrapper for Camera::newCamera().
 * SEE ALSO
 *   Camera::newCamera()
 *****/
capref
Camera_newCamera(unsigned int nwidth, unsigned int nheight,
		 pixel_format nfmt)
{
  return (capref)Camera::newCamera(nwidth, nheight, nfmt);
}

/****f* libcamera/Camera_delete
 * SYNOPSIS
 *   void Camera_delete(capref ref)
 * DESCRIPTION
 *   C wrapper for 'delete camref;'.  Releases resources allocated
 *   in the Camera class.  'ref' is not usable after this call.
 * SEE ALSO
 *   Camera_newCamera()
 *****/
void
Camera_delete(capref ref)
{
  Camera *obj = (Camera *)ref;
  delete obj;
}

/****f* libcamera/Camera_initialize
 * SYNOPSIS
 *   int Camera_initialize(capref ref, unsigned int nwidth,
 *                         unsigned int nheight, pixel_format nfmt,
 *                         char *device)
 * DESCRIPTION
 *   C wrapper for Camera::initialize().
 * SEE ALSO
 *   Camera::initialize()
 *****/
int
Camera_initialize(capref ref, unsigned int nwidth, unsigned int nheight,
                   pixel_format nfmt, char *device)
{
  Camera *obj = (Camera *)ref;
  return obj->initialize(nwidth,nheight,nfmt,device);
}

/****f* libcamera/Camera_init
 * SYNOPSIS
 *   int Camera_initialize(capref ref, unsigned int nwidth, unsigned int nheight)
 * DESCRIPTION
 *   C wrapper for Camera::initialize().
 * SEE ALSO
 *   Camera::initialize()
 *****/
int
Camera_init(capref ref, unsigned int nwidth, unsigned int nheight)
{
  Camera *obj = (Camera *)ref;
  return obj->initialize(nwidth,nheight);
}

int
Camera_setup(capref ref)
{
  Camera *obj = (Camera *)ref;
  return obj->initialize();
}

/****f* libcamera/Camera_close
 * SYNOPSIS
 *   void Camera_close(capref ref)
 * DESCRIPTION
 *   C wrapper for Camera::close().
 * SEE ALSO
 *   Camera::close()
 *****/
void
Camera_close(capref ref, void)
{
  Camera *obj = (Camera *)ref;
  obj->close();
}

/****f* libcamera/Camera_capture
 * SYNOPSIS
 *   Picture *Camera_capture(capref ref)
 * DESCRIPTION
 *   C wrapper for Camera::capture().
 * SEE ALSO
 *   Camera_get(), Camera::capture(), Camera::get()
 *****/
Picture *
Camera_capture(capref ref)
{
  Camera *obj = (Camera *)ref;
  return obj->capture();
}

/****f* libcamera/Camera_get
 * SYNOPSIS
 *   Picture *Camera_get(capref ref)
 * DESCRIPTION
 *   C wrapper for Camera::get().
 * SEE ALSO
 *   Camera_capture(), Camera::get(), , Camera::capture()
 *****/
Picture *
Camera_get(capref ref)
{
  Camera *obj = (Camera *)ref;
  return obj->get();
}

/****f* libcamera/Camera_getTimestamp
 * SYNOPSIS
 *   long Camera_getTimestamp(capref ref)
 * DESCRIPTION
 *   C wrapper for Camera::getTimestamp().
 * SEE ALSO
 *   Camera::getTimestamp()
 *****/
long
Camera_getTimestamp(capref ref)
{
  Camera *obj = (Camera *)ref;
  return obj->getTimestamp();
}

/****f* libcamera/Camera_getTime
 * SYNOPSIS
 *   long Camera_getTime(capref ref)
 * DESCRIPTION
 *   C wrapper for Camera::getTime().
 * SEE ALSO
 *   Camera::getTime()
 *****/
long
Camera_getTime(capref ref)
{
  Camera *obj = (Camera *)ref;
  return obj->getTime();
}

/****f* libcamera/Camera_getFramerate
 * SYNOPSIS
 *   double Camera_getFramerate(capref ref)
 * DESCRIPTION
 *   C wrapper for Camera::getFramerate().
 * SEE ALSO
 *   Camera::getFramerate()
 *****/
double
Camera_getFramerate(capref ref)
{
  Camera *obj = (Camera *)ref;
  return obj->getFramerate();
}

/****f* libcamera/Camera_framerateOn
 * SYNOPSIS
 *   int Camera_framerateOn(capref ref, int state)
 * DESCRIPTION
 *   C wrapper for Camera::framerateOn().
 * SEE ALSO
 *   Camera::framerateOn()
 *****/
int
Camera_framerateOn(capref ref, int state)
{
  Camera *obj = (Camera *)ref;
  return obj->framerateOn(state);
}

/****f* libcamera/Camera_getFileDescriptor
 * SYNOPSIS
 *   int Camera_getFileDescriptor(capref ref)
 * DESCRIPTION
 *   C wrapper for Camera::getFileDescriptor().
 * SEE ALSO
 *   Camera::getFileDescriptor()
 *****/
int
Camera_getFileDescriptor(capref ref)
{
  Camera *obj = (Camera *)ref;
  return obj->getFileDescriptor();
}
