/*
 * Author: Petter Reinholdtsen <pere@td.org.uit.no>
 * Date:   2000-06-01
 *
 * Toplevel framegrabber capture class.
 */

#ifndef _camera_h
#define _camera_h

#ifdef __mc68000__
/* Cross compiling to the Eyebot */
#define HAVE_EYEBOT
#else
//#define HAVE_V4L2
#define HAVE_EYEBOT
#define HAVE_DUMMYCAMERA
#endif

#include "picture.h"

/****h* libcamera/0_module
 *
 * DESCRIPTION
 *   Superclass for all camera drivers.  Subsclasses need to implement
 *   the following methods:
 *      destructor (and make sure to call Camera::close() in it)
 *      initialize() - to test if it can capture pictures
 *      close()      - to clean up after initialize
 *      capture()    - and call gotFrame() when a frame was successfully
 *                     captured.
 *      getTime()    - with as good resolution as possible.
 *   The factory method should also be updated to use the new camera
 *   driver.
 *
 * EXAMPLE
 *
 *     Camera *camera = Camera::newCamera(width, height, pix_rgb24);
 *     if (NULL == camera)
 *       fatal_error();
 *     camera->framerateOn(true);
 *     ...
 *     while (!done)
 *      {
 *         Picture *snapshot = camera->capture();
 *         double *fps = camera->getFramerate();
 *         ...
 *      }
 *     ...
 *     delete camera;
 *****/

class Camera
{
 public:
  Camera() { clear(); }
  virtual ~Camera() { close(); }

/****f* libcamera/Camera::initialize
 * SYNOPSIS
 *   bool Camera::initialize(unsigned int nwidth = 0,
 *                   unsigned int nheight = 0,
 *                   pixel_format nfmt = pix_unknown,
 *                   char *device = 0)
 * DESCRIPTION
 *   Initializes camera driver to the given resolution and pixel
 *   format.  Returns true on success and false on failure.  If
 *   nwidth or nheigth is 0, the driver can replace it with any
 *   value.  If pixel format is pix_unknown, the devices default
 *   pixel format is used.
 *
 *   device is platform dependend and mostly unused.
 * SEE ALSO
 *   close()
 *****/
  virtual bool initialize(unsigned int nwidth = 0, unsigned int nheight = 0,
			  pixel_format nfmt = pix_unknown,
			  char *device = 0) = 0;
/****f* libcamera/Camera::close
 * SYNOPSIS
 *   void Camera::close(void)
 * DESCRIPTION
 *   Release all resources allocated by initialize().  A new call to
 *   initialize is required before calling any of the other object
 *   methods in this class.
 * SEE ALSO
 *   initialize()
 *****/
  virtual void close(void) { clear(); }

/****f* libcamera/Camera::capture
 * SYNOPSIS
 *   Picture *Camera::capture(void)
 * DESCRIPTION
 *   Capture a new frame and return a pointer to this frame.
 * SEE ALSO
 *   get()
 *****/
  virtual Picture *capture(void) = 0;

/****f* libcamera/Camera::get
 * SYNOPSIS
 *   Picture *Camera::get(void)
 * DESCRIPTION
 *   Return the last captured frame.
 * SEE ALSO
 *   capture()
 *****/
  Picture *get(void) { return &current_frame; }

/****f* libcamera/Camera::getTimestamp
 * SYNOPSIS
 *   long Camera::getTimestamp(void)
 * DESCRIPTION
 *   Return a best guess on the currents frames timestamp in 1/1000
 *   seconds since some time in the past.
 * SEE ALSO
 *   getTime()
 *****/
  long getTimestamp(void) { return timestamp; }

/****f* libcamera/Camera::getFramerate
 * SYNOPSIS
 *   double Camera::getFramerate(void)
 * DESCRIPTION
 *   Return current framerate in frames/second if frame rate
 *   calculation is turned on, othervise return 0.0.
 * SEE ALSO
 *   framerateOn()
 *****/
  double getFramerate(void) { return framerate; }

/****f* libcamera/Camera::framerateOn
 * SYNOPSIS
 *   bool Camera::framerateOn(bool state)
 * DESCRIPTION
 *   Turn on or off frame rate calculation.  Default is 'off'.
 * RETURN VALUE
 *   Return the old state before turngin framerate calculation on or off.
 * SEE ALSO
 *   getFramerate()
 *****/
  bool framerateOn(bool state)
  {
    bool old = calcFramerate;
    calcFramerate = state;
    if (!calcFramerate)
      {
	timestamp = 0;
	framerate = 0.0;
      }
    return old;
  };

/****f* libcamera/Camera::getFileDescriptor
 * SYNOPSIS
 *   int Camera::getFileDescriptor(void)
 * DESCRIPTION
 *   Return a select()able file descriptor assosiated with this
 *   device, or -1 if no such file descriptor exists.
 *****/
  virtual int getFileDescriptor(void) { return -1; };

/****f* libcamera/Camera::getTime
 * SYNOPSIS
 *   long Camera::getTime(void)
 * DESCRIPTION
 *   Get current time in 1/1000 seconds since some time in the past.
 *   This value is used for timestamping frames and calculating
 *   frame rate.  The actual resolution is platform dependent.
 *****/
  virtual long getTime(void) = 0;

/****f* libcamera/Camera::newCamera
 * SYNOPSIS
 *   static Camera *Camera::newCamera(unsigned int nwidth,
 *                                    unsigned int nheight,
 *                                    pixel_format nfmt)
 * DESCRIPTION
 *   Camera fabric.  Based on the current platform, the camera drivers
 *   compiled into the library and the setting of environment variable
 *   CAMERA, return a pointer to a working camera implementation
 *   supporting the given resolution and pixel format.  Use resolution
 *   0x0 and format pix_unknown to get the dirivers default values.
 *   Call delete on the pointer when it is no longer needed.
 * RETURN VALUE
 *   Returns NULL if no working camera driver could be found.
 *****/
  static Camera *newCamera(unsigned int nwidth, unsigned int nheight,
			   pixel_format nfmt);
  
 protected:
  Picture current_frame;
  bool calcFramerate;
  bool debugging;

  void gotFrame(void)
    {
      if (!calcFramerate || /* Not calculating frame rate */
	  0 == timestamp)   /* or first time */
        {
          timestamp = getTime();
          return;
        }

      /* The frame rate is calculated using the TCP RTT algorithm */
      long now = getTime();
      double fr = (1000.0 / (now - timestamp));
      if (0.0 == framerate)
        framerate = fr; /* second time */
      else
        framerate = 0.9 * framerate + 0.1 * fr;
      timestamp = now;
    }

 private:
  long timestamp;
  double framerate;
  void clear(void) {
    current_frame.width = 0;
    current_frame.height = 0;
    current_frame.format = pix_unknown;
    current_frame.data = 0;
    current_frame.datasize = 0;
    framerateOn(false); 
    debugging = true;
  }
};

#endif /* _camera_h */
