#include "dummycamera.hh"

#ifdef HAVE_DUMMYCAMERA

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/time.h>

#include <string>
/* ANSI C++ uses ostringstream instead - this is an G++ extention */
#include <strstream>

string imagedir = "images/";

// files to use as frames for fake camera driver
static char *fake_files[] = {
  "img0.ppm",
  "img1.ppm",
  "img2.ppm",
  "img3.ppm",
  "img4.ppm",
  "img5.ppm",
  "img6.ppm",
  "img7.ppm",
  "img8.ppm"
};

#define ARRAY_SIZE(a) (int)(sizeof(a)/sizeof((a)[0]))

/*****f* libcamera/DummyCamera::swapRgbToBgr
 * SYNOPSIS
 *   void DummyCamera::swapRgbToBgr(Picture *img)
 * DESCRIPTION
 *   Convert 24 or 32 bit Picture RGB  data to BGR data.
 *****/
void
DummyCamera::swapRgbToBgr(Picture *img)
{
  for (int i = 0; i < img->datasize; i+=img->bytes_per_pixel)
    {
      unsigned char val = img->data[i];
      img->data[i] = img->data[i+2];
      img->data[i+2] = val;
    }
}

// maximum allowable line length when reading a ppm file
#define PPM_MAX_LINE 70

/****f* libcamera/DummyCamera::loadImage
 * SYNOPSIS
 *   int DummyCamera::loadImage(const char *fname, Picture * img)
 * DESCRIPTION
 *   Quick and dirty PPM picture loader.
 ****/
int
DummyCamera::loadImage(const char *fname, Picture * img)
{
    int depth, newsize;
    int width, height;
    FILE *fp;
    char line[PPM_MAX_LINE+1];

    fp = fopen(fname, "rb");
    if (NULL == fp) {
        return -1;
    }

    fgets(line, PPM_MAX_LINE, fp);
    if (0 != strncmp("P6", line, 2)) {
        fclose(fp);
        return -2;
    }

    do {
      fgets(line, PPM_MAX_LINE, fp);
    } while ('#' == line[0]);

    sscanf(line, "%d %d", &width, &height);

    /* 0 width or height means any / default resolution */
    if (0 == img->width)
      img->width = width;
    if (0 == img->height)
      img->height = height;

    if (img->height != height || img->width != width)
      {
	fclose(fp);
	return -3;
      }


    fgets(line, PPM_MAX_LINE, fp);
    sscanf(line, "%d", &depth);
    switch (depth)
      {
      case 255:
	switch (img->format)
	  {
	  case pix_bgr32:
	  case pix_rgb32:
	    img->bytes_per_pixel = 4;
	    break;
	  case pix_bgr24:
	  case pix_rgb24:
	    img->bytes_per_pixel = 3;
	    break;
	  default:
	    fclose(fp);
	    return -4;
	    break;
	  }
	img->bytes_per_line = img->bytes_per_pixel * img->width;
	newsize = img->width * img->height * img->bytes_per_pixel;
	break;
      default:
        fclose(fp);
        return -5;
	break;
      }

    if (newsize > img->datasize)
      { /* Need more space for the pixel data */
	if (NULL != img->data)
	  delete img->data;
	img->data = new unsigned char[newsize];
	img->datasize = newsize;
      }
    if (3 == img->bytes_per_pixel) /* 24 bit */
      fread(img->data, img->bytes_per_pixel, img->width * img->height, fp);
    else /* 32 bit */
      { /* Expand 24 bit to 32 bit */
	for (int i = 0; i < img->width * img->height; i++)
	  fread(img->data + img->bytes_per_pixel*i, 3, 1, fp);
      }
    fclose(fp);
    if (pix_bgr24 == img->format || pix_bgr32 == img->format)
      {
	printf("Converting RGB to BGR\n");
	swapRgbToBgr(img);
      }
    return 0;
}

bool
DummyCamera::initialize(unsigned int nwidth, unsigned int nheight,
		       pixel_format nfmt, char *)
{
  if (count)
    close();

  pictures = new Picture[ARRAY_SIZE(fake_files)];
  for (int i = 0; i < ARRAY_SIZE(fake_files); i++)
    {
      ostrstream filename;
      picture_init(&pictures[index], nwidth, nheight, nfmt, 0, 0, 0);
      filename << imagedir << fake_files[i] << ends;
      char *file = filename.str();
      int retval;
      if (0 != (retval = loadImage(file, &pictures[index])))
	fprintf(stderr, "Failed to load \"%s\" %d!\n", file, retval);
      else 
	index++;
      delete[] file;
    }
  count = index;
  index = 0;

  /* Where any of the loaded images in the right format? */
  if (0 < count)
    {

      /* Make sure select always report data waiting on the fd */
      int fd[2];
      if (0 == ::pipe(fd))
	{
	  char *blah = "image ready";
	  dummyfd = fd[0];
	  ::write(fd[1], blah, strlen(blah)+1);
	  ::close(fd[1]);
	}

      return true;
    }

  return false;
}

void
DummyCamera::close(void)
{
  for (int i = 0; i < count; i++)
    if (0 != pictures[i].data)
      delete pictures[i].data;
  delete pictures;
  pictures = 0;

  ::close(dummyfd);
  dummyfd = -1;

  Camera::close();
}

Picture *
DummyCamera::capture(void)
{
  if (count)
    {
      Picture *retval;
      retval = &pictures[index];
      index = (index + 1) % count;  /* set up for next frame */
      if (NULL != retval->data)
	{
	  gotFrame();
	  return retval;
	}
    }
  return NULL;
}

long
DummyCamera::getTime(void)
{
  struct timeval tv;
  gettimeofday(&tv, NULL);
  return tv.tv_sec * 1000 + tv.tv_usec / 1000;
}

#endif /* HAVE_DUMMYCAMERA */
