/* -*- C++ -*-
 *
 * Author: Petter Reinholdtsen <pere@td.org.uit.no>
 * Date:   2000-05-10
 *
 * Eyebot camera frame capture class.  Modeled after CMVision frame
 * capture class.
 *
 * Use 'free(malloc(1));' in your program if the linker complains like
 * this:
 *   /user/res/uwarobot/mc/lib/libmygcc.a(_op_new.o): In function *
 *   `__builtin_new': libgcc2.c(.text+0x2c): undefined reference to *
 *   `malloc'
 *   /user/res/uwarobot/mc/lib/libmygcc.a(_op_delete.o): In function
 *   `__builtin_delete': libgcc2.c(.text+0xe): undefined reference to
 *   `free'
 */

#include "eyebotcamera.hh"

#ifdef HAVE_EYEBOT

#include <eyebot.h>
#include <picproc.h>

bool
EyebotCamera::initialize(unsigned int nwidth, unsigned int nheight,
		      pixel_format nfmt, char *device)
{
  static colimage cimg;
  if (0 == nwidth)
    nwidth = imagecolumns;
  if (0 == nheight)
    nheight = imagerows;

  starttime = 0;
  
  /* Only support given resolution at the moment */
  if (imagecolumns != nwidth || imagerows != nheight)
    {
      if (debugging)
	LCDPrintf("Invalid resolution %dx%d\n", nwidth, nheight);
      return false;
    }

  if (NULL != device && debugging)
    LCDPrintf("Ignoring param device=%s\n", device);

  switch (CAMInit(NORMAL))
    {
    case COLCAM: /* QuickCam */
      {
	/*
	 * It just seem to work better if you do like this.  Possibly
	 * a bug in the QuickCam color driver [pere 2000-06-07]
	 */
	int b, h, s;
	CAMGet(&b, &h, &s);
	CAMSet(b, h-1, s-1);
	CAMSet(b, h, s);
	CAMMode(NOAUTOBRIGHTNESS);
	break;
      }
    case COLCAM+1:/* Eyecam */
      CAMMode(AUTOBRIGHTNESS);
      starttime = getTime();
      break;
    default:
      if (debugging)
	LCDPrintf("Unsupported camera version\n");
      return false; /* no color camera */
      break;
    }

  switch (nfmt)
    {
    case pix_rgb24: /* Normal colimage format */
      picture_init(&current_frame, nwidth, nheight, pix_rgb24, 3,
		   &cimg[0][0][0], sizeof(cimg));
      break;
    default:
      if (debugging)
	LCDPrintf("Unsupported pixel format\n");
      return false;
      break;
    }

  return true;
}

void
EyebotCamera::close(void)
{
  CAMRelease();
  Camera::close();
}

Picture *
EyebotCamera::capture(void)
{
  if (0 != CAMGetColFrame((colimage*)current_frame.data, 0))
    return NULL;
  Camera::gotFrame();
  if (0 != starttime && starttime + 20000 < getTime())
    CAMMode(NOAUTOBRIGHTNESS);
  return &current_frame;
}

long
EyebotCamera::getTime(void)
{
  return OSGetCount() * 10;
}

#endif /* HAVE_EYEBOT */
