/*=========================================================================
    capture.cc
  -------------------------------------------------------------------------
    Example code for video capture under Video4Linux II
  -------------------------------------------------------------------------
    Copyright 1999, 2000
    Anna Helena Reali Costa, James R. Bruce
    School of Computer Science
    Carnegie Mellon University
  -------------------------------------------------------------------------
    This source code is distributed "as is" with absolutely no warranty.
    See LICENSE, which should be included with this distribution.
  -------------------------------------------------------------------------
    Revision History:
      2000-02-05:  Ported to work with V4L2 API
      1999-11-23:  Quick C++ port to simplify & wrap in an object (jbruce) 
      1999-05-01:  Initial version (annar)
  =========================================================================*/

#include "v4l2camera.hh"

#ifdef HAVE_V4L2

#include <iostream>
#include <string.h>
#include <assert.h>

//==== Capture Class Implementation =======================================//

bool V4L2Camera::initialize(unsigned int nwidth, unsigned int nheight,
			 pixel_format nfmt, char *device)
{
  struct v4l2_requestbuffers req;
  int err;
  int i;

  if (NULL == device)
    device = DEFAULT_VIDEO_DEVICE;
  if (0 == nwidth)
    nwidth = DEFAULT_IMAGE_WIDTH;
  if (0 == nheight)
    nheight = DEFAULT_IMAGE_HEIGHT;

  // Open the video device
  vid_fd = open(device, O_RDONLY);
  if(vid_fd == -1){
    printf("Could not open video device [%s]\n",device);
    return(false);
  }

  /* Turn off capture */
  i = V4L2_BUF_TYPE_CAPTURE;
  err = ioctl(vid_fd, VIDIOC_STREAMOFF, &i);
  if(err){
    printf("VIDIOC_STREAMOFF returned error: %s\n",strerror(errno));
  }

  fmt.type = V4L2_BUF_TYPE_CAPTURE;
  err = ioctl(vid_fd, VIDIOC_G_FMT, &fmt);
  if(err){
    printf("G_FMT returned error: %s\n",strerror(errno));
    return(false);
  }

  fmt.fmt.pix.width = nwidth;
  fmt.fmt.pix.height = nheight;
  fmt.fmt.pix.pixelformat = map_to_v4l2_format(nfmt);
  // ioctl(vid, VIDIOC_S_FMT, &fmt);
  // Set video format
  // fmt.fmt.pix.pixelformat = nfmt;

  // need to repeat following twice?
  err = ioctl(vid_fd, VIDIOC_S_FMT, &fmt);
  if(err){
    printf("S_FMT returned error %s\n",strerror(errno));
    return(false);
  }
  assert(fmt.fmt.pix.width == nwidth && fmt.fmt.pix.height == nheight);

  // Request mmap-able capture buffers
  req.count = STREAMBUFS;
  req.type  = V4L2_BUF_TYPE_CAPTURE;
  err = ioctl(vid_fd, VIDIOC_REQBUFS, &req);
  if(err < 0 || req.count < 1){
    printf("REQBUFS returned error '%s', count %d\n",
	   strerror(errno),req.count);
    return(false);
  }

  for(i=0; i<req.count; i++){
    vimage[i].vidbuf.index = i;
    vimage[i].vidbuf.type = V4L2_BUF_TYPE_CAPTURE;
    err = ioctl(vid_fd, VIDIOC_QUERYBUF, &vimage[i].vidbuf);
    if(err < 0){
      printf("QUERYBUF returned error %s\n",strerror(errno));
      return(false);
    }

    vimage[i].data = (char*)mmap(0, vimage[i].vidbuf.length, PROT_READ,
				 MAP_SHARED, vid_fd, vimage[i].vidbuf.offset);
    if(vimage[i].data == MAP_FAILED){
      printf("mmap() returned error %s\n", strerror(errno));
      return(false);
    }
  }

  for(i=0; i<req.count; i++){
    if((err = ioctl(vid_fd, VIDIOC_QBUF, &vimage[i].vidbuf))){
      printf("QBUF returned error %s\n",strerror(errno));
      return(false);
    }
  }

  // Turn on streaming capture
  err = ioctl(vid_fd, VIDIOC_STREAMON, &vimage[0].vidbuf.type);
  if(err){
    printf("STREAMON returned error %s\n",strerror(errno));
    return(false);
  }

  picture_init(&current_frame, nwidth, nheight, pix_rgb32, 4, NULL, 0);

  return(true);
}

void V4L2Camera::close()
{
  int i,t;

  if(vid_fd >= 0){
    t = V4L2_BUF_TYPE_CAPTURE;
    ioctl(vid_fd, VIDIOC_STREAMOFF, &t);

    for(i=0; i<STREAMBUFS; i++){
      if(vimage[i].data){
        munmap(vimage[i].data,vimage[i].vidbuf.length);
      }
    }
  }

  ::close(vid_fd);
  vid_fd = -1;

  Camera::close();
}

static struct v4l2_buffer tempbuf;

Picture *V4L2Camera::capture()
{
  if (-1 == vid_fd)
    return NULL;

  // struct v4l2_buffer tempbuf;
  int err;

  fd_set          rdset;
  struct timeval  timeout;
  int		  n;

  FD_ZERO(&rdset);
  FD_SET(vid_fd, &rdset);
  timeout.tv_sec = 1;
  timeout.tv_usec = 0;
  n = select(vid_fd + 1, &rdset, NULL, NULL, &timeout);
  err = -1;
  if (n == -1)
    fprintf(stderr, "capture select error: %s.\n", strerror(errno));
  else if (n == 0)
    fprintf(stderr, "capture select timeout: %s\n", strerror(errno));
  else if (FD_ISSET(vid_fd, &rdset))
    err = 0;
  if(err) return(NULL);

  // Grab last frame
  tempbuf.type = vimage[0].vidbuf.type;
  err = ioctl(vid_fd, VIDIOC_DQBUF, &tempbuf);
  if(err) printf("DQBUF returned error %s\n",strerror(errno));

  gotFrame();

  // Set current to point to captured frame data
  current_frame.data = (unsigned char *)vimage[tempbuf.index].data;

  // Initiate the next capture
  err = ioctl(vid_fd, VIDIOC_QBUF, &tempbuf);
  if(err) printf("QBUF returned error %s\n",strerror(errno));

  if (NULL == current_frame.data)
    return NULL;

  return(&current_frame);
}

long
V4L2Camera::getTime(void)
{
  struct timeval tv;
  gettimeofday(&tv, NULL);
  return tv.tv_sec * 1000 + tv.tv_usec / 1000;
}

int
V4L2Camera::map_to_v4l2_format(pixel_format format)
{
  int v4l2_format = 0;
  switch (format)
    {
    case pix_yuv9:
      v4l2_format = V4L2_PIX_FMT_YUV410; /* YUV 4:1:0 planar */
      break;
    case pix_yuv12:
      v4l2_format = V4L2_PIX_FMT_YUV420; /* YUV 4:2:0 planar */
      break;
    case pix_yuyv:
      v4l2_format = V4L2_PIX_FMT_YUYV;   /* YUV 4:2:2 packed */
      break;
    case pix_rgb24:
      v4l2_format = V4L2_PIX_FMT_RGB24;
      break;
    case pix_rgb32:
      v4l2_format = V4L2_PIX_FMT_RGB32;
      break;
    case pix_bgr24:
      v4l2_format = V4L2_PIX_FMT_BGR24;
      break;
    case pix_bgr32:
      v4l2_format = V4L2_PIX_FMT_BGR32;
      break;
    }
  fprintf(stderr, "Mapped %d to %d\n", format, v4l2_format);
  return v4l2_format;
}

#endif /* HAVE_V4L2 */
