#ifndef _language_h
#define _language_h

/****h* language/_module
 * NAME
 *
 *   The CIIPS Glory soccer language library
 *
 * DESCRIPTION
 *   Simple but effective and flexible standard for inter-robot and
 *   Master control Program (MCP) communications.
 *
 *   Use this library to communicate internally between robots and
 *   from the overhead tracking system to the robots.
 *
 *   All coordinates are gived in millimeters, where positive Y runs
 *   from origo to opponents goal, and positive X runs from origo to
 *   the right when seen from the home goal.
 *
 *       [opponent]
 *           +
 *
 *           |
 *           Y
 *           |
 *      - ---+-X- +
 *           |
 *           |
 *           |
 *
 *           -
 *
 *         [home]
 *
 *   Velocity is given in mm/second. Angles are anticlockwise and the
 *   unit is radians * 1024.  Global angles are gives relative to the
 *   positive X axes.
 *
 *   The API is described in parselanguage.h, and the events are
 *   described in language.h.
 *
 *   Protocol:
 *
 *     |<timestamp>|<size><type><type-data>|<size><type><type-data>...|
 *
 *   size is number of bytes following the size byte (sizeof(<type>) +
 *   sizeof(<type-data>).  All types are packed as mc68k endian.
 *
 * EXAMPLE
 *
 *      void my_lang_handler(langEvent *event, void *ref)
 *      {
 *         ...
 *      }
 *      ...
 *      lang_channel channel;
 *      ...
 *      channel = lang_init();
 *      if (0 == channel)
 *        fatalerror();
 *      while (!done)
 *        {
 *          int robotid;
 *          void *ref;
 *          ...
 *          for (robotid = 0; robotid < maxrobots; robotid++)
 *            {
 *               langEventTrack event;
 *               fillRobotInfo(event, robotid);
 *               lang_send(channel, (langEvent*)&event);
 *            }
 *          lang_sendFlush(channel);
 *          ...
 *          if (lang_messageWaiting(channel))
 *             lang_receive(channel, my_lang_handler, ref);
 *        }
 *      lang_release(channel);
 *
 * AUTHOR:
 *    Stephen Humble - initial list of messages (2000)
 *    Petter Reinholdtsen - language API and message packing/unpacking. (2000)
 *****/

/****s* language/langEventPing
 * DESCRIPTION
 *   Ping is a test event. The recipiant is expected to return a
 *   pong event.
 * SEE ALSO
 *   langEventPong()
 * SOURCE
 */
typedef struct {
  unsigned char type;
  int timestamp;
  unsigned char to;
  unsigned char from;
} langEventPing;
/*****/

/****s* language/langEventPong
 * DESCRIPTION
 *   Pong is a test event sent as a reply to a Ping message.  The
 *   timestamp from the ping packet is the only new member of the pong
 *   message.
 * SEE ALSO
 *   langEventPing()
 * SOURCE
 */
typedef struct {
  unsigned char type;
  int timestamp;
  unsigned char to;
  unsigned char from;

  int time;
} langEventPong;
/*****/

/****s* language/langEventTrack
 * DESCRIPTION
 *  Track messages are used to describe position and velocity of an
 *  object(ball robot etc).  Position x,y are in millimeters.
 *
 *  Track is used to tell a x,y position and v=velocity of movement
 *  a is direction in angles.
 *
 *  name is the description of the object ie cLANBall ,cLANRobot,cLANGoal.
 *  num is a sub idenfifier usefull where there is .>1 instance of an object
 * SEE ALSO
 *   lang_rad2angle(), lang_deg2angle(), lang_angle2rad(), lang_angle2deg()
 * SOURCE
 */
typedef struct {
  unsigned char type;
  int timestamp;
  unsigned char to;
  unsigned char from;

  unsigned char name; /* Object type (ball, friend, foe) */
  unsigned char num;  /* Object id 1..255 - 0 is unknown */
  short int x, y;     /* in mm relative to field center */
  short int heading;  /* counter-clockwise, 0 is along positive X axis */
  short int velocity; /* mm/second */
} langEventTrack;

enum { /* Track / position objects */
  langNameBall          = 'O',
  langNameFriend        = 'F',
  langNameEnemy         = 'E',
  langNameBGoal         = 'G',
  langNameBDefence      = 'g',
  langNameYGoal         = 'H',
  langNameYDefence      = 'h',
  langNameOrigin        = '0',
  langNameCursor        = 'x',
  langNameZone          = 'Z',
  langNameWall          = 'W',
  langNamePath          = 'P',

  langNameCornerYLeft   = 'y',
  langNameCornerYRight  = 'Y',
  langNameCornerBLeft   = 'b',
  langNameCornerBRight  = 'B'
};
/*****/

/****s* language/langEventCommand
 * DESCRIPTION
 *   Issue a command to one or all of the robots.
 * SOURCE
 */
typedef struct {
  unsigned char type;
  int timestamp;
  unsigned char to;
  unsigned char from;

  unsigned char cmd;
  unsigned char pri;
  unsigned char name;
  unsigned char num;
  unsigned char name2;
  unsigned char num2;
} langEventCommand;
/*****/

/****s* language/langEventPowerStatus
 * DESCRIPTION
 *   Report the current power (battery) status of the sending robot.
 * SOURCE
 */
typedef struct {
  unsigned char type;
  int timestamp;
  unsigned char to;
  unsigned char from;

  unsigned short int level; /* 0-1024 */
} langEventPowerStatus;
/*****/

/****s* language/langEventText
 * DESCRIPTION
 *   Event for raw data and text string transfers used to print
 *   messages, log error messages on the MCP or display messages on
 *   the eyebot display.  Text strings should be transfered with the
 *   trailing zero byte.
 * SEE ALSO
 *   langEventTrack()
 * SOURCE
 */
typedef struct {
  unsigned char type;
  int timestamp;
  unsigned char to;
  unsigned char from;

  unsigned char file;   /* describes what to do with the text */
  unsigned char length; /* Length of data */
  char *data;           /* Pointer to the text data. */
} langEventText;

enum { /* text file types */
  langTextFilePrint = 0,
  langTextFileError = 1,
  langTextFileLog   = 2
};
/*****/

/****s* language/langEventAskTrack
 * DESCRIPTION
 *   Request for track event.  Used to prompt another robot or the
 *   overhead tracking system for the location of something.
 *
 *   name is the description of the object ie cLANBall ,cLANRobot,cLANGoal.
 *   num is a sub identifier usefull where there is .>1 instance of an object
 * SEE ALSO
 *   langEventTrack()
 * SOURCE
 */
typedef struct{
  unsigned char type;
  int timestamp;
  unsigned char to;
  unsigned char from;

  unsigned char name;
  unsigned char num;
} langEventAskTrack;
/*****/

/****s* language/langEventPlayMode
 * DESCRIPTION
 *   Notification of a change in play mode.  Mostly messages
 *   originating from the referee.
 * SOURCE */
typedef struct{
  unsigned char type;
  int timestamp;
  unsigned char to;
  unsigned char from;

  unsigned char mode;
} langEventPlayMode;

/* From the simulation league soccer server protocol yellow=left,blue=right*/
enum {
  langModePlayBlue = 0,  /* direction */
  langModePlayYellow,

  langModeBeforeKickOff, /* play modes */

  langModeKickOffBlue,
  langModeKickOffYellow,

  langModeFreeKickBlue,
  langModeFreeKickYellow,

  langModeHalfTime,
  langModeTimeUp,

  langModeGoalBlue,
  langModeGoalYellow
};
/*****/

/****s* language/langEventUnknown
 * DESCRIPTION
 *   Unknown event.  Used when a unknown/unhandled event was received
 *   by lang_receive().  length specifies the length of data, and data
 *   points to the message received (after the 'type' byte).
 * SEE ALSO
 *   lang_receive()
 * SOURCE
 */
typedef struct
{
  unsigned char type;
  int timestamp;
  unsigned char to;
  unsigned char from;

  unsigned char length;
  unsigned char *data;
} langEventUnknown;
/*****/

/****s* language/langEventAny
 * DESCRIPTION
 *   The common members in all event structs.  Time is measured in
 *   1/100 seconds since some time in the past.  Each sender might use
 *   different reference time.
 * SOURCE
 */
typedef struct {
  unsigned char type;
  int timestamp;
  unsigned char to;
  unsigned char from;
} langEventAny;
/*****/

/****s* language/langEvent
 * DESCRIPTION
 *   The 'super type' of all event structs.  A union of all langEvent
 *   structs.  The 'type' member is the event type.  These are named
 *   langTypeName where 'Name' matches the event name.  Type in a
 *   langEventPing event would then be langTypePing.
 * SOURCE
 */
typedef union {
  unsigned char type;
  langEventAny any;

  langEventPing ping;
  langEventPing pong;
  langEventTrack track;
  langEventCommand command;
  langEventPowerStatus powerstatus;
  langEventText text;
  langEventPlayMode playmode;

  /* Unhandled at the moment */
  langEventAskTrack    asktrack;

  langEventUnknown unknown;
} langEvent;

enum {
  langTypeCommand     = 'C',
  langTypeText        = 'd',
  langTypePing        = 'p',
  langTypePong        = 'P',
  langTypeAskTrack    = 't',
  langTypeTrack       = 'T',
  langTypeUnknown     = 'u',
  langTypePowerStatus = 'w',
  langTypePlayMode    = 'm'
};

/*****/

/* ******************* Yet to be converted ****************************** */


/* ******************* BYTE CODES FOR OBJECTS *************************** */
/*  Definitions of name codes to be used in Track and Command messages */
/*  if there are multiple units of the same object the */
/*  objects are numbered using the number byte */
/*  cnLANCursor is an imaginary location which can for example */
/*  be used by the move command - usefull for experimentation */
/*  and as a simple robot control method */
/* all names below are for zones */
#define cLANPlayPosition 'p'
#define cLANField 'f'

/* names below are for track classification type */
#define cLANZone 'z'
#define cLANAlive 'A'

/* ************************** COMMAND MESSAGE ************************* */
/* type code for sLANCommand message's */
/* sLANCommand is the structure used to contain a command. */
/* the command message is a 2 operand (ie source-destination) operation
synactic structure with priority information. Some examples of this are
given below */
/*  codes for sLANCommand->cmd */
/*  2 names are given for each command so that a robot could for example */
/*  be told to Block an enemy robot from the Ball, Intercept a Ball, */
/*  Follow a team member, Kick a ball at an enemy, */
/*  deflect the ball at the goals, Avoid an area of the field,  */
/*  or many other possibilities */
/*  Spin and Wave are used to identify a robot whose location is not known */
/*  Wait ,Start and Move are used for ROBOCUP to enable us to comply with */
/*  robocup ruels for kickoff and penalty shots */
/*  Avoid ,Kick , Follow , Push,Intercept and Deflect are high level  */
/*  commands for smart robots with some on board intelligence, */
enum {
  langCmdSpin      = 'S',
  langCmdWait      = 'W',
  langCmdStart     = 's',
  langCmdMove      = 'M',
  langCmdAvoid     = 'A',
  langCmdKick      = 'k',
  langCmdFollow    = 'a',
  langCmdBlock     = 'B',
  langCmdPush      = 'P',
  langCmdIntercept = 'I',
  langCmdDeflect   = 'D',
/* commands below are some other robot commands */
  langCmdLearnPath = 'L',
  langCmdDrivePath = 'R',
  langCmdBackup    = 'b',
  langCmdCalibrate = 'c',
  langCmdExplore   = 'e',
  langCmdBeserk    = 'K'
};

/*  codes for sLANCommand->pri command modifier code - */
/*  used to specify how to treat a command -  */
/*  commands can be qued , executed immediatlly , or added to a list of */
/*  directives. */
/*  For example you could tell a robot to Allways move inside a goal */
/*  zone, Spin arround Now, kick a ball then move to a location. */
enum {
  langPriNow     = 'N',
  langPriNext    = 'n',
  langPriAllways = 'Z',
  langPriCancel  = 'C'
};

/*  codes below are used to indicate various messages which tell robot */
/*  about it's environment - probably only need to be called once or */
/*  twice per game walls, zones on the field , - all of these are name */
/*  based , these routines allow us to provide robots with additional */
/*  information about the soccer field */
/*  not yet complete !!!!! */
/* this is used to tell robots the color of following items being described */
/*  struct sLANColor{ */
/*    char type,size; */
/*    unsigned char red,green,blue; */
/*  } */

#endif /* _language_h */
