/*
 * Author: Petter Reinholdtsen <pere@td.org.uit.no>
 * Date:   2000-07-30
 *
 * Language logger program.  Connect to radiod, and log every language
 * message to stdout.
 */

#include <eyebot.h>
#include "parselanguage.h"

#include <stdio.h>
#include <unistd.h>
#include <signal.h>

int
lang_handler(langEvent *event, void *ref)
{
  ref = 0; /* Quiet down compiler */
  if (!event)
    return -1;
  printf("%d->%d: %d ", event->any.from, event->any.to, event->any.timestamp);
  switch (event->type)
    {
    case langTypeCommand:
      {
        langEventCommand *command = (langEventCommand *)event;
        printf("command ");
        switch (command->cmd)
          {
          case langCmdStart:
            printf("start ");
            break;
          case langCmdWait:
            printf("wait ");
            break;
          default:
            printf("[unknown] ");
            break;
          }
        break;
      }
    case langTypeTrack:
      {
        langEventTrack *track = (langEventTrack *)event;
        printf("track ");
        switch (track->name)
          {
          case langNameBall:
            printf("ball ");
            break;
          case langNameFriend:
            printf("friend ");
            break;
          case langNameEnemy:
            printf("enemy ");
            break;
          default:
            printf("[unknown] ");
            break;
          }
        printf("%d (%dx%d) %d %d", track->num, track->x, track->y,
               track->heading, track->velocity);
        break;
      }
    case langTypePowerStatus:
      {
        langEventPowerStatus *status = (langEventPowerStatus *)event;
        printf("powerstatus %d", status->level);
        break;
      }
    case langTypeText:
      { /* Make sure to match langTextFile* order in language.h */
        const char *files[] = {"print", "error", "log"};
        langEventText *text = (langEventText *)event;
        printf("text %s \"%s\"", files[text->file], text->data);
        break;
      }
    case langTypeUnknown:
      {
        langEventUnknown *unknown = (langEventUnknown *)event;
        int i;
        printf("Unknown pkg len=%d:\"", unknown->length);
        for (i = 0; i < unknown->length; i++)
          printf(" %02x", unknown->data[i]);
        printf("\" ");
        break;
      }
    default:
      printf("Unhandled type %d", event->type);
      break;
    }
  printf("\n");
  return 0;
}

static int running = 1;
static void (*oldhandler)(int);

void
sighandler(int signum)
{
  running = 0;
  if (SIG_IGN != oldhandler && SIG_DFL != oldhandler)
    oldhandler(signum);
}

int
main()
{
  lang_channel ch;
  printf("Language logger\n");

  oldhandler = signal(SIGINT, sighandler);
  if (SIG_ERR == oldhandler) {
    printf("Failed to install signal handler\n");
    return 1;
  }

  ch = lang_init();
  if (0 != ch)
    printf("lang_init() ok\n");
  else
    {
      printf("lang_init() failed\n");
      return 2;
    }

  while(running)
    {
      if (lang_messageWaiting(ch))
        lang_receive(ch, lang_handler, 0);
      sleep(1);
    }

  lang_release(ch);
  return 0;
}
