#include <eyebot.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <time.h>
#include <assert.h>

#include "parselanguage.h"

#ifdef TEST
#undef RADIOInit
#undef RADIOTerm
#undef RADIOCheck
#undef RADIORecv
#undef RADIOSend
#define RADIOInit() 0
#define RADIOTerm() 0
#define RADIOCheck() 0
#define RADIORecv(t, m, l) 1
#define RADIOSend(t, l, m) 0
#endif

static const int CHANNEL = 0xabba;

/*****i* language/pack
 * SYNOPSIS
 *   int pack(BYTE *buffer, const char *format, ...);
 * DESCRIPTION
 *   Internal function.
 *
 *   Pack values into m68k endian format for network transport.
 *   Returns number of bytes packed into the buffer.  Supports the
 *   following formats codes:
 *     d - 32 bit signed int
 *     D - 32 bit unsigned int
 *     w - 16 bit signed short
 *     W - 16 bit unsigned short
 *     c - 8 bit signed char
 *     C - 8 bit unsigned char
 *     f - 32 bit IEEE floating point
 * SEE ALSO
 *   unpack(), lang_send()
 *****/
static int
pack(BYTE *buffer, const char *format, ...)
{
  BYTE *bufstart = buffer;
  int pos = 0;
  va_list ap;
  va_start(ap, format);

  while ('\0' != format[pos])
    {
      switch (format[pos++])
        {
        case 'c':
          {
            char val = va_arg(ap, char);
            *buffer++ = (val & 0xFF);
            break;
          }
        case 'C':
          {
            unsigned char val = va_arg(ap, unsigned char);
            *buffer++ = (val & 0xFF);
            break;
          }
        case 'w':
          {
            short val = va_arg(ap, short);
            *buffer++ = (val & 0xFF00) >> 8;
            *buffer++ = (val & 0x00FF);
            break;
          }
        case 'W':
          {
            unsigned short val = va_arg(ap, unsigned short);
            *buffer++ = (val & 0xFF00) >> 8;
            *buffer++ = (val & 0x00FF);
            break;
          }
        case 'd':
          {
            int val = va_arg(ap, int);
            *buffer++ = (val & 0xFF000000) >> 24;
            *buffer++ = (val & 0x00FF0000) >> 16;
            *buffer++ = (val & 0x0000FF00) >>  8;
            *buffer++ = (val & 0x000000FF);
            break;
          }
        case 'D':
          {
            unsigned int val = va_arg(ap, unsigned int);
            *buffer++ = (val & 0xFF000000) >> 24;
            *buffer++ = (val & 0x00FF0000) >> 16;
            *buffer++ = (val & 0x0000FF00) >>  8;
            *buffer++ = (val & 0x000000FF);
            break;
          }
        case 'f':
          {
            union {
              unsigned int val;
              float f;
            } mem;
            mem.f = va_arg(ap, float);
            assert(sizeof(float) == sizeof(unsigned int));
            *buffer++ = (mem.val & 0xFF000000) >> 24;
            *buffer++ = (mem.val & 0x00FF0000) >> 16;
            *buffer++ = (mem.val & 0x0000FF00) >>  8;
            *buffer++ = (mem.val & 0x000000FF);
          }
          break;
        default:
          printf("Unknown format in pack()!\n");
          /* XXX Not sure how to return error message */
          break;
        } /* switch (format[pos++]) */
    } /* while (0 != format[pos]) */
  va_end(ap);
  return buffer - bufstart;
}

/*****i* language/unpack
 * SYNOPSIS
 *   int unpack(BYTE *buffer, const char *format, ...);
 * DESCRIPTION
 *   Internal function.
 *
 *   Unpack the values stored using pack().  Supports the same format
 *   codes as pack(), but requires pointers to the variables to store
 *   values in them.  Return number of bytes unpacked from the buffer.
 * SEE ALSO
 *   pack(), lang_receive()
 *****/
static int
unpack(BYTE *buffer, const char *format, ...)
{
  BYTE *bufstart = buffer;
  int pos = 0;
  va_list ap;
  va_start(ap, format);

  while (0 != format[pos])
    {
      switch (format[pos++])
        {
        case 'c':
          {
            char *valp = va_arg(ap, char *);
            *valp = *buffer++;
            break;
          }
        case 'C':
          {
            unsigned char *valp = va_arg(ap, unsigned char *);
            *valp = *buffer++;
            break;
          }
        case 'w':
          {
            short *valp = va_arg(ap, short *);
            *valp = (buffer[0] << 8) | buffer[1];
            buffer += 2;
            break;
          }
        case 'W':
          {
            unsigned short *valp = va_arg(ap, unsigned short *);
            *valp = (buffer[0] << 8) | buffer[1];
            buffer += 2;
            break;
          }
        case 'd':
          {
            int *valp = va_arg(ap, int *);
            *valp =
              (buffer[0] << 24) | (buffer[1] << 16) |
              (buffer[2] <<  8) |  buffer[3];
            buffer += 4;
            break;
          }
        case 'D':
          {
            unsigned int *valp = va_arg(ap, unsigned int *);
            *valp =
              (buffer[0] << 24) | (buffer[1] << 16) |
              (buffer[2] <<  8) |  buffer[3];
            buffer += 4;
            break;
          }
        case 'f':
          {
            union {
              unsigned int val;
              float f;
            } mem;
            float *valp = va_arg(ap, float *);
            assert(sizeof(float) == sizeof(unsigned int));
            mem.val =
              (buffer[0] << 24) | (buffer[1] << 16) |
              (buffer[2] <<  8) |  buffer[3];
            buffer += 4;
            *valp = mem.f;
          }
          break;
        default:
          printf("Unknown format in unpack()!\n");
          /* XXX Not sure how to return error message */
          break;
        } /* switch (format[pos++]) */
    } /* while (0 != format[pos]) */
  return buffer - bufstart;
}

static BYTE strbuf[MSGMAXLEN];
static int bufpos = 0;
static int lasttimestamp = 0;
static int lastto = -1;

/****f* language/lang_init
 * SYNOPSIS
 *   lang_channel lang_init(void);
 * DESCRIPTION
 *   Initialize the language library.  Call this and make sure the
 *   return value is not 0 before calling any of the other language
 *   functions.
 * RESULT
 *   channel handle on success, 0 on failure.
 * SEE ALSO
 *   lang_send(), lang_release()
 *****/
lang_channel
lang_init(void)
{
  bufpos = 0;
  lasttimestamp = 0;
  lastto = -1;

  if (0 == RADIOInit())
    return CHANNEL;
  else
    return 0;
}

/****f* language/lang_release
 * SYNOPSIS
 *   int lang_release(lang_channel ch);
 * DESCRIPTION
 *   Release all resources allocated by lang_init().  After this
 *   function returns, call lang_init() before using this channel
 *   again.
 * RESULT
 *   0 on success, -1 of failure.
 * SEE ALSO
 *   lang_init()
 *****/
int
lang_release(lang_channel ch)
{
  if (ch != CHANNEL)
    return -1;
  return RADIOTerm();
}

/****f* language/lang_messageWaiting
 * SYNOPSIS
 *   int lang_messageWaiting(lang_channel ch);
 * DESCRIPTION
 *   Check if there are radio messages waiting to be processed.  If
 *   this function return true (1), lang_receive() will not block.
 * RESULT
 *   0 if message queue is empty, 1 if there might be messages waiting
 *   or -1 if the channel is bogus.
 * SEE ALSO
 *   lang_init(), lang_receive()
 *****/
int
lang_messageWaiting(lang_channel ch)
{
  if (ch != CHANNEL)
    return -1;

  return RADIOCheck();
}

/****i* language/currentTime
 * DESCRIPTION
 *   Return the current time in 1/100 seconds since some time in the
 *   past.
 ****/
static int
currentTime(void)
{
#if defined(__mc68000__)
  return OSGetCount();
#else
  static int ref = 0;
  if (0 == ref)
    ref = time(NULL);
  return (time(NULL)-ref)*100; /* XXX Make this more accurate */
#endif
}

/****f* language/lang_receive
 * SYNOPSIS
 *   typedef int (*lang_handler_t)(langEvent *event, void *ref);
 *   int lang_receive(lang_channel ch, lang_handler_t handler, void *ref);
 * DESCRIPTION
 *   Parse radio messages and pass them on to the language handler as
 *   native structs.
 * PARAMETERS
 *   handler  Function pointer to call for each received event.
 *   ref      Reference passed on to the handler.
 * RESULT
 *   Number of events received.
 * SEE ALSO
 *   lang_send()
 *****/
int
lang_receive(lang_channel ch, lang_handler_t handler, void *ref)
{
  BYTE from;
  BYTE to = BROADCAST; /* do not know who this packet was sent to */
  int msglen;
  BYTE msg[MSGMAXLEN];
  int timestamp;
  int pos = 0;
  int processed=0;

  if (ch != CHANNEL)
    return 0;

  while (RADIOCheck())
    {
      RADIORecv(&from, &msglen, msg);

      if (msglen < 4)
        continue;

      /* Read timestamp and move to <type> position */
      pos = unpack(msg, "d", &timestamp);

      while (pos < msglen)
      {
        int size = msg[pos++];
        switch (msg[pos++]) /* Type */
        {
        case langTypeTrack: /* ball and robot position */
          {
            langEventTrack event;

            unpack(&msg[pos], "CCwwww", &event.name, &event.num,
                   &event.x, &event.y, &event.heading, &event.velocity);
            event.timestamp = timestamp;
            event.type = langTypeTrack;
            event.from = from;
            event.to   = to;
            handler((langEvent*)&event, ref);
            break;
          }
        case langTypeCommand:
          {
            langEventCommand cmd;
            unpack(&msg[pos], "CCCCCC", &cmd.cmd, &cmd.pri, &cmd.name,
                   &cmd.num, &cmd.name2, &cmd.num2);
            cmd.timestamp = timestamp;
            cmd.type = langTypeCommand;
            cmd.from = from;
            cmd.to   = to;
            handler((langEvent*)&cmd, ref);
            break;
          }
        case langTypePing: /* Send reply, keep timestamp */
          {
            langEventPong pong;
            pong.timestamp = currentTime();
            pong.type = langTypePong;
            pong.to   = from;
            pong.time = timestamp;
            lang_send(ch, (langEvent*)&pong);
            lang_sendFlush(ch);
            break;
          }
        case langTypePong:
          {
            langEventPong event;

            unpack(&msg[pos], "d", &event.time);

            event.timestamp = timestamp;
            event.type = langTypePong;
            event.from = from;
            event.to   = to;

            handler((langEvent*)&event, ref);
            break;
          }
        case langTypePowerStatus:
          {
            langEventPowerStatus event;
            unpack(&msg[pos], "w", &event.level);

            event.timestamp = timestamp;
            event.type = langTypePowerStatus;
            event.from = from;
            event.to   = to;

            handler((langEvent*)&event, ref);
            break;
          }
        case langTypeText:
          {
            langEventText event;

            unpack(&msg[pos], "C", &event.file);

            event.length = size - 1;
            event.data = (char*)&msg[pos+1];

            event.timestamp = timestamp;
            event.type = langTypeText;
            event.from = from;
            event.to   = to;

            handler((langEvent*)&event, ref);
            break;
          }
        case langTypePlayMode:
          {
            langEventPlayMode event;
            unpack(&msg[pos], "C", &event.mode);

            event.timestamp = timestamp;
            event.type = langTypePlayMode;
            event.from = from;
            event.to   = to;

            handler((langEvent*)&event, ref);
            break;
          }
        default:
          {
            langEventUnknown event;

            event.timestamp = timestamp;
            event.type = langTypeUnknown;
            event.from = from;
            event.to   = to;
            event.data = &msg[0];
            event.length = msglen;
            pos = msglen;

            handler((langEvent*)&event, ref);
            break;
          }
        }
        pos += size;
        processed++;
      }
    } /* while (RADIOCheck) */
  return processed;
}

/****f* language/lang_send
 * SYNOPSIS
 *   int lang_send(lang_channel ch, langEvent *event);
 * DESCRIPTION
 *   Buffer information given in 'event' to the other soccer players.
 *   Call lang_sendFlush() to send the information.
 * PARAMETERS
 *   event  Pointer to langEvent structure, where the type describes
 *          which specific struct this is.
 * RESULT
 *   0 on success and -1 if the event type is unknown.
 * SEE ALSO
 *   lang_receive(), lang_sendFlush()
 *****/
int
lang_send(lang_channel ch, langEvent *event)
{
  if (CHANNEL != ch)
    return -1;

  /*
   * Should we continue with the current buffer, or do we need to send
   * the packed off before packing a new event?
   */
  if (0 != bufpos && (lasttimestamp != event->any.timestamp ||
                      lastto != event->any.to ||
                      bufpos+sizeof(langEvent) > sizeof(strbuf)))
    {
      if (0 != lang_sendFlush(ch))
        return -1;
    }
#if defined(TEST)
  printf(__FUNCTION__ ":%d: type=%d bufpos=%d\n", __LINE__, event->type,
         bufpos);
#endif

  if (lasttimestamp != event->any.timestamp || lastto != event->any.to)
    {
      bufpos += pack(&strbuf[bufpos], "d", event->any.timestamp);
      lasttimestamp = event->any.timestamp;
      lastto        = event->any.to;
#if defined(TEST)
      printf(__FUNCTION__ ":%d: bufpos=%d\n", __LINE__, bufpos);
#endif
    }

  switch (event->type)
    {
    case langTypeTrack:
      {
        langEventTrack *track = (langEventTrack *)event;
        strbuf[bufpos] = pack(&strbuf[bufpos+1], "CCCwwww",
                              track->type, track->name, track->num,
                              track->x, track->y,
                              track->heading, track->velocity);
        bufpos+= strbuf[bufpos]+1;
        break;
      }
    case langTypeCommand:
      {
        langEventCommand *cmd = (langEventCommand*)event;
        strbuf[bufpos] = pack(&strbuf[bufpos+1], "CCCCCCC",
                              cmd->type, cmd->cmd, cmd->pri,
                              cmd->name, cmd->num,
                              cmd->name2, cmd->num2);
        bufpos += strbuf[bufpos]+1;
        break;
      }
    case langTypePing:
      {
        /* Empty message, only size and type */
        strbuf[bufpos] = pack(&strbuf[bufpos+1], "C", langTypePing);
        bufpos+= strbuf[bufpos]+1;
        break;
      }
    case langTypePong:
      {
        langEventPong *pong = (langEventPong *)event;
        strbuf[bufpos] = pack(&strbuf[bufpos+1], "Cd", langTypePong,
                              pong->time);
        bufpos+= strbuf[bufpos]+1;
        break;
      }
    case langTypePowerStatus:
      {
        langEventPowerStatus *status = (langEventPowerStatus *)event;
        strbuf[bufpos] = pack(&strbuf[bufpos+1], "Cw",
                              status->type, status->level);
#if defined(TEST)
  printf(__FUNCTION__ ":%d: bufpos=%d\n", __LINE__, bufpos);
#endif

        bufpos+= strbuf[bufpos]+1;
#if defined(TEST)
  printf(__FUNCTION__ ":%d: bufpos=%d\n", __LINE__, bufpos);
#endif

        break;
      }
    case langTypeText:
      {
        /*
         * Send text message.  If the message is to long for one
         * packet, cut it in smaller pieces before sending it.
         */
        langEventText *text = (langEventText*)event;

        int space = sizeof(strbuf) - bufpos; /* Space left if msg buffer */

#if defined(TEST)
        printf(__FUNCTION__ ":%d: text bufpos=%d length=%d space=%d \"%s\"\n",
               __LINE__, bufpos, text->length, space, text->data);
#endif

        if (text->length + 3 > space)
          { /* Message to long for radio packet */
            char *data = text->data;
            int length = text->length;
            int pos = 0;
            while (pos < length)
              {
                char msgbuf[MSGMAXLEN - (4 + 3) + 1]; /* Max text length */
#define MIN(a,b) ((a<b) ? (a) : (b))
                int i = MIN(space-4, (int)sizeof(msgbuf)-1);

                memcpy(&msgbuf[0], &data[pos], i);
                if ('\0' != msgbuf[i-1])
                  {
                    msgbuf[i] = '\0';
                    i++;
                    pos--;
                  }
#if defined(TEST)
  printf(__FUNCTION__ ":%d: subtext bufpos=%d length=%d \"%s\"\n",
         __LINE__, bufpos, i, msgbuf);
#endif
                text->data = &msgbuf[0];
                text->length = i;
                if (0 != lang_send(ch, (langEvent *)text))
                  {
                    /* Restore packet to original state */
                    text->data = data;
                    text->length = length;
                    return -1;
                  }

                pos += i;
              }
            /* Restore packet to original state */
            text->data = data;
            text->length = length;
          }
        else
          {
            strbuf[bufpos] = pack(&strbuf[bufpos+1], "CC",
                                  text->type, text->file) + text->length;
            memcpy(&strbuf[bufpos+3], text->data, text->length);
            bufpos+= strbuf[bufpos]+1; /* Content + length byte */
          }
        break;
      }
    case langTypePlayMode:
      {
        langEventPlayMode *mode = (langEventPlayMode *)event;
        strbuf[bufpos] = pack(&strbuf[bufpos+1], "CC", langTypePlayMode,
                              mode->mode);
        bufpos+= strbuf[bufpos]+1;
        break;
      }
    default:
      return -1;
      break;
    }
  lang_sendFlush(ch);
  return 0;
}

/****f* language/lang_sendFlush
 * SYNOPSIS
 *   int lang_sendFlush(lang_channel ch);
 * DESCRIPTION
 *   Flush sendbuffer with all the information buffered by
 *   lang_send().  Does not do anything if there are no packages in
 *   the buffer.
 * RESULT
 *   0 on success and -1 on failure
 * SEE ALSO
 *   lang_send()
 *****/
int
lang_sendFlush(lang_channel ch)
{
  int retval = -1;

  if (CHANNEL != ch)
    return -1;

  if (0 < bufpos)
    retval = RADIOSend(lastto, bufpos, strbuf);

#if defined(TEST)
  printf(__FUNCTION__ ":%d: bufpos=%d\n", __LINE__, bufpos);
#endif

  if (0 == retval)
    bufpos = 0;
  return retval;
}

/****f* language/lang_transCmd
 * SYNOPSIS
 *   char *lang_transCmd(unsigned char code);
 * DESCRIPTION
 *   Returns english language strings translations of character codes.
 * RESULT
 *   Pointer to a static string.
 * SEE ALSO
 *   lang_send()
 *****/
const char *
lang_transCmd(unsigned char code)
{
  switch(code){
  case langCmdSpin:      return("Spin");
  case langCmdWait:      return("Wait");
  case langCmdStart:     return("Start");
  case langCmdMove:      return("Move");
  case langCmdAvoid :    return("Avoid");
  case langCmdKick:      return("Kick");
  case langCmdFollow:    return("Folo");
  case langCmdBlock:     return("Block");
  case langCmdPush:      return("Push");
  case langCmdIntercept: return("Inter");
  case langCmdDeflect:   return("Defl");
/* commands below are some other robot commands */
  case langCmdLearnPath: return("Learn");
  case langCmdDrivePath: return("Drive");
  case langCmdBackup:    return("Backup");
  case langCmdCalibrate: return("Cal");
  case langCmdExplore:   return("Explor");
  case langCmdBeserk:    return("Beserk");
  default:               return("?");
  }
  return("weird error");
}

/****f* language/lang_transName
 * SYNOPSIS
 *   char *lang_transName(unsigned char code);
 * DESCRIPTION
 *   Returns english language strings translations of character codes.
 * RESULT
 *   Pointer to a static string.
 * SEE ALSO
 *   lang_send()
 *****/
const char *
lang_transName(unsigned char code)
{
  switch(code){
  case langNameBall:         return("Ball");
  case langNameFriend:       return("Friend");
  case langNameEnemy:        return("Enemy");
  case langNameBGoal:        return("BlGoal");
  case langNameBDefence:     return("BlDef");
  case langNameYGoal:        return("YeGoal");
  case langNameYDefence:     return("YeDef");
  case langNameOrigin:       return("Origin");
  case langNameCursor:       return("Curs");
  case langNameCornerYLeft:  return("LYeCnr");
  case langNameCornerYRight: return("RYeCnr");
  case langNameCornerBLeft:  return("LBlCnr");
  case langNameCornerBRight: return("RBlCnr");
  case langNameZone:         return("Zone");
  case langNameWall:         return("Wall");
  default:    return("?");
  }
  return("weird error");
}

/****f* language/lang_transPri
 * SYNOPSIS
 *   char *lang_transPri(unsigned char code);
 * DESCRIPTION
 *   Returns english language strings translations of character codes.
 * RESULT
 *   Pointer to a static string.
 * SEE ALSO
 *   lang_send()
 *****/
const char *
lang_transPri(unsigned char code)
{
  switch(code){
  case langPriNow:     return("Now");
  case langPriNext:    return("next");
  case langPriAllways: return("Always");
  case langPriCancel:  return("Cancel");
  default:    return("?");
  }
  return("weird error");
}

/****f* language/lang_transType
 * SYNOPSIS
 *   char *lang_transType(unsigned char code);
 * DESCRIPTION
 *   Returns english language strings translations of character codes.
 * RESULT
 *   Pointer to a static string.
 * SEE ALSO
 *   lang_send()
 *****/
const char *
lang_transType(unsigned char code)
{
  switch(code){
  case langTypeCommand:     return("Cmd");
  case langTypeText:        return("Text");
  case langTypePing:        return("ping");
  case langTypePong:        return("Pong");
  case langTypeAskTrack:    return("AskTrack");
  case langTypeTrack:       return("Track");
  case langTypeUnknown:     return("?");
  case langTypePowerStatus: return("Power");
  default:    return("?");
  }
  return("weird error");
}

#ifdef TEST
#define testtype(b, type, tval, code, pcode) {\
  type val = (type)tval; \
  type val2; \
  type *valp = (type*)b; \
  memset(b, 0, 4); \
  pack(b, code, val); \
  if (*valp != val) \
    { \
      fprintf(stderr, "Type " #type "("code") pack test failed\n"); \
      fprintf(stderr, "  Val: " pcode " \"%02x %02x %02x %02x\"\n", \
              val, b[0], b[1], b[2], b[3]); \
    } \
  unpack(b, code, &val2); \
  if (val != val2) \
    { \
      fprintf(stderr, "Type " #type "("code") unpack test failed\n"); \
      fprintf(stderr, "  Val2: "pcode"!="pcode" \"%02x %02x %02x %02x\"\n",\
              val, val2, b[0], b[1], b[2], b[3]); \
    } \
}
void
test_pack_unpack(void)
{
  BYTE buffer[100];
  testtype(buffer, int, rand(), "d", "%d");
  testtype(buffer, unsigned int, rand(), "D", "%ud");
  testtype(buffer, short, rand(), "w", "%d");

  testtype(buffer, unsigned int, 1, "D", "%d");
  testtype(buffer, unsigned int, 0xabbabad, "D", "%d");

  testtype(buffer, short, 0, "w", "%d");
  testtype(buffer, short, 1, "w", "%d");

  testtype(buffer, char,   0, "c", "%d");
  testtype(buffer, char, 254, "c", "%d");
  testtype(buffer, BYTE,   0, "C", "%d");
  testtype(buffer, BYTE, 254, "C", "%d");
}

void
test_text(lang_channel ch, char *text)
{
  langEventText event;

  event.type = langTypeText;
  event.from = 0xAA;
  event.to = 0xF0;
  event.timestamp = 0xbaddad;

  event.file = langTextFileLog;
  event.length = strlen(text)+1;
  event.data = text;

  lang_send(ch, (langEvent*)&event);
}

void
test_lang(void)
{
  lang_channel ch = lang_init();
  langEventPowerStatus event;

  event.timestamp = 0xbaddad;
  event.type = langTypePowerStatus;
  event.from = 0xAA;
  event.to   = 0xF0;

  lang_send(ch, (langEvent*)&event);
  lang_send(ch, (langEvent*)&event);
  lang_send(ch, (langEvent*)&event);

  test_text(ch, "Testing long, long, text.  This should be broken in pieces.");

  lang_sendFlush(ch);
  lang_release(ch);
}

int
main()
{
  test_pack_unpack();
  test_lang();
  return 0;
}
#endif
