/*
 * Author: Petter Reinholdtsen <pere@td.org.uit.no>
 * Date:   2000-05-23
 *
 * Simple test progam for the soccer language library.  Listen in on
 * the communication, and print all messages.
 */

#include <eyebot.h>
#include "parselanguage.h"

#include <stdio.h>

int
lang_handler(langEvent *event, void *ref)
{
  ref = 0; /* Quiet down compiler */
  switch (event->type)
    {
    case langTypeCommand:
      {
        langEventCommand *command = (langEventCommand *)event;
        switch (command->cmd)
          {
          case langCmdStart:
            LCDPrintf("%d/%2d Go\n", command->timestamp, command->from);
            break;
          case langCmdWait:
            LCDPrintf("%d/%2d Reset\n", command->timestamp, command->from);
            break;
          }
        break;
      }
    case langTypeTrack:
      {
        langEventTrack *track = (langEventTrack *)event;
        LCDPrintf("%d/%2d %c%c %1d\n%4dx%5d/%3d\n",
                  track->timestamp, track->from, track->type,
                  track->name, track->num, track->x, track->y,
                  track->heading, track->velocity);
        break;
      }
    case langTypePowerStatus:
      {
        langEventPowerStatus *status = (langEventPowerStatus *)event;
        LCDPrintf("%d/%2d %c%4d\n", status->timestamp, status->from,
                  event->type, status->level);
        break;
      }
    case langTypeText:
      {
        langEventText *text = (langEventText *)event;

        LCDPrintf("T%d:%s\n", text->file, text->data);
        break;
      }
    case langTypeUnknown:
      {
        langEventUnknown *unknown = (langEventUnknown *)event;
        int i;
        LCDPrintf("Unknown pkg l=%d:\n", unknown->length);
        for (i = 0; i < unknown->length; i++)
          LCDPrintf("%02x ", unknown->data[i]);
        LCDPrintf("\n");
        break;
      }
    default:
      LCDPrintf("Unhandled type %d\n", event->type);
      break;
    }
  return 0;
}

void
fillBallPos(langEventTrack *ballpos, int timestamp)
{
  ballpos->to = BROADCAST;
  ballpos->timestamp = timestamp;
  ballpos->type = langTypeTrack;
  ballpos->name = langNameBall;
  ballpos->num  = 0;
  ballpos->x = 0;
  ballpos->y = 100;
  ballpos->heading  = 0;
  ballpos->velocity = -1;
}

void
sendPowerStatus(lang_channel ch, int timestamp)
{
  langEventPowerStatus event;
  event.timestamp = timestamp;
  event.type = langTypePowerStatus;
  event.to = BROADCAST;
  event.level = (short int)OSGetAD(1); /* 10 bit value */
  lang_send(ch, (langEvent *)&event);
}

int
main()
{
  int last_power_status = 0;
  int key;
  lang_channel ch;
  LCDPutString("Soccer sniffer\n");

  ch = lang_init();
  if (0 != ch)
    LCDPutString("lang_init() ok\n");
  else
    LCDPutString("lang_init() failed\n");

  LCDMenu("Go", "Rset", "Ball", "END");
  while(KEY4 != (key = KEYRead()))
    {
      int stamp = OSGetCount();
      if (lang_messageWaiting(ch))
        lang_receive(ch, lang_handler, 0);

      /* Send power/battery status every 60 second */
      if (last_power_status+6000 < stamp)
        {
          sendPowerStatus(ch,stamp);
          last_power_status = stamp;
        }

      switch (key)
        {
        case KEY1:
          {
            langEventCommand start;

            start.to = BROADCAST;
            start.timestamp = stamp;
            start.type = langTypeCommand;
            start.cmd = langCmdStart;

            lang_send(ch,(langEvent*)&start);
            LCDPrintf("%d Start\n", stamp);
            break;
          }
        case KEY2:
          {
            langEventCommand wait;

            wait.to = BROADCAST;
            wait.timestamp = stamp;
            wait.type = langTypeCommand;
            wait.cmd = langCmdWait;

            lang_send(ch,(langEvent*)&wait);
            LCDPrintf("%d Wait\n", stamp);
            break;
          }
        case KEY3:
          {
            langEventTrack ballpos;

            fillBallPos(&ballpos, stamp);

            lang_send(ch,(langEvent*)&ballpos);
            LCDPrintf("%d\ntO(x=0,y=100,phi=0)\n", stamp);
            break;
          }
        }
      lang_sendFlush(ch);
      OSSleep(5);
    }

  lang_release(ch);
  return 0;
}
