/****h* libpicproc/_module
 * NAME
 *   Picture processing library
 * DESCRIPTION
 *   Image processing functions operating on a common Picture struct.
 *   This library do not depend no any other libraries then the
 *   standard C library.
 *****/

#include "picture.h"
#include "picproc.h"

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

/****f* libpicproc/picture_init
 * SYNOPSIS
 *   #include "picproc.h"
 *   void picture_init(Picture *pic, picture_coord width, picture_coord height,
 *                     pixel_format format, int bytes_per_pixel,
 *                     unsigned char *data, unsigned int datasize)
 * DESCRIPTION
 *   Initialize all the members of a the picture struct.
 * EXAMPLE
 *   Picture pic;
 *   ...
 *   picture_init(&pic, width, heigth, format, bpp,
 *                malloc(width*heigth*bpp), width*heigth*bpp);
 ****/
void
picture_init(Picture *pic, picture_coord width, picture_coord height,
             pixel_format format, int bytes_per_pixel,
             unsigned char *data, unsigned int datasize)
{
  pic->width = width;
  pic->height = height;
  pic->format = format;
  pic->data = data;
  pic->datasize = datasize;
  pic->bytes_per_pixel = bytes_per_pixel;
  pic->bytes_per_line = bytes_per_pixel * width;

  pic->xoffset = 0;
  pic->obformat = 0;
  pic->obdata = 0;
  pic->flags = 0;
}

/****f* libpicproc/picture_new
 * SYNOPSIS
 *   #include "picproc.h"
 *   Picture *picture_new(picture_coord width, picture_coord height,
 *                        pixel_format format)
 * DESCRIPTION
 *   Allocate a new picture with the given size and format.  Call
 *   picture_delete() to release the allocated resources.
 * EXAMPLE
 *   Picture *pic = picture_new(320,240, pix_rgb24);
 * SEE ALSO
 *   picture_delete()
 ****/
Picture *
picture_new(picture_coord width, picture_coord height, pixel_format format)
{
  Picture *pic;
  unsigned int datasize = 0;
  int bytes_per_pixel = 0;
  int bytes_per_line = 0;

  switch (format)
    {
    case pix_bitmap:
      bytes_per_pixel = 0;
      datasize = width * height/8;
      bytes_per_line = width/8;
      break;
    case pix_grey4:
    case pix_grey:
      bytes_per_pixel = 1;
      datasize = width * height;
      break;
    case pix_bgr24:
    case pix_rgb24:
      bytes_per_pixel = 3;
      datasize = width * height * bytes_per_pixel;
      break;
    case pix_bgr32:
    case pix_rgb32:
    case pix_hsv32:
      bytes_per_pixel = 4;
      datasize = width * height * bytes_per_pixel;
      break;
    case pix_yuyv:
      bytes_per_pixel = 0;
      datasize = width * height * 2;
      bytes_per_line = width * 2;
      break;
    }

  if (0 == datasize) /* Unhandled pixel format */
    return 0;


  /*
   * One memory block for both, with the Picture struct first and the
   * pixel data behind this.
   */
  pic = calloc(1, sizeof(Picture)+datasize);

  if (NULL != pic)
    {
      picture_init(pic, width, height, format, bytes_per_pixel,
                   ((unsigned char *)pic)+sizeof(Picture), datasize);
      if (bytes_per_line)
        pic->bytes_per_line = bytes_per_line;
    }
  return pic;
}

/****f* libpicproc/picture_delete
 * SYNOPSIS
 *   #include "picproc.h"
 *   int picture_delete(Picture *pic)
 * DESCRIPTION
 *   Release the resources allocated by picture_new().
 * RETURN VALUE
 *   0 of success, -1 on failure.
 * SEE ALSO
 *   picture_new()
 ****/
int
picture_delete(Picture *pic)
{
  if (NULL == pic)
    return -1;
  free(pic);
  return 0;
}

/****f* libpicproc/picture_clear
 * SYNOPSIS
 *   #include "picproc.h"
 *   void picture_clear(Picture *img)
 * DESCRIPTION
 *   Make all pixels in the given picture black.
 ****/
void
picture_clear(Picture *img)
{
  if (0 == img || 0 == img->data)
    return;
  switch (img->format)
    {
    default: /* Most formats use zero as black */
      memset(img->data, 0, img->datasize);
      break;
    }
}

/****f* libpicproc/picproc_drawPixel
 * SYNOPSIS
 *   #include "picproc.h"
 *   int picproc_drawPixel(Picture *img, picture_coord column, picture_coord row,
 *                         int color)
 * DESCRIPTION
 *   Set color at pixel position x,y.  The given color must match
 *   the image format.
 * NOTE
 *   Currently only support pix_bitmap and pix_rgb24 format.
 ****/
int
picproc_drawPixel(Picture *img, picture_coord column, picture_coord row, int color)
{
  int retval = 0;
  switch (img->format)
    {
    case pix_bitmap:
      retval = picproc_bitmap_drawPixel(img, column, row, color);
      break;
    case pix_rgb24:
    case pix_rgb32:
    case pix_bgr24:
    case pix_bgr32:
      retval = picproc_rgb24_drawPixel(img, column, row, color);
      break;
    default:
      retval = -1; /* Unsupported at the moment */
      break;
    }
  return retval;
}

static void
_plotPointsOnCircle(Picture *img, picture_coord xCentre, picture_coord x,
                    picture_coord yCentre, picture_coord y,
                    int color)
{
  int left = 0 ,right = img->width, top = 0, bottom = img->height;

  if ( (xCentre+x < right) && (yCentre+y < bottom) )
    picproc_drawPixel(img, xCentre+x, yCentre+y, color);

  if ( (xCentre-x > left) && (yCentre+y < bottom) )
    picproc_drawPixel(img, xCentre-x, yCentre+y, color);

  if ( (xCentre+x < right) && (yCentre-y > top) )
    picproc_drawPixel(img, xCentre+x, yCentre-y, color);

  if ( (xCentre-x > left) && (yCentre-y > top) )
    picproc_drawPixel(img, xCentre-x, yCentre-y, color);

  if ( (xCentre+y < right) && (yCentre+x < bottom) )
    picproc_drawPixel(img, xCentre+y, yCentre+x, color);

  if ( (xCentre-y > left) && (yCentre+x < bottom) )
    picproc_drawPixel(img, xCentre-y, yCentre+x, color);

  if ( (xCentre+y < right) && (yCentre-x > top) )
    picproc_drawPixel(img, xCentre+y, yCentre-x, color);

  if ( (xCentre-y > left) && (yCentre-x > top) )
    picproc_drawPixel(img, xCentre-y, yCentre-x, color);
}

/****f* libpicproc/picproc_drawCircle
 * SYNOPSIS
 *   #include "picproc.h"
 *   int picproc_drawCircle(Picture *img, picture_coord column, picture_coord row,
 *                          int radius, int color)
 * DESCRIPTION
 *   Draw circle with the given radius and center at (x,y) using the
 *   given color.  The given color must match the image format.
 *   Calls picproc_drawPixel() to draw in the image.
 * SEE ALSO
 *   picproc_drawPixel()
 ****/
int
picproc_drawCircle(Picture *img, picture_coord column, picture_coord row,
                   int radius, int color)
{
  int x, y, p;

  if (column >= img->width || row >= img->height || radius < 0)
    return -1;

  x = 0;
  y = radius;
  p = 1-radius;
  _plotPointsOnCircle(img, column, x, row, y, color);
  while (x < y)
    {
      if (p < 0)
        {
          x = x+1;
          p = p+2*x+1;
        }
      else
        {
          x = x+1;
          y = y-1;
          p = p+2*(x-y)+1;
        }
      _plotPointsOnCircle(img,column, x, row, y, color);
    }
  return 0;
}

/****f* libpicproc/picproc_drawLine
 * SYNOPSIS
 *   #include "picproc.h"
 *   int picproc_drawLine(Picture *pic, picture_coord column1, picture_coord row1,
 *                        picture_coord column2, picture_coord row2, int color)
 * DESCRIPTION
 *   Draw line from (column1,row1) to (column2,row2) using the
 *   Bresenham Algorithm with the given color.  The given color must
 *   match the image format.  Calls picproc_drawPixel() to draw in the
 *   image.
 * RESULT
 *   0 on success and -1 if coordinates are out of range
 * SEE ALSO
 *   picproc_drawPixel()
 ****/
int
picproc_drawLine(Picture *pic, picture_coord column1, picture_coord row1,
                 picture_coord column2, picture_coord row2, int color)
{
  int x,y, deltax,deltay, xchange, ychange;
  int i, error;

  if (column1 >= pic->width || column2 >= pic->width ||
      row1 >= pic->height || row2 > pic->height)
    return -1;

  x=column1;
  y=row1;      /* starting point */
  error=0;
  deltax = column2-column1;  /* difference */
  deltay = row2-row1;

  if (deltax < 0)
    {
      xchange = -1;
      deltax = -deltax;
    }
  else
    xchange = 1;
  if (deltay < 0)
    {
      ychange = -1;
      deltay = -deltay;
    }
  else
    ychange = 1;

  if (deltax < deltay)
    {
      for (i=0; i < deltay+1; i++)
        {
          picproc_drawPixel(pic, x,y, color);
          y     += ychange;
          error += deltax;
          if (error > deltay)
            {
              x += xchange;
              error -= deltay;
            }
        }
    }
  else
    {
      for (i=0; i < deltax+1; i++)
        {
          picproc_drawPixel(pic, x,y, color);
          x     += xchange;
          error += deltay;
          if (error > deltax)
            {
              y += ychange;
              error -= deltax;
            }
        }
    }
  picproc_drawPixel(pic, column2,row2, color); /* add last pixel to line */

  return 0;
}

/****f* libpicproc/picproc_drawBox
 * SYNOPSIS
 *   #include "picproc.h"
 *   int picproc_drawBox(Picture *pic, picture_coord column1, picture_coord row1,
 *                       picture_coord column2, picture_coord row2, int color)
 * DESCRIPTION
 *   Draw box with corners at (column1,row1) and (column2,row2) using the given
 *   color.  The given color must match the image format.  Calls
 *   picproc_drawLine() to draw the edges.
 * RESULT
 *   0 on success and -1 if coordinates are out of range
 * SEE ALSO
 *   picproc_drawLine(), picproc_drawPixel()
 ****/
int
picproc_drawBox(Picture *pic, picture_coord column1, picture_coord row1,
                picture_coord column2, picture_coord row2, int color)
{
  if (0 == picproc_drawLine(pic, column1, row1, column1, row2, color) &&
      0 == picproc_drawLine(pic, column1, row1, column2, row1, color) &&
      0 == picproc_drawLine(pic, column1, row2, column2, row2, color) &&
      0 == picproc_drawLine(pic, column2, row1, column2, row2, color) )
    return 0;
  else
    return -1;
}

/****f* libpicproc/picproc_drawArea
 * SYNOPSIS
 *   #include "picproc.h"
 *   int picproc_drawArea(Picture *pic, picture_coord column1, picture_coord row1,
 *                       picture_coord column2, picture_coord row2, int color)
 * DESCRIPTION
 *   Fill area with corners at (column1,row1) and (column2,row2) using
 *   the given color.  The given color must match the image format.
 *   Calls picproc_drawPixel() to draw the edges.
 * RESULT
 *   0 on success and -1 if coordinates are out of range
 * SEE ALSO
 *   picproc_drawPixel()
 ****/
int
picproc_drawArea(Picture *pic, picture_coord column1, picture_coord row1,
                picture_coord column2, picture_coord row2, int color)
{
  int row, column;
  for (row = row1; row <= row2; row++)
    for (column = column1; column <= column2; column++)
      picproc_drawPixel(pic, column, row, color);

  return 0;
}

/****f* libpicproc/picproc_drawArrow
 * SYNOPSIS
 *   #include "picproc.h"
 *   int picproc_drawArrow(Picture *pic, picture_coord column,
 *                         picture_coord row, double angle, int color)
 * DESCRIPTION
 *   Draw box with corners at (column1,row1) and (column2,row2) using
 *   the given color.  The given color must match the image format.
 *   Calls picproc_drawLine() to draw the edges.  The angle is
 *   specified in radians and 0 radians is along the x axis, and
 *   positive rotation is counter-clockwise.
 * RESULT
 *   0 on success and -1 if coordinates are out of range
 * SEE ALSO
 *   picproc_drawLine(), picproc_drawPixel()
 ****/
int
picproc_drawArrow(Picture *pic, picture_coord column, picture_coord row,
                  int radius, int base, double angle, int color)
{
  int i;
  int p[3][2];
  int pn[3][2];
  double mat[2];

  /* The points making up the arrow */
  p[0][0] = 0;
  p[0][1] = -(base+1)/2;
  p[1][0] = 0;
  p[1][1] = (base+1)/2;
  p[2][0] = radius;
  p[2][1] = 0;

  mat[0] = cos(angle);
  mat[1] = sin(angle);

  /* Rotate array points */
  for(i=0; i<3; i++)
    {
      pn[i][0] =  (mat[0]*p[i][0] - mat[1]*p[i][1]) + column;
      pn[i][1] = -(mat[1]*p[i][0] + mat[0]*p[i][1]) + row;
    }

  picproc_drawLine(pic, pn[0][0], pn[0][1], pn[1][0], pn[1][1], color);
  picproc_drawLine(pic, pn[1][0], pn[1][1], pn[2][0], pn[2][1], color);
  picproc_drawLine(pic, pn[0][0], pn[0][1], pn[2][0], pn[2][1], color);

  return 0;
}


/****f* libpicproc/picproc_pnmEncode
 * SYNOPSIS
 *   #include "picproc.h"
 *   typedef long (*picproc_writer)(int ref, const void *buf, long count);
 *   int picproc_pnmEncode(Picture *pic, picproc_writer writer, int ref)
 * DESCRIPTION
 *   Encode given picture as PPM or PGM and send the result to
 *   writer().  The writer prototype is supposed to be compatible with
 *   UNIX write(), to make it easy to write to file.
 * EXAMPLE
 *   Picture *pic = get_picture();
 *   FILE *fp = fopen("picture.ppm", "w");
 *   picproc_pnmEncode(pic, (picproc_writer)write, fileno(fp));
 *   fclose(fp);
 * RESULT
 *   0 on success and -1 on failure.
 * NOTE
 *   Only work on pixel formats bitmap, grey4, grey, rgb24 and rgb32
 *   at the moment.
 ****/
int
picproc_pnmEncode(Picture *pic, picproc_writer writer, int ref)
{
  char *sig;
  int levels;

  if (NULL == pic)
    return -1;

  switch (pic->format)
    {
    case pix_rgb24:
    case pix_rgb32:
    case pix_bgr24:
    case pix_bgr32:
      sig = "P3";
      levels = 255;
      break;
    case pix_grey4:
      sig = "P2";
      levels = 15;
      break;
    case pix_grey:
      sig = "P2";
      levels = 255;
      break;
    case pix_bitmap:
      sig = "P2";
      levels = 1;
      break;
    default:
      return -1;
      break;
    }
  {
    int pos = 0;
    char temp[100];
    long length;
    int row, column;
    length = sprintf(temp, "%s\n%d %d\n%d\n",
                     sig, pic->width, pic->height, levels);
    writer(ref, temp, length);

    for(row=0; row < pic->height; row++)
      for(column=0; column < pic->width; column++)
        {
          switch (pic->format)
            {
            case pix_rgb24:
            case pix_rgb32:
              length = sprintf(temp,"%u %u %u \n",
                               (unsigned int)pic->data[pos],
                               (unsigned int)pic->data[pos+1],
                               (unsigned int)pic->data[pos+2]);
              break;
            case pix_bgr24:
            case pix_bgr32:
              length = sprintf(temp,"%u %u %u \n",
                               (unsigned int)pic->data[pos+2],
                               (unsigned int)pic->data[pos+1],
                               (unsigned int)pic->data[pos]);
            case pix_grey4:
            case pix_grey:
              length = sprintf(temp,"%u\n",(unsigned int)pic->data[pos]);
              break;
            case pix_bitmap:
              length = sprintf(temp,"%u\n",
                               picproc_bitmap_getPixel(pic, column, row));
              break;
            default:
              return -1;
              break;
            }
          writer(ref, temp, length);
          pos += pic->bytes_per_pixel; /* += 0 for bitmaps ! */
        }
  }
  return 0;
}
