/***********************************************************************************************************/
/*                 V I S I O N  -  L I B R A R Y                                                           */
/*                                                                                                         */
/* Philippe LECLERCQ                                                                                       */
/* -----------------                                                                                       */
/*                  August 24th 2000                                                                       */
/*                                                                                                         */
/*                                                                                   lecle-p@ee.uwa.edu.au */
/***********************************************************************************************************/

#include "vision.h"

/***********************************************************************************************************/
/* GENERAL FUNCTIONS */
/*********************/
                      /* denoise and region process functrion */
static void VIS_Denoise_and_process(VIS_structs *structs, Picture *pic_info, int class, int number);
                      /* Denoise every pixels */
static int  VIS_Denoise(int len, VIS_color_process *object, Picture *pic_info, int uplimit);
                      /* Denoise every pixels f i.e. no for, one pixel around */
static int  VIS_Denoisef(VIS_color_process *object, Picture *pic_info, int uplimit);
                      /* Denoise 2 i.e. every 2 pixels */
static int  VIS_Denoise2(int len, VIS_color_process *object, Picture *pic_info, int uplimit);
                      /* Denoise 2 i.e. every two pixels f i.e. fast (no for, one pixel around) */
static int  VIS_Denoise2f(VIS_color_process *object, Picture *pic_info, int uplimit);
                      /* Find the limits of segmented object */
static void VIS_Find_limits(int* piTop, int* piBot, int* piRight, int* piLeft, VIS_color_process *object);

/***********************************************************************************************************/
/* SOCCER FUNCTIONS */
/********************/
                     /* Load default values */
static void VIS_Load_defaults(VIS_structs *structs);
                     /* Show status of levels */
static void VIS_Show_levels(int r, int g, int b, int l);
                     /* Show status of offset */
static void VIS_Show_offset(int alpha_offset, float beta_offset);
                     /* Calibration of levels */
static void VIS_Cal_color(VIS_structs *structs, VIS_cameraCapture_t camera, void *cameradata);
                     /* Camera angles offset */
static void VIS_Cal_offset(VIS_structs *structs, VIS_cameraCapture_t camera, void *cameradata);
                     /* Adjust single level */
static int  VIS_Cal_single_level(VIS_structs *structs, int RGB, VIS_cameraCapture_t camera,
				 void *cameradata);
                     /* Adjust second level */
static int  VIS_Cal_second_level (VIS_structs *structs, int R_level, VIS_cameraCapture_t camera,
                                  void *cameradata);
                     /* Single level processing */
static void VIS_Cal_single_process(VIS_structs *structs, Picture *pic_info, int mn, int RGB,
				    VIS_color_region *goal);
                     /* Second level processing */
static int  VIS_Cal_second_process(VIS_structs *structs, Picture *pic_info, int R_level, int mn);
                     /* Calibration find pixels with one condition*/
static int  VIS_Cal_find_pixels(int mn, int level, int L);
                     /* Calibration find pixels 2 i.e. with two conditions */
static int  VIS_Cal_find_pixels2(int mn1, int mn2, int level1, int level2, int L);
                     /* Find blue */
static int  VIS_Find_blue(int mn, int B, int L);
                     /* Find red i.e. yellow goal and orange ball */
static int  VIS_Find_red(int r_mn, int g_mn, int R, int G, int L);

/* convert radian to degrees */
static double rad2deg(radians rad) { return 180*rad/M_PI;}

/***********************************************************************************************************/
/* GENERAL FUNCTIONS */
/*********************/

/****f* libvision/VIS_Init_cam
 * SYNOPSIS
 *   int VIS_Init_cam(int mode)
 * DESCRIPTION
 *   Initialisation of the camera. 10 retries.
 *   Input  : mode of the camera.
 *   Output : camversion or INITERROR.
 * SEE ALSO
 *   N/A
 ****/
int VIS_Init_cam(int cammode)
{
  int  camversion=0;
  int  number=0;
  char rotation[]={'|', '/', '-', '\\'};

  LCDClear();
  LCDSetPrintf(0,0, "  Camera Init.");
  LCDSetPrintf(1,0, "  ------------");
  LCDSetPrintf(3,6, "[ ]");

  /* Wait for camera initialisation, get camera type */
  while ((INITERROR==(camversion=CAMInit(NORMAL))) && (number<=10) && (NOCAM!=camversion)){
    OSWait(CAMWAIT);
    LCDSetChar(3,7, rotation[(number++)%4]);
  }

  /* Bad initialisation after 10 times */
  if (INITERROR==camversion) { VIS_Error_display("! Camera error !", -1,"   Cam. init."); return camversion;}

  /* No camera connected */
  if (NOCAM==camversion) { VIS_Error_display("! Camera error !", -1,"   No Camera"); return camversion;}

  /* Camera initialised, return version */
  CAMMode(cammode);
  AUBeep();
  return camversion;
}

/****f* libvision/VIS_Init_bwimg
 * SYNOPSIS
 *   void VIS_Init_bwimg(Picture *LCD_info)
 * DESCRIPTION
 *   BW LCD picture initialisation.
 *   Input  : 'Picture' structure.
 *   Output : 0 no errors, -1 LCD_info not set up or declared size <0.
 * SEE ALSO
 *   VIS_Compute_bw(), VIS_Draw_limits(), VIS_Draw_border()
 ****/
int VIS_Init_bwimg(Picture *LCD_info)
{
  /* Init picture */
  if (NULL!=LCD_info && 0<LCD_info->datasize) {
    memset(LCD_info->data, 0, LCD_info->datasize);
    return 0;
  }
  else { VIS_Error_display("!  Init bwimg  !", -1, "     failed"); return -1;}
}

/****f* libvision/VIS_Autobrightness_delay
 * SYNOPSIS
 *   void VIS_Autobrightness_delay(VIS_cameraCapture_t camera, void *cameradata, int number)
 * DESCRIPTION
 *   Let time to camera for autobrightness.
 *   Input  : camera info (VIS_cameraCapture_t and cameradata), number of waits.
 *   Output : N/A
 * SEE ALSO
 *   N/A
 ****/
void VIS_Autobrightness_delay(VIS_cameraCapture_t camera, void *cameradata, int number)
{
  char Key = 0;

  /* Initialise show progress display */
  VIS_Init_show_progress("*Autobrightness*\n");
  LCDMenu("No", " ", " ", "Yes");

  /* Wait for Yes or No */
  KEYGetBuf(&Key);
  while (Key!=KEY1 && Key!=KEY4) {
    KEYGetBuf(&Key);
  }
  AUBeep();

  /* Wait loop... */
  if (Key==KEY4) {
    int i;
    LCDMenu(" ", " ", " ", " ");
    for (i=0; i<number; i++) {
      /* Wait : i.e. take a picture */
      (void)camera(cameradata);
      
      /* Still alive display */
      VIS_Show_progress(i, number);
    }
  }
}

/****f* libvision/VIS_Compute_bw
 * SYNOPSIS
 *   void VIS_Compute_bw(BYTE *bwimg, BYTE *temp, Picture *pic_info)
 * DESCRIPTION
 *   Get a 2D binary array and make a linear BW array for LCD display.
 *   Input  : a BW linear array for LCD display, a 2D binary array, a Picture struct.
 *   Output : N/A
 * SEE ALSO
 *   VIS_Init_bwimg(), VIS_Draw_limits(), VIS_Draw_border()
 ****/
void VIS_Compute_bw(VIS_structs *structs, BYTE *temp, Picture *pic_info)
{
  int row=0;
  int imgrows=pic_info->height, imgcols=pic_info->width;
  int loop=structs->Algo_info->loop;
  int pixels=pic_info->width*pic_info->height;
  int datasize=pixels>>(2*loop);

  for (row=imgrows-1; row>=0; row--) { /*for (row=0; row<imagerows; row++) {*/
    int col;
    for (col=0; col<imgcols; col+=8) {
      int value=0x00;
      int i;
      for (i=7; i>=0; i--) { /*for (i=0; i<8; i++) {*/
        int offset=((row*imgcols)+(col+7-i))>>(2*loop); /* [row][col+(7-i)] */
        if ((offset<datasize) && (temp[offset])) {
          value |= (1<<i);
        }
      }
      structs->LCD_info->data[(row<<4)+(col>>3)]=value; /* bwimg[row*128/8+col/8]=value; */
    }
  }
}

/****f* libvision/VIS_Draw_limits
 * SYNOPSIS
 *   void VIS_Draw_limits(BYTE *bwimg, VIS_color_region *region, int value)
 * DESCRIPTION
 *   Draw object limits and center on bw picture.
 *   value==0 -> erase, value>0 -> draw.
 *   Input  : a BW linear array, the region to draw and a draw (i.e. >0) / erase (i.e. ==0) value.
 *   Output : N/A
 * SEE ALSO
 *   VIS_Init_bwimg(), VIS_compute_bw(), VIS_Draw_border()
 ****/
void VIS_Draw_limits(BYTE *bwimg, VIS_color_region *region, int value)
{
  int top=region->coord.top, bot=region->coord.bot;
  int right=region->coord.right, left=region->coord.left;

  if (value) {
    bwimg[(top<<4)+(right>>3)] |= 1<<(7-(right) % 8); /* bwimg[top*128/8+right/8] |= 1<<(7-(right) % 8); */
    bwimg[(top<<4)+(left>>3)] |= 1<<(7-(left) % 8);   /* bwimg[top*128/8+left/8] |= 1<<(7-(left) % 8); */
    bwimg[(bot<<4)+(right>>3)] |= 1<<(7-(right) % 8); /* bwimg[bot*128/8+right/8] |= 1<<(7-(right) % 8); */
    bwimg[(bot<<4)+(left>>3)] |= 1<<(7-(left) % 8);   /* bwimg[bot*128/8+left/8] |= 1<<(7-(left) % 8); */
    bwimg[((top+bot)<<3)+((right+left)>>4)] |= 1<<(7-((right+left)>>1) % 8);
    /* bwimg[((top+bot)/2)*128/8+((right+left)/2)/8] |= 1<<(7-((right+left)/2) % 8);*/
  }
  else {
    bwimg[(top<<4)+(right>>3)] &= ~(1<<(7-(right) % 8)); /* bwimg[top*128/8+right/8] = 0x00; */
    bwimg[(top<<4)+(left>>3)] &= ~(1<<(7-(left) % 8));  /* bwimg[top*128/8+left/8] = 0x00; */
    bwimg[(bot<<4)+(right>>3)] &= ~(1<<(7-(right) % 8)); /* bwimg[bot*128/8+right/8] = 0x00; */
    bwimg[(bot<<4)+(left>>3)] &= ~(1<<(7-(left) % 8));  /* bwimg[bot*128/8+left/8] = 0x00; */
    bwimg[((top+bot)<<3)+((right+left)>>4)] &= ~(1<<(7-((right+left)>>1) % 8)); 
    /* bwimg[((top+bot)/2)*128/8+((right+left)/2)/8]=0x00; */
  }
}
/****f* libvision/VIS_Draw_border
 * SYNOPSIS
 *   void VIS_Draw_border(BYTE *bwimg, int rows, int cols)
 * DESCRIPTION
 *   Draw borders in a 1D array for LCD display
 *   Input  : a 1D BW array, line and column of the border.
 *   Output : N/A
 * SEE ALSO
 *   VIS_Init_bwimg(), VIS_compute_bw(), VIS_Draw_limits()
 ****/
void VIS_Draw_border(BYTE *bwimg, int rows, int cols)
{
  int i;

  for (i=cols; i>=0; i--) {                       /* for (i=0; i<=cols; i++) {*/
    bwimg[(rows<<4)+(i>>3)] |= 1<<(7-(i) % 8);    /* bwimg[rows*128/8+i/8] |= 1<<(7-(i) % 8); */
  }
  for (i=rows-1; i>=0; i--) {                     /* for (i=0; i<rows; i++) {*/
    bwimg[(i<<4)+(cols>>3)] |= 1<<(7-(cols) % 8); /* bwimg[i*128/8+cols/8] |= 1<<(7-(cols) % 8); */
  }
}
/****f* libvision/VIS_Error_display
 * SYNOPSIS
 *   void VIS_Error_display(char title[16], int error, char msg[16])
 * DESCRIPTION
 *   Display error messages.
 *   Input  : a box title (16 char max.), an error code an error message (16 char max.).
 *   Output : N/A
 * SEE ALSO
 *   VIS_Init_show_progress(), VIS_Show_progress()
 ****/
void VIS_Error_display(char title[16], int error, char msg[16])
{
  AUTone(110, 500);
  LCDClear();
  LCDSetPrintf(0,0, title);
  LCDSetPrintf(1,0, "----------------");
  LCDSetPrintf(2,0, "  Error code :");
  LCDSetPrintf(3,0, "      %d", error);
  LCDSetPrintf(4,0, "   Message :");
  LCDSetPrintf(5,0, msg);
  LCDMenu("", "", "", "Cont");
  KEYWait(KEY4);
}

/****f* libvision/VIS_Init_show_progress
 * SYNOPSIS
 *   void VIS_Init_show_progress(char title[16])
 * DESCRIPTION
 *   Initialise the progress show of a loop (clear screen, display title,...).
 *   Input  : a loop title (16 char max.).
 *   Output : N/A
 * SEE ALSO
 *   VIS_Error_display(), VIS_Show_progress()
 ****/
void VIS_Init_show_progress(char title[16])
{
  LCDClear();
  LCDSetPrintf(0, 0, title);
  LCDSetPrintf(2, 0, "I          I\n");
  LCDSetPrintf(2,13, "[ ]\n");
}

/****f* libvision/VIS_Show_progress
 * SYNOPSIS
 *   void VIS_Show_progress(int actual, int max)
 * DESCRIPTION
 *   Displays the progress of a loop.
 *   Input  : the number of the actual loop and the total number of loops.
 *   Output : N/A
 * SEE ALSO
 *   VIS_Error_display(), VIS_Init_show_progress()
 ****/
void VIS_Show_progress(int actual, int max)
{
  int  col;
  char rotation[]={'|', '/', '-', '\\'};

  if (actual==0) {
    LCDSetPrintf(4,0, "  Please Wait");
  }
  col=(int) (actual/(max/10))+1;
  if (col>10) { col=10;}
  LCDSetChar(2,col, '.');
  LCDSetChar(2,14, rotation[actual % 4]);
}
                      /****************************************/
                      /* Denoise and region process function */
                      /****************************************/
static void VIS_Denoise_and_process(VIS_structs *structs, Picture *pic_info, int class, int number)
{
  VIS_algo          *Algo_info=structs->Algo_info;
  int               loop=Algo_info->loop;
  int               offset=(Algo_info->depth)*(1<<loop);
  VIS_color_region  *region=&structs->region[number];
  VIS_coord         *coord=&region->coord;
  VIS_color_process *object=&structs->objects[number];

  /* Denoise function selection */
  if      (Algo_info->fast && loop)  { object->index=VIS_Denoise2f(object, pic_info, structs->uplimit);}
  else if (Algo_info->fast && !loop) { object->index=VIS_Denoisef(object, pic_info, structs->uplimit);}
  else if (!Algo_info->fast && loop) { object->index=VIS_Denoise2(Algo_info->depth, object, pic_info,
								  structs->uplimit);}
  else                               { object->index=VIS_Denoise(Algo_info->depth, object, pic_info,
								 structs->uplimit);}

  /* Initialisation of region array : ball, ygoal, bgoal, top, bot, right, left */
  if (object->index>=0) {
    int top=pic_info->height+1,  bot=-1, right=-1, left=pic_info->width+1;
    VIS_Find_limits(&top, &bot, &right, &left, object);
    region->color=class;
    coord->top=top-offset; coord->bot=bot+offset; coord->right=right+offset; coord->left=left-offset;
  }
  else {
    region->color=VIS_NOTHING;
  }
}
                      /************************/
                      /* Denoise every pixels */
                      /************************/
static int VIS_Denoise(int len, VIS_color_process *object, Picture *pic_info, int uplimit)
{
  int  l_index=0;
  int  row;
  int  imgcols=pic_info->width;
  BYTE *temp=object->table, *X=object->X_list, *Y=object->Y_list;

  for (row=uplimit+len+1; row<=(pic_info->height-len-2); row++) { 
    int col;
    for (col=len+2; col<=(imgcols-len-2); col++) {
      int res=0, depth;
      for (depth=1; depth<=len; depth++) {
	int pos=row*imgcols+col;
        res+=(temp[pos-imgcols*depth] & temp[pos+depth] & temp[pos+imgcols*depth] & temp[pos-depth]);
      }
      if (res==len) { X[l_index]=row; Y[l_index]=col; l_index++;}
    }
  }
  return l_index-1;
}
                      /********************************************************/
                      /* Denoise every pixels f i.e. no for, one pixel around */
                      /********************************************************/
static int VIS_Denoisef(VIS_color_process *object, Picture *pic_info, int uplimit)
{
  int  l_index=0;
  int  row;
  int  imgcols=pic_info->width;
  BYTE *temp=object->table, *X=object->X_list, *Y=object->Y_list;

  for (row=uplimit+2; row<=pic_info->height-1-2; row++) {
    int col;
    for (col=3; col<=imgcols-1-2; col ++) {
      int pos=row*imgcols+col;
      if (temp[pos-imgcols] & temp[pos+1] & temp[pos+imgcols] & temp[pos-1]) 
	{ X[l_index]=row; Y[l_index]=col; l_index++;}
    }
  }
  return l_index-1;
}
                      /***********************************/
                      /* Denoise 2 i.e. every two pixels */
                      /***********************************/
static int VIS_Denoise2(int len, VIS_color_process *object, Picture *pic_info, int uplimit)
{
  int  l_index=0;
  int  row_index;
  int  imrows=(pic_info->height>>1);
  int  imcols=(pic_info->width>>1);
  BYTE *temp=object->table, *X=object->X_list, *Y=object->Y_list;

  for (row_index=((uplimit>>1)+len+1); row_index<=imrows-len-1; row_index++) {
    int row=(row_index<<1);
    int col_index;
      for (col_index=len+1; col_index<=imcols-len-1; col_index++) {
      int res=0, depth;
      for (depth=1<<1; depth<=(len<<1); depth+=2) {
	int pos=row_index*imcols+col_index;
        res+=(temp[pos-imcols*depth] & temp[pos+depth] & temp[pos+imcols*depth] & temp[pos-depth]);
      }
      if (res==len) {
	int col=(col_index<<1); 
	X[l_index]=row; Y[l_index]=col; l_index++;
      }
    }
  }
  return l_index-1;
}
                      /**************************************************************************/
                      /* Denoise 2 i.e. every two pixels f i.e. fast (no for, one pixel around) */
                      /**************************************************************************/
static int VIS_Denoise2f(VIS_color_process *object, Picture *pic_info, int uplimit)
{
  int  l_index=0;
  int  row_index;
  int  imrows=(pic_info->height>>1), imcols=(pic_info->width>>1);
  BYTE *temp=object->table, *X=object->X_list, *Y=object->Y_list;

  for (row_index=(uplimit>>1)+1; row_index<=imrows-2; row_index++) {
    int row=(row_index<<1);
    int col_index;
    for (col_index=3; col_index<=imcols-2; col_index++) {
      int pos=row_index*imcols+col_index;
      if (temp[pos-imcols] & temp[pos+1] & temp[pos+imcols] & temp[pos-1]) {
	int col=(col_index<<1);
	X[l_index]=row; Y[l_index]=col; l_index++;
      }
    }
  }
  return l_index-1;
}
                      /********************************************/
                      /* Find the limits of the segmented objects */
                      /********************************************/
static void VIS_Find_limits(int* piTop, int* piBot, int* piRight, int* piLeft, VIS_color_process *object)
{
  int  i;
  int  right=*piRight, left=*piLeft;
  int  l_index=object->index;
  BYTE *Y=object->Y_list, *X=object->X_list;

  /* Look for right and left values */
  for (i=l_index; i>=0; i--) { /*for (i=0; i<=l_index; i++) {*/
    int y_value=Y[i];
    if      (y_value<left) { left=y_value;}   /*if (Y[i]<*piLeft) { *piLeft=Y[i];}*/
    else if (y_value>right) { right=y_value;} /*if (Y[i]>*piRight) { *piRight=Y[i];}*/
  }
  /* Assign values */
  *piTop=X[0]; *piBot=X[l_index]; /* Values are inserted from TOP TO BOTTOM in denoising function... */
  *piLeft=left; *piRight=right;
}

/***********************************************************************************************************/
/* SOCCER FUNCTIONS */
/********************/
                     /**************************************/
                     /* Testing RGB Calibration for soccer */
                     /**************************************/
/****f* libvision/VIS_Soccer_test
 * SYNOPSIS
 *   int VIS_Soccer_test(void)
 * DESCRIPTION
 *   Exemple program for using the libvision.
 *   Calibrates, looks at the ball and goals, displays them and displays ball distance.
 *   Input  : N/A
 *   Output : 0 if no erros occured, !=0 if not.
 * SEE ALSO
 *   N/A
 ****/
int VIS_Soccer_test(void)
{
  /* EXTERNAL VARIABLES, initialised to 0 or NULL for testing... */
     VIS_cameraCapture_t camera=VIS_cameraCapture;
     void                *cameradata=NULL;
     Picture             *pic_info=NULL;
     BYTE                LCD_info_data[VIS_BWIMAGECOLUMNS*VIS_BWIMAGEROWS/8];
     Picture             LCD_pic;
     Picture             *LCD_info;
  /* SOCCER TEST PROGRAM VARIABLES */
     int                 drow=-1, dcol=-1;
     int                 timea=0, timeb=0, timec=0, timed=0;
     int                 error=0;
     char                line[6]=".....";
  /* LIBVISION LIBRARY VARIABLES */
     VIS_structs         *structs=NULL; /* Main structure pointer */
     VIS_algo            temp_algo;     /* Temporary Algo structure for modifications */

  /* Camera initialisation */
  error=VIS_Init_cam(AUTOBRIGHTNESS);
  if (INITERROR==error) { VIS_Error_display("!  Cam. error !", -1, "    init."); return -1;}

  /* EXTERNAL pic_info initialisation, camera must be initialised first ! */
  pic_info=camera(cameradata);
  if (NULL==pic_info) { VIS_Error_display("! Init. error !", -1, "  pic_info==0."); return -1;}

  /* EXTERNAL LCD display structure initialisation */
  picture_init(&LCD_pic, VIS_BWIMAGECOLUMNS, VIS_BWIMAGEROWS, pix_bitmap, 0,
	       &LCD_info_data[0], sizeof(LCD_info_data));
  LCD_info=&LCD_pic;
  if (LCD_info==NULL) { VIS_Error_display("! Init. error !", -1, " LCD_info init."); return -1;}

  /* Main structure initialisation and pointer */
  structs=VIS_Init_var(LCD_info, pic_info);
  if (structs==NULL) {VIS_Error_display("! structs err. !", 10, " structs init."); return 10;}

  /* Color and distance lookup tables calibration menu */
  error=VIS_Init_calibration(structs, camera, cameradata);
  if (error) { return error;}

  /* Color classes lookup table initialisation */
  error=VIS_Init_color_classes(structs);
  if (error) { return error;}
  
  /* Distance lookup table initialisation */
  error=VIS_Init_distance(structs, pic_info);
  if (error) { return error;}

  /* Modify defaults parameters before execution */
  temp_algo.depth=1; temp_algo.fast=1; temp_algo.step=2; /* temp_algo.loop ONLY modified internally */
  error=VIS_Modify_algo(structs, &temp_algo, pic_info);
  if (error) { return error;}

  /* BW picture initialisation */
  error=VIS_Init_bwimg(LCD_info);
  if (error) { return error;}

  /* Draw border of camera picture */
  VIS_Draw_border(LCD_info->data, pic_info->height+1, pic_info->width+1);

  /* Main loop */
  LCDClear(); LCDMenu(" ", " ", " ", "END");
  while (KEY4!=KEYRead()) {
    
    /* Declarations */
    VIS_distance     distance;      /* Temporary Distance structure to get results */
    VIS_color_region *result=NULL;  /* Result pointer to the region structure in the VIS_structs */

    timea=OSGetCount(); /* Beginning of function timer */

    /* Get camera picture */
    pic_info=camera(cameradata);
    if (pic_info==NULL) { VIS_Error_display("!  Get picture !", 20, "  returned NULL");}

    timeb=OSGetCount(); /* After getting picture timer */

    /* Process picture, find classes */
    result=VIS_Find_classes(structs, pic_info);
    if (result==NULL) { VIS_Error_display("! Find classes ", 30, "  returned NULL");}

    timec=OSGetCount(); /* After processing picture timer */

    /* Draw dots on LCD panel and initialise status string */
    if (result[0].color==VIS_BALL) {
      VIS_Draw_limits(LCD_info->data, &result[0], 1);
      line[0]='b';
    }
    else { line[0]='.'; drow=-1; dcol=-1;}

    if (result[1].color==VIS_YGOAL) {
      VIS_Draw_limits(LCD_info->data, &result[1], 1);
      line[1]='y'; line[2]='g';
    }
    else { line[1]='.'; line[2]='.';}

    if (result[2].color==VIS_BGOAL) {
      VIS_Draw_limits(LCD_info->data, &result[2], 1);
      line[3]='b'; line[4]='g'; line[5]='\0';
    }
    else { line[3]='.'; line[4]='.'; line[5]='\0';}

    /* Middle of camera pixel */
    LCD_info->data[30*128/8+40/8] |= 1<<(7-(40) % 8);

    /* Display */
    LCDPutImage(LCD_info->data);

    /* Erase dots on LCD panel, get ball distance */
    if (result[0].color==VIS_BALL) {
      VIS_Draw_limits(LCD_info->data, &result[0], 0);
      VIS_Get_position(structs, &distance, VIS_BALL);
      drow=distance.d_row; dcol=distance.d_col;
    }
    if (result[1].color==VIS_YGOAL) {VIS_Draw_limits(LCD_info->data, &result[1], 0);}
    if (result[2].color==VIS_BGOAL) {VIS_Draw_limits(LCD_info->data, &result[2], 0);}

    timed=OSGetCount(); /* Before displaying timer */

    /* Display informations */
    LCDSetPrintf(0,10,"%s", line);
    LCDSetPrintf(1,12,"%d ", structs->uplimit);
    LCDSetPrintf(2,11,"%d ", drow);
    LCDSetPrintf(3,11,"%d ", dcol);
    LCDSetPrintf(4,10,"Frame");
    LCDSetPrintf(5,11,"%1.1f ", 1/(0.01*(timed-timea)));
    LCDSetPrintf(6,11,"t:%d ", timec-timeb);
  }

  /* Release LIBVISION variables */
  VIS_Release_var(structs);

  /* Release EXTERNAL variables */
#if defined(USE_MALLOC)
  free(LCD_info->data);
  free(LCD_info);
#endif

  return error;
}

/****f* libvision/VIS_Init_var
 * SYNOPSIS
 *   VIS_structs *VIS_Init_var(Picture *LCD_info, Picture *pic_info)
 * DESCRIPTION
 *   Allocate memory for the different structures used by libvision.
 *   Allocate dynamic or static memory depending if USE_MALLOC as been defined.
 *   Input  : a Picture for LCD and a Picture for camera structures.
 *   Output : NULL or pointer to the initialised structure.
 * SEE ALSO
 *   VIS_Release_var(), VIS_Check_init()
 ****/
VIS_structs *VIS_Init_var(Picture *LCD_info, Picture *pic_info)
{
#if defined(USE_MALLOC) /* Use malloc dynamic allocation */

  int l_index, size=0;
  int error=0;

  /* Initialise structure */
  VIS_structs *structs=calloc(1, sizeof(*structs));
  if (NULL==structs) {
    VIS_Error_display("! struct. error !", 200, " structs NULL"); goto error1;
  }

  /*-----------------*/
  /* Allocate memory */
  /*-----------------*/

  /* LCD_info test and initialise internal pointer */
  structs->LCD_info=LCD_info;
  if (structs->LCD_info==NULL) {
    VIS_Error_display("! Init. error !", 200, " LCD_info NULL"); goto error2;
    }

  /* pic_info test only */
  if (pic_info==NULL) {
    VIS_Error_display("! Init. error !", 200, " pic_info NULL"); goto error2;
    }
  /* Algo_info initialise and test */
  structs->Algo_info=(VIS_algo *) malloc(sizeof(VIS_algo));
  if (structs->Algo_info==NULL) {
    VIS_Error_display("! Malloc error !", 200, " Algo_info init."); goto error2;
  }
  /* RGB_space initialise and test */
  structs->RGB_space=(VIS_color_def *) malloc(sizeof(VIS_color_def));
  if (structs->RGB_space==NULL) {
    VIS_Error_display("! Malloc error !", 200, "RGB_space init."); goto error3;
  }
  /* region initialise and test */
  structs->region=(VIS_color_region *) malloc(3*sizeof(VIS_color_region));
  if (structs->region==NULL) {
    VIS_Error_display("! Malloc error !", 200, "  region init."); goto error4;
  }
  /* CAM_offset initialise and test */
  structs->CAM_offset=(VIS_camera_offset *) malloc(sizeof(VIS_camera_offset));
  if (structs->CAM_offset==NULL) {
    VIS_Error_display("! Malloc error !", 200, "CAM_offset init"); goto error5;
  }
  /* Dist_tables initialise and test */
  structs->Dist_tables=(VIS_distance_tables *) malloc(sizeof(VIS_distance_tables));
  if (structs->Dist_tables==NULL) {
    VIS_Error_display("! Malloc error !", 200, "Dist_tables in."); goto error6;
  }
  /* objects initialise and test */
  structs->objects=(VIS_color_process *) malloc(3*sizeof(VIS_color_process));
  if (NULL==structs->objects) {
    VIS_Error_display("! Malloc error !", 200, " objects init."); goto error7;
  }

  /* Load default values */
  VIS_Load_defaults(structs);

  /* VIS_color_process objects initialisation and test */
  error=0;
  size=(pic_info->height>>structs->Algo_info->loop)*(pic_info->width>>structs->Algo_info->loop);
  for (l_index=0; l_index<3; l_index++) {
    structs->objects[l_index].size=size;
    structs->objects[l_index].table=(BYTE *) malloc(size);
    if (structs->objects[l_index].table==NULL) { error=210+l_index;}

    structs->objects[l_index].X_list=(BYTE *) malloc(size);
    if (structs->objects[l_index].X_list==NULL) { error=220+l_index;}

    structs->objects[l_index].Y_list=(BYTE *) malloc(size);
    if (structs->objects[l_index].Y_list==NULL) { error=230+l_index;}
  }
  if (error) { VIS_Error_display("! Malloc error !", error, " objects init."); goto error8;}

  /* Flags if the lookup tables have been processed or not */
  structs->color_done=FALSE;
  structs->dist_done=FALSE;

  /* Init done successfully */
  structs->init_success=TRUE;

  return structs;

  /* Something went wrong, release the already allocated memory */
 error8:
  {
    for (l_index=0; l_index<3; l_index++)
      {
        if (NULL != structs->objects[l_index].table)
          free(structs->objects[l_index].table);
        if (NULL != structs->objects[l_index].X_list)
          free(structs->objects[l_index].X_list);
        if (NULL != structs->objects[l_index].Y_list)
          free(structs->objects[l_index].Y_list);
      }
  }
  free(structs->objects);
 error7:
  free(structs->Dist_tables);
 error6:
  free(structs->CAM_offset);
 error5:
  free(structs->region);
 error4:
  free(structs->RGB_space);
 error3:
  free(structs->Algo_info);
 error2:
  free(structs);
 error1:
  return NULL;

#else /* Use static variables */

  /* Declare structures */
#define size imagerows*imagecolumns
  static BYTE gtable0[size], gtable1[size], gtable2[size];
  static BYTE gX_list0[size], gX_list1[size], gX_list2[size];
  static BYTE gY_list0[size], gY_list1[size], gY_list2[size];
#undef size
  static VIS_algo            gAlgo_info;
  static VIS_structs         gstructs;
  static VIS_color_def       gRGB_space;
  static VIS_color_region    gregion[3];
  static VIS_camera_offset   gCAM_offset;
  static VIS_color_process   gobjects[3];
  static VIS_distance_tables gDist_tables;
  
  /* Check pic_info */
  if ((pic_info->height!=imagerows) || (pic_info->width!=imagecolumns)) {
    VIS_Error_display("!   pic_info   !", -1, "have bad infos");
    return NULL;
  }

  /* Flags if the lookup tables have been processed or not */
  gstructs.color_done=FALSE;
  gstructs.dist_done=FALSE;

  /* Upper goal limit */
  gstructs.uplimit=0;
  
  /* Attributions of main structures */
  gstructs.LCD_info=LCD_info;
  gstructs.Algo_info=&gAlgo_info;
  gstructs.RGB_space=&gRGB_space;
  gstructs.region=&gregion[0];
  gstructs.CAM_offset=&gCAM_offset;
  gstructs.Dist_tables=&gDist_tables;
  gstructs.objects=&gobjects[0];
  gstructs.objects[0].table=&gtable0[0];
  gstructs.objects[1].table=&gtable1[0];
  gstructs.objects[2].table=&gtable2[0];
  gstructs.objects[0].X_list=&gX_list0[0];  
  gstructs.objects[1].X_list=&gX_list1[0];  
  gstructs.objects[2].X_list=&gX_list2[0];
  gstructs.objects[0].Y_list=&gY_list0[0];  
  gstructs.objects[1].Y_list=&gY_list1[0];  
  gstructs.objects[2].Y_list=&gY_list2[0];  

  /* Load default values */
  VIS_Load_defaults(&gstructs);

  /* Init done successfully */
  gstructs.init_success=TRUE;

  return &gstructs;

#endif
}
/****f* libvision/VIS_Release_var
 * SYNOPSIS
 *   void VIS_Release_var(VIS_structs *structs)
 * DESCRIPTION
 *   Free memory used by malloc or reset some internal values if malloc not used
 *   (i.e. USE_MALLOC defined or not). 
 *   Input  : a libvision struct (VIS_structs).
 *   Output : N/A
 * SEE ALSO
 *   VIS_Init_var(), VIS_Check_init()
 ****/
void VIS_Release_var(VIS_structs *structs)
{
#if defined(USE_MALLOC) /* Use malloc dynamic allocation */
  int l_index;
  for (l_index=0; l_index<3; l_index++) {
    free(structs->objects[l_index].table);
    free(structs->objects[l_index].X_list);
    free(structs->objects[l_index].Y_list);
  }
  free(structs->objects);
  free(structs->Dist_tables);
  free(structs->CAM_offset);
  free(structs->region);
  free(structs->RGB_space);
  free(structs->Algo_info);
  free(structs);
  
#else

  /* Set flags to FALSE back */
  structs->color_done=FALSE;
  structs->dist_done=FALSE;
  structs->init_success=FALSE;

#endif
}
/****f* libvision/VIS_Check_init
 * SYNOPSIS
 *   int VIS_Check_init(VIS_structs *structs)
 * DESCRIPTION
 *   Checks if the initialisation of the VIS_structs has been done properly.
 *   Input  : a libvision struct (VIS_structs).
 *   Output : an error code >0 if an error occured.
 * SEE ALSO
 *   VIS_Init_var(), VIS_Release_var()
 ****/
int VIS_Check_init(VIS_structs *structs)
{
  int l_index;
  int error=0;

  /* LCD_info test */
  if (structs->LCD_info==NULL) { error+=1;}

  /* Algo_info initialise and test */
  if (structs->Algo_info==NULL) { error+=2;}

  /* RGB_space initialise and test */
  if (structs->RGB_space==NULL) { error+=4;}

  /* region initialise and test */
  if (structs->region==NULL) { error+=8;}

  /* CAM_offset initialise and test */
  if (structs->CAM_offset==NULL) { error+=16;}

  /* Dist_tables initialise and test */
  if (structs->Dist_tables==NULL) { error+=32;}

  /* objects initialise and test */
  if (structs->objects==NULL) { error+=64;}

  /* VIS_color_process objects initialisation and test */
  for (l_index=0; l_index<3; l_index++) {
    if (structs->objects[l_index].table==NULL) { error+=128<<l_index;}

    if (structs->objects[l_index].X_list==NULL) { error+=1024<<l_index;}

    if (structs->objects[l_index].Y_list==NULL) { error+=8192<<l_index;}
  }

  /* If error, display error message */
  if (error) { VIS_Error_display("! Init Var err !", error, "Wrong var init");}

  return error;
}
/****f* libvision/VIS_Init_calibration
 * SYNOPSIS
 *   void VIS_Init_calibration(VIS_structs *structs, VIS_cameraCapture_t camera, void *cameradata)
 * DESCRIPTION
 *   Calibration menu : color levels calibration or angle offsets calibration.
 *   Also checks the VIS_structs.
 *   Input  : a libvision struct (VIS_structs) and camera info (VIS_cameraCapture_t and cameradata).
 *   Output : returns 0 if VIS_structs is set up properly, >0 if not or if the color lookup table went wrong.
 * SEE ALSO
 *   N/A
 ****/
int VIS_Init_calibration(VIS_structs *structs, VIS_cameraCapture_t camera, void *cameradata)
{
  int  error=0;
  int  robot=0, team=0, robot_id=0;
  char Key=0;

  /* Check VIS_structs initialisation */
  error=VIS_Check_init(structs);

  /* If structure ok then go... */
  if (!error) {
    /* Menu */
    while (Key!=KEY4) {
      LCDClear();
      LCDSetPrintf(0,0, "  Calibrations");
      LCDSetPrintf(1,0, "  ------------");
      LCDSetPrintf(3,0, "   Color cal.");
      LCDSetPrintf(4,0, "   Angle cal.");
      LCDMenu("Col","Ang"," ","Ret");

      Key=KEYWait(ANYKEY);

      if (Key==KEY1) {
	/* Get robot team and number */
	robot=VIS_Team_robot();
	team=(int) (robot/10);
	robot_id=robot-10*team;
	
	/* (Re)initialise uplimit to default value */
	structs->uplimit=VIS_GOAL_UP[team][robot_id];

	/* Wait for autobrightness */
        VIS_Autobrightness_delay(camera, cameradata, 60);

        /* Color calibration */
        VIS_Cal_color(structs, camera, cameradata);
      }

      if (Key==KEY2) {
	/* Color classes lookup table initialisation */
	error=VIS_Init_color_classes(structs);

	/* Wait for autobrightness */
        VIS_Autobrightness_delay(camera, cameradata, 60);

	if (structs->color_done) {
	  /* Distance calibration */
	  VIS_Cal_offset(structs, camera, cameradata);
	} else
	  VIS_Error_display("! init error !", error, "no color class");
      }
    }
  }
  return error;
}
/****f* libvision/VIS_Init_color_classes
 * SYNOPSIS
 *   void VIS_Init_color_classes(VIS_structs *structs)
 * DESCRIPTION
 *   Checks the VIS_structs and initialise the color look-up table.
 *   Input  : a libvision structure (VIS_structs).
 *   Output : returns 0 if VIS_struct is set up properly, >0 if not or if color levels not set up.
 * SEE ALSO
 *   VIS_Init_distance()
 ****/
int VIS_Init_color_classes(VIS_structs *structs)
{
  int R=0, G=0, B=0;
  int error=0, temp=0;

  /* Check if lookup table needs to be updated */
  if (structs->color_done)
    return 0;
  
  /* Check VIS_struct */
  error=VIS_Check_init(structs);

  /* Check that color levels are all set up */
#if (VIS_SIM==0)
  if (-1 == structs->RGB_space->RGB_levels.R_level ||
      -1 == structs->RGB_space->RGB_levels.G_level ||
      -1 == structs->RGB_space->RGB_levels.B_level)
    error += (1 << 16);
#endif

  /* If structure ok then go... */
  if (!error) {
    
    /* Initialise show progress display */
    VIS_Init_show_progress("* Color class. *\n");

    /* Init color classes array */
    for (R=0; R<VIS_VALUES; R++) {
      for (G=0; G<VIS_VALUES; G++) {
        for (B=0; B<VIS_VALUES; B++) {

	  /* Luminosity */
	  int L;
          L=VIS_COEF*(R+G+B);

          /* Supposed class for the R,G,B triplet */
          temp=VIS_Find_blue(structs->RGB_space->RGB_levels.B_level, VIS_COEF*B, L);
          temp+=VIS_Find_red(structs->RGB_space->RGB_levels.R_level, structs->RGB_space->RGB_levels.G_level,
                             VIS_COEF*R, VIS_COEF*G, L);

          /* In conflict case (mostly for low R,G or B values) put 0 */
          if (temp==3 || temp>4) {
            structs->RGB_space->RGB_class[R][G][B]=0;
          }
          else {
            structs->RGB_space->RGB_class[R][G][B]=temp;
          }
        }
      }
      /* Still alive display... */
      VIS_Show_progress(R, VIS_VALUES);
    }
    structs->color_done=TRUE; /* Color classes lookup table initialised */
  }
  else {
    VIS_Error_display("! color  class !", error, "  init. failed");
  }
  return error;
}
/****f* libvision/VIS_Init_distance
 * SYNOPSIS
 *   void VIS_Init_distance(VIS_structs *structs, Picture *pic_info)
 * DESCRIPTION
 *   Check the VIS_structs and initialise distances lookup tables.
 *   Input  : a libvision structure (VIS_structs) and a 'Picture' structure.
 *   Output : returns 0 if VIS_struct is set up properly, >0 if not or if beta angle not set up.
 * SEE ALSO
 *   VIS_Init_color_classes(), VIS_Get_position()
 ****/
int VIS_Init_distance(VIS_structs *structs, Picture *pic_info)
{
  int                 error=0;
  int                 row=0, col=0;
  int                 d_norm=0, d_bord=0;
  int                 imgrows=pic_info->height, imgcols=pic_info->width;
  float               alpha=0, beta=0;
  float               temp, cos_alpha, sin_alpha;
  
  VIS_distance_tables *Dist_tables=structs->Dist_tables;
  
  /* Check if lookup table needs to be updated */
  if (structs->dist_done)
    return 0;
  
  /* Check VIS_structs */
  error=VIS_Check_init(structs);
  
  /* Check that color levels are all set up */
#if (VIS_SIM==0)
  if (-1==structs->CAM_offset->beta)
    error+=(1<<16);
#endif
  
  /* If structure ok then go... */
  if (!error) {
    
    /* Initialise show progress display */
    VIS_Init_show_progress("* Distance cal.\n");
    
    for (row=0; row<imgrows; row++) {
      beta=(0.75*(0.5*structs->CAM_offset->angle))*(row-(0.5*imgrows))/(0.5*imgrows);
      beta+=structs->CAM_offset->beta;

      temp=(float) (structs->CAM_offset->height-VIS_BALL_RADIUS);

      d_norm=(short) (temp/beta);
      d_bord=(short) ((temp-(0.5*VIS_BALL_RADIUS))/beta);

      for (col=0; col<imgcols; col++) {
	
        alpha=0.5*structs->CAM_offset->angle*(( (float) (2*col)/(imgcols))-1);
	cos_alpha=cos(alpha);
	sin_alpha=sin(alpha);
	
        Dist_tables->Dist_angle[VIS_ALPHA][row][col]=alpha;
        Dist_tables->Dist_angle[VIS_BETA][row][col]=beta;
	
        Dist_tables->Dist_dist[VIS_NORMAL][row][col]=d_norm;
        Dist_tables->Dist_dist[VIS_BORDER][row][col]=d_bord;
	
        Dist_tables->Dist_coord[VIS_NORMAL][VIS_ROW][row][col]=(short)d_norm*cos_alpha;
        Dist_tables->Dist_coord[VIS_NORMAL][VIS_COL][row][col]=(short)d_norm*sin_alpha;
        Dist_tables->Dist_coord[VIS_BORDER][VIS_ROW][row][col]=(short)d_bord*cos_alpha-VIS_BALL_RADIUS;
        Dist_tables->Dist_coord[VIS_BORDER][VIS_COL][row][col]=(short)d_bord*sin_alpha;
      }

      /* Still alive display... */
      VIS_Show_progress(row,imgrows);
    }
    structs->dist_done=TRUE; /* Distance lookup table initialsed */
  }
  else {
    if (error==1<<16) { VIS_Error_display("!init. distance!", error, " beta not init.");}
    else { VIS_Error_display("!init. distance!", error, "  init. failed");}
  }
  return error;
}
/****f* libvision/VIS_Get_position
 * SYNOPSIS
 *   int VIS_Get_distance(VIS_structs *structs, VIS_distance *distance)
 * DESCRIPTION
 *   Returns the distance of the ball by looking into the lookup tables.
 *   A VIS_distance struct is made of 3 integers : d_row, d_col and dist.
 *   Where dist is the direct distance between the robot and the ball,
 *         (d_row,d_col) are the relative coordinates of the ball compared to the robot position,
 *   So that dist^2=d_row^2+d_col^2.
 *   d_row is the depth of the ball compared to the robot position,
 *   d_col is the side distance of the ball compared to the robot.
 *   Input  : a libvision structure (VIS_structs) and a VIS_distance result structure.
 *            a class number, the 3 classes are defined as VIS_BALL, VIS_YGOAL and VIS_BGOAL.
 *   Output : 0 on success, 1 if distance table is not initialised.
 * SEE ALSO
 *   VIS_Init_distance()
 ****/
int VIS_Get_position(VIS_structs *structs, VIS_distance *distance, int class) {
  int       border=0;
  int       pcol=0, prow=0;
  VIS_coord coord;

  /* Select object to process */
  switch(class) 
    {
    case VIS_BALL :
      coord=structs->region[0].coord;
      break;
    case VIS_YGOAL :
      coord=structs->region[1].coord;
      break;
    case VIS_BGOAL :
      coord=structs->region[2].coord;
      break;
    default :
      VIS_Error_display("! Get position !", 1000, "  bad argument");
      break;
    }

  /* Check if distance lookup table is updated */
  if (!structs->dist_done)
    return 1;

  /* Look for ball position */
  if (coord.bot>=imagerows-5) {
    if (coord.left<=4) {
      prow=coord.top; pcol=coord.right;
      border=VIS_BORDER;
    }
    else if (coord.right>=imagecolumns-5) {
      prow=coord.top; pcol=coord.left;
      border=VIS_BORDER;
    }
    else {
      prow=coord.top; pcol=(coord.right+coord.left)>>1;
      border=VIS_BORDER;
    }
  }
  else if (coord.left<=5) {
    prow=(coord.top+coord.bot)>>1; pcol=coord.right;
    border=VIS_NORMAL;
  }
  else if (coord.right>=imagecolumns-5) {
    prow=(coord.top+coord.bot)>>1; pcol=coord.left;
    border=VIS_NORMAL;
  }
  else {
    prow=(coord.top+coord.bot)>>1;
    pcol=(coord.right+coord.left)>>1;
    border=VIS_NORMAL;
  }

  /* Get values in the tables */
  distance->d_row=structs->Dist_tables->Dist_coord[border][VIS_ROW][(short) prow][(short) pcol];
  distance->d_col=structs->Dist_tables->Dist_coord[border][VIS_COL][(short) prow][(short) pcol];
  distance->dist=structs->Dist_tables->Dist_dist[border][(short) prow][(short) pcol];
  distance->alpha=structs->Dist_tables->Dist_angle[VIS_ALPHA][(short) prow][(short) pcol];
  distance->beta=structs->Dist_tables->Dist_angle[VIS_BETA][(short) prow][(short) pcol];

  return 0;
}
/****f* libvision/VIS_Find_classes
 * SYNOPSIS
 *   void VIS_Find_classes(VIS_structs *structs, Picture *pic_info)
 * DESCRIPTION
 *   Looks for the ball and the two goals, get their coordinates.
 *   Input  : a libvision structure (VIS_structs) and a 'Picture' structure.
 *   Output : returns a pointer to the VIS_color_region part of the VIS_structs structure.
 * SEE ALSO
 *   N/A
 ****/
VIS_color_region* VIS_Find_classes(VIS_structs *structs, Picture *pic_info)
{
  static const int objects_val[5][3]= {{0,0,0}, {1,0,0}, {0,1,0}, {0,0,0}, {0,0,1}};
  int              loop=structs->Algo_info->loop;
  int              loop_offset=2>>loop;
  int              imrows=((pic_info->height)>>loop), imcols=((pic_info->width)>>loop);
  int              bytes_per_line=pic_info->bytes_per_line, bytes_per_pixel=pic_info->bytes_per_pixel;
  int              row_index=0;
  BYTE             *table0=structs->objects[0].table;
  BYTE             *table1=structs->objects[1].table;
  BYTE             *table2=structs->objects[2].table;
  unsigned char    *data=pic_info->data;
  VIS_color_def    *rgb_space=structs->RGB_space;
  
  /* row_index and col_index : processed picture indexes (i.e. objects indexes) */
  for (row_index=(structs->uplimit>>loop); row_index<=imrows-loop_offset; row_index++) {
    int row=(row_index<<loop);   /* Camera picture index */
    int col_index=0;             /* Processed picture index */
    for (col_index=loop_offset+1; col_index<=imcols-loop_offset; col_index++) {
      int col=(col_index<<loop); /* Camera picture index */
      
      /* Get class of the current pixel */
      int pos=(row*bytes_per_line)+(col*bytes_per_pixel); /* Camera picture index */
      int val_index=rgb_space->RGB_class[data[pos+0]>>VIS_SHIFT_BITS]
	  /* Get class of the pixel */  [data[pos+1]>>VIS_SHIFT_BITS]
	                                [data[pos+2]>>VIS_SHIFT_BITS];
      /* Fill in the binary tables (undenoised) for the 3 objects */
      int offset=row_index*imcols+col_index; /* Objects table index */
      const int *val=objects_val[val_index]; /* TRUE / FALSE lookup */
      table0[offset]=val[0]; table1[offset]=val[1]; table2[offset]=val[2]; /* Fill in tables */
    }
  }
  
  /* Denoise binary tables for the 3 objects and get objects limits */
  VIS_Denoise_and_process(structs, pic_info, VIS_BALL, 0);
  VIS_Denoise_and_process(structs, pic_info, VIS_YGOAL, 1);
  VIS_Denoise_and_process(structs, pic_info, VIS_BGOAL, 2);
  
  return (structs->region);
}
/****f* libvision/VIS_Team_robot
 * SYNOPSIS
 *   int VIS_Team_robot(void)
 * DESCRIPTION
 *   Returns a robot team and ID.
 *   Input  : N/A
 *   Output : a two digits integer xy where x is the team and y is the ID.
 * SEE ALSO
 *   N/A
 ****/
int VIS_Team_robot(void)
{
  int  team=0;
  BYTE robot_id=0;

  robot_id=OSMachineID(); /* Get ID of the robot */
  team=CAMInit(NORMAL);   /* Quickcam : COLCAM=16, Eyecam : COLCAM+1 (i.e. 17) */

  switch(team)
    {
    case COLCAM+1: /* EyeCam */
    case 32 :      /* Fixes the 32 instead of 17 (COLCAM+1) bug for EyeCam */
      return (robot_id-1);
      break; 
    default :      /* QuickCam */  
      return (10+robot_id-1);
      break;
    }
}
/****f* libvision/VIS_Modify_algo
 * SYNOPSIS
 *   void VIS_Modify_algo(VIS_structs *structs,  VIS_algo *new_algo, Picture *new_pic)
 * DESCRIPTION
 *   Modify algorithm values and reallocate processing memory.
 *   Use realloc() dynamic memory reallocation if USE_MALLOC is defined.
 *   Also checks VIS_structs.
 *   Input  : 'VIS_structs' to modify, a new 'VIS_algo' structure and the current 'Picture' parameters.
 *            'VIS_algo' is 4 integers : depth, fast, step and loop (modified automatically).
 *            Program is checking error in 'VIS_algo' values, display them and return an error code.
 *            If an error occurs, no changes are made.
 *            Program displaying error message and returning error code if a realloc() error occurs.
 *   Output : an error code : ==0 -> no errors; >0 -> an error occured.
 * SEE ALSO
 *   VIS_Init_var(), VIS_Release_var()
 ****/
int VIS_Modify_algo(VIS_structs *structs,  VIS_algo *new_algo, Picture *new_pic)
{
  int error=0;

  /* Check VIS_structs */
  error=VIS_Check_init(structs);

  /* If structure ok then go... */
  if (!error) {

    /* error checking */
    if ((new_algo->step!=1) && (new_algo->step!=2)) {
      error=410; VIS_Error_display("! Param. error !", error, "step=only 1 or 2"); return error;
    }
    if ((new_algo->fast==1) && (new_algo->depth>1)) {
      error=420; VIS_Error_display("! Param. error !", error, "fast for depth=1"); return error;
    }

    /* Modify if no errors */
    if (!error) {
      /* Modify Algo_info struct */
      structs->Algo_info->depth=new_algo->depth; structs->Algo_info->fast=new_algo->fast;
      structs->Algo_info->step=new_algo->step; structs->Algo_info->loop=new_algo->step-1;

#if defined(USE_MALLOC)
      
      int size=0, l_index=0;
      
      /* VIS_color_process objects initialisation */
      size=(new_pic->height>>structs->Algo_info->loop)*(new_pic->width>>structs->Algo_info->loop);
      for (l_index=0; l_index<3; l_index++) {
        structs->objects[l_index].table=(BYTE *) realloc(structs->objects[l_index].table, size);
        if (structs->objects[l_index].table==NULL) { error=510+l_index;}

        structs->objects[l_index].X_list=(BYTE *) realloc(structs->objects[l_index].X_list, size);
        if (structs->objects[l_index].X_list==NULL) { error=520+l_index;}

        structs->objects[l_index].Y_list=(BYTE *) realloc(structs->objects[l_index].Y_list, size);
        if (structs->objects[l_index].Y_list==NULL) { error=530+l_index;}
      }
      if (error) { VIS_Error_display("! Realloc err. !", error, " objects init."); return error;}

#else

      /* Check pic_info */
      if ((new_pic->height!=imagerows) || (new_pic->width!=imagecolumns)) {
	VIS_Error_display("!   new_pic    !", -1, "have bad infos");
	return -1; /* -1 : external error (new_pic does not belong to libvision) */
      }      
      
#endif
    }
  }
  /* No errors occured... */
  return error;
}
                     /************************/
                     /* Load defaults values */
                     /************************/
static void VIS_Load_defaults(VIS_structs *structs)
{
  int robot=0, team=0, robot_id=0;

  /* Get robot team and number */
  robot=VIS_Team_robot();
  team=(int) (robot/10);
  robot_id=robot-10*team;

  /* RGB Levels defaults values */
  structs->RGB_space->RGB_levels.R_level=VIS_CAM_RLEVEL[team][robot_id];
  structs->RGB_space->RGB_levels.G_level=VIS_CAM_GLEVEL[team][robot_id];
  structs->RGB_space->RGB_levels.B_level=VIS_CAM_BLEVEL[team][robot_id];

  /* Camera defaults values */
  structs->CAM_offset->height=VIS_CAM_HEIGHT[team][robot_id];
  structs->CAM_offset->lenght=VIS_CAM_LENGHT[team][robot_id];
  structs->CAM_offset->angle=VIS_CAM_ANGLE[team][robot_id];
  structs->CAM_offset->alpha=VIS_CAM_ALPHA[team][robot_id];
  structs->CAM_offset->beta=VIS_CAM_BETA[team][robot_id];

  /* Algorithm default values */
  structs->Algo_info->depth=VIS_DEPTH;
  structs->Algo_info->fast=VIS_FAST;
  structs->Algo_info->step=VIS_STEP;
  structs->Algo_info->loop=VIS_STEP-1;

  /* Upper goal limit */
  structs->uplimit=VIS_GOAL_UP[team][robot_id];
}
                     /*************************/
                     /* Show status of levels */
                     /*************************/
static void VIS_Show_levels(int r, int g, int b, int l)
{
  /* Calibration levels status */
  LCDClear();
  LCDSetPrintf(0,0, "* Calibration *");
  LCDSetPrintf(1,0, "---------------");
  LCDSetPrintf(2,3, "Red   : %d", r);
  LCDSetPrintf(3,3, "Green : %d", g);
  LCDSetPrintf(4,3, "Blue  : %d", b);
  LCDSetPrintf(5,3, "Limit : %d", l);  
}
                     /*************************/
                     /* Show status of offset */
                     /*************************/
static void VIS_Show_offset(int alpha_offset, float beta_offset)
{
  LCDClear();
  LCDSetPrintf(0,0, "* Offset cal. *");
  LCDSetPrintf(1,0, "---------------");
  LCDSetPrintf(3,0, "  Alpha : %d ", alpha_offset);
  LCDSetPrintf(4,0, "  Beta  : %2.1f ", rad2deg(beta_offset));
}
                     /*************************/
                     /* Calibration of levels */
                     /*************************/
static void VIS_Cal_color(VIS_structs *structs, VIS_cameraCapture_t camera, void *cameradata)
{
  int  R_level=0, G_level=0, B_level=0;
  char Key=0;

  /* Default values */
  R_level=structs->RGB_space->RGB_levels.R_level;
  G_level=structs->RGB_space->RGB_levels.G_level;
  B_level=structs->RGB_space->RGB_levels.B_level;

  /* Show default, ask for calibration */
  VIS_Show_levels(R_level,  G_level, B_level, structs->uplimit);
  LCDMenu("No"," "," ","Yes");

  /* Wait for Yes or No */
  KEYGetBuf(&Key);
  while (Key!=KEY1 && Key!=KEY4) {
    KEYGetBuf(&Key);
  }
  AUBeep();

  /* Beginning of calibration... */
  if (Key==KEY4) {
    R_level=VIS_Cal_single_level(structs, 1, camera, cameradata);
    G_level=VIS_Cal_second_level(structs, R_level, camera, cameradata);
    B_level=VIS_Cal_single_level(structs, 3, camera, cameradata);

    /* Color levels changed, color classes lookup tables need to be updated */
    structs->color_done=FALSE;

    /* Display result of calibration */
    VIS_Show_levels(R_level, G_level, B_level, structs->uplimit);
    LCDMenu(" ", " ", " ", "END");
    KEYWait(KEY4);
  }

  /* Setting up levels */
  structs->RGB_space->RGB_levels.R_level=R_level;
  structs->RGB_space->RGB_levels.G_level=G_level;
  structs->RGB_space->RGB_levels.B_level=B_level;
}
                     /********************************/
                     /* Camera vertical angle offset */
                     /********************************/
static void VIS_Cal_offset(VIS_structs *structs, VIS_cameraCapture_t camera, void *cameradata)
{
  Picture          *pic_info=camera(cameradata); /* Initialise local pic_info */
  int              imgrows=pic_info->height, imgcols=pic_info->width;
  int              middle=0;
  int              distance=260; /* in mm */
  char             Key=0;
  BYTE             test=0;
  BYTE             alpha_offset=structs->CAM_offset->alpha;
  float            beta_offset=structs->CAM_offset->beta;
  float            beta_th=0, beta_r=0;
  ServoHandle      panning_camera=0;
  VIS_color_region *result=NULL;

  /* Ask for calibration */
  VIS_Show_offset(alpha_offset, beta_offset);
  LCDMenu("No", " ", " ", "Yes");

  /* Wait for Yes or No */
  while (Key!=KEY1 && Key!=KEY4) {
    Key=KEYRead();
  }

  /* Yes, calibration... */
  if (Key==KEY4) {

    /*--------------------------*/
    /* Alpha offset calibration */
    /*--------------------------*/

    /* Panning camera servo init */
    panning_camera=SERVOInit(SERVO11);

    /* BW picture initialisation */
    VIS_Init_bwimg(structs->LCD_info);

    /* Draw border of camera picture */
    VIS_Draw_border(structs->LCD_info->data, pic_info->height, pic_info->width);

    /* Alpha offset display */
    LCDClear();
    LCDSetPrintf(0,11, "Alpha");
    LCDSetPrintf(2,11, "Mid.");
    LCDMenu(" ", " ", " ", "Stop");
    
    /* Pan from 0 to 255 until middle is found */
    alpha_offset=0; test=0; Key=0;

    while ((!test) && (Key=KEYRead()!=KEY4)) {
      SERVOSet(panning_camera, alpha_offset);

      /* Getting image */
      OSWait(50); /* Avoid tilting camera problems... */
      pic_info=camera(cameradata);
      if (NULL==pic_info) { VIS_Error_display("!  Alpha cal.  !", -1, " pic_info==NULL");}

      /* Process to find classes */
      result=VIS_Find_classes(structs, pic_info);
      if (NULL==result) { VIS_Error_display("!  Alpha cal.  !", 610, "  result==NULL");}
      
      if (result[0].color==VIS_BALL) {
        middle=(result[0].coord.right+result[0].coord.left)/2;
        VIS_Draw_limits(structs->LCD_info->data, &result[0], 1);
      }
      LCDPutImage(structs->LCD_info->data);
      if (result[0].color==VIS_BALL) {
        VIS_Draw_limits(structs->LCD_info->data, &result[0], 0);
      }

      /* Display values */
      LCDSetPrintf(1,11, "%d ", alpha_offset);
      LCDSetPos(3,11);
      if (result[0].color==VIS_BALL) { LCDPrintf("%d ", middle-(imgcols/2));}
      else { LCDPrintf("    ");}
      LCDSetPrintf(4,11, "%d ", result[0].coord.right);
      LCDSetPrintf(5,11, "%d ", result[0].coord.left);
      
      if (middle==(imgcols/2)) { test=1;}
      else { alpha_offset++;}

      if (alpha_offset>255) { AUTone(110,500); alpha_offset=0;}
    }

    /* Wait for anykey */
    LCDMenu("  A", "N Y", "K E", "Y");
    KEYWait(ANYKEY);

    /* Panning camera servo release */
    SERVORelease(panning_camera);

    /*-------------------------*/
    /* Beta offset calibration */
    /*-------------------------*/

    /* Offset calibration display */
    LCDClear();
    LCDSetPrintf(0,0, "* Offset cal. *");
    LCDSetPrintf(1,0, "---------------");
    LCDSetPrintf(2,0, "  Align. ball");
    LCDSetPrintf(3,0, " Adj. distance");
    LCDSetPrintf(4,0, " with : + or -");
    LCDSetPrintf(5,0, "Cal : calibrate");
    LCDMenu(" ", " ", " ", "Nxt");

    /* Wait for "Nxt" */
    KEYWait(KEY4);

    /* Initialise display */
    LCDClear();
    LCDSetPrintf(0,11, "Dist\n");
    LCDSetPrintf(3,10, "Offset");
    LCDMenu(" - ", " + ", " ", "Cal");
    Key=0;

    /* BW picture initialisation */
    VIS_Init_bwimg(structs->LCD_info);

    /* Draw border of camera picture */
    VIS_Draw_border(structs->LCD_info->data, pic_info->height, pic_info->width);

    /* Waiting for "Cal" */
    while (Key!=KEY4) {

      /* Getting image */
      pic_info=camera(cameradata);
      if (NULL==pic_info) { VIS_Error_display("!   Beta cal.  !", -1, " pic_info==NULL");}

      /* Process to find classes */
      result=VIS_Find_classes(structs, pic_info);
      if (NULL==result) { VIS_Error_display("!   beta cal.  !", 620, "  result==NULL");}

      /* Display border of ball class */
      if (result[0].color==VIS_BALL) {
        middle=(result[0].coord.right+result[0].coord.left)/2;
        VIS_Draw_limits(structs->LCD_info->data, &result[0], 1);
     }
      else { middle=-1;}
      LCDPutImage(structs->LCD_info->data);
      if (result[0].color==VIS_BALL) {
        VIS_Draw_limits(structs->LCD_info->data, &result[0], 0);
      }

      /* Reading a key from keyboard */
      Key=KEYRead();

      /* Changing distance value and displaying it */
      if (Key==KEY1) { distance-=10;}
      if (Key==KEY2) { distance+=10;}

      /* Process angles, offset and distance */
      beta_th=atan( (float) (structs->CAM_offset->height-VIS_BALL_RADIUS)/distance);
      beta_r=( (float) 0.75*structs->CAM_offset->angle/2)*
           ((( (float) (result[0].coord.top+result[0].coord.bot)/2)-( (float) imgrows/2))/
             ( (float) imgrows/2));
      beta_offset=beta_th-beta_r;

      /* Display information on LCD */
      LCDSetPrintf(1,11, "%d cm", (int) (distance/10));
      LCDSetPrintf(4,11, "%3.1f ", rad2deg(beta_offset));
    }

    /* Modifiy values */
    structs->CAM_offset->alpha=alpha_offset;
    structs->CAM_offset->beta=beta_offset;
    structs->dist_done=FALSE;

    /* Display new values */
    VIS_Show_offset(alpha_offset, beta_offset);
    LCDMenu(" ", " ", " ", "Nxt");

    /* Wait for Nxt */
    KEYWait(KEY4);
  }
}
                     /***********************/
                     /* Adjust single level */
                     /***********************/
static int VIS_Cal_single_level(VIS_structs *structs, int RGB, VIS_cameraCapture_t camera, void *cameradata)
{
  int              test=0, number=0;
  int              length=0, mn=0;
  int              cal_length=0;
  char             Key=0;
  Picture          *pic_info=camera(cameradata); /* Initialise local picture */
  VIS_color_region goal;
  
  /* BW picture initialisation */
  VIS_Init_bwimg(structs->LCD_info);

  /* Draw border of camera picture */
  VIS_Draw_border(structs->LCD_info->data, pic_info->height, pic_info->width);

  /* Level adjust display */
  LCDClear();
  if (RGB==1) {
    LCDSetPrintf(0,0, "Red Calibration");
    LCDSetPrintf(1,0, "---------------");
    LCDSetPrintf(6,0, "the yellow goal");
  }
  else if (RGB==3){
    LCDSetPrintf(0,0, "Blue Calibration");
    LCDSetPrintf(1,0, "----------------");
    LCDSetPrintf(6,0, "the blue goal");
  }
  LCDSetPrintf(3,0, "Please put robot");
  LCDSetPrintf(4,0, "in the middle of");
  LCDSetPrintf(5,0, "the field facing");
  LCDMenu(" ", " ", " ", "Nxt");

  /* Wait for "Nxt" */
  KEYWait(KEY4);

  /* Calibration loop */
  test=0; length=0; number=0;
  cal_length=(int) 4+(2*pic_info->width * /* Length of the goal in pixels */
		      atan(( (float) VIS_GOAL/2.0)/ (float) VIS_CAL_DIST) /
		      structs->CAM_offset->angle);
  LCDClear(); LCDMenu(" ", " ", " ", "END");

  while (!test) {

    if (number) { AUTone(110,500);} /* Bad calibration beep */
    else { AUBeep();}               /* First try beep */

    /* Get picture */
    pic_info=camera(cameradata);

    /* Initial "magic number" */
    mn=20;
    do {
      /* Get position of the goal */
      VIS_Cal_single_process(structs, pic_info, mn, RGB, &goal);
      length=goal.coord.right-goal.coord.left;

      /* Display values */
      if (RGB==1) { LCDSetPrintf(0,12, "Red\n");}
      else if (RGB==3) { LCDSetPrintf(0,11, "Blue\n");}
      LCDSetPrintf(1,11, "Level\n");
      LCDSetPrintf(2,12, "%d \n", mn);
      LCDSetPrintf(3,11, "Size\n");
      LCDSetPrintf(4,12, "%d \n", length);
      LCDSetPrintf(5,12, "%d \n", goal.coord.top);
      
      mn++; number=1;

      Key=KEYRead();
    } while ((length>=cal_length) && (Key!=KEY4) && (mn<=100));
    
    /* Check level */
    pic_info=camera(cameradata);
    VIS_Cal_single_process(structs, pic_info, mn-1, RGB, &goal);
    length=goal.coord.right-goal.coord.left;   
    if ((length>=cal_length-1 && length<=cal_length+1) || Key==KEY4) { test=1;}
  }
  
  /* Update upper limit value */
  if (RGB==1) { structs->uplimit=goal.coord.top;}
  else {if (goal.coord.top<structs->uplimit) { structs->uplimit=goal.coord.top;}}
  return (mn-1);
}
                     /***********************/
                     /* Adjust second level */
                     /***********************/
static int VIS_Cal_second_level (VIS_structs *structs, int R_level, 
				 VIS_cameraCapture_t camera, void *cameradata)
{
  int     test=0, number=0;
  int     length=0, mn=0;
  int     cal_length;
  char    Key=0;
  Picture *pic_info=camera(cameradata); /* Initialise local picture */

  /* BW picture initialisation */
  VIS_Init_bwimg(structs->LCD_info);

  /* Draw border of camera picture */
  VIS_Draw_border(structs->LCD_info->data, pic_info->height, pic_info->width);

  /* Level adjust display */
  LCDClear();
  LCDSetPrintf(0,0, "Gr. Calibration");
  LCDSetPrintf(1,0, "---------------");
  LCDSetPrintf(3,0, "Please put robot");
  LCDSetPrintf(4,0, "in the middle of");
  LCDSetPrintf(5,0, "the field facing");
  LCDSetPrintf(6,0, "the yellow goal");
  LCDMenu(" ", " ", " ", "Nxt");

  /* Wait for "Nxt" */
  KEYWait(KEY4);

  /* Calibration Loop */
  test=0; length=0; number=0;
  cal_length=(int) 4+(2*pic_info->width *
		      atan(( (float) VIS_GOAL/2.0)/ (float) VIS_CAL_DIST) /
		      structs->CAM_offset->angle);

  LCDClear(); LCDMenu(" ", " ", " ", "END");

  while (!test) {
    if (number) { AUTone(110,500);} /* Bad calibration beep */
    else { AUBeep();}               /* First try beep */

    /* Get picture */
    pic_info=camera(cameradata);

    /* Initial "magic number" */
    mn=45;
    do {
      length=VIS_Cal_second_process(structs, pic_info, R_level, mn);

      LCDSetPrintf(0,11, "Green\n");
      LCDSetPrintf(1,11, "Level\n");
      LCDSetPrintf(2,12, "%d \n",mn);
      LCDSetPrintf(3,11, "Size\n");
      LCDSetPrintf(4,12, "%d \n",length);

      mn--; number=1;

      Key=KEYRead();
    } while ((length<cal_length-1) && (Key!=KEY4) && (mn>=0));

    /* Check level */
    pic_info=camera(cameradata);
    length=VIS_Cal_second_process(structs, pic_info, R_level, mn-1);
    if (((length>=cal_length-1) && (length<=cal_length+1)) || (Key==KEY4)) { test=1;}
  }
  return mn;
}
                     /***************************/
                     /* Single level processing */
                     /***************************/
static void VIS_Cal_single_process(VIS_structs *structs, Picture *pic_info, int mn, int RGB,
				   VIS_color_region *goal)
{
  int               row=0;
  int               top=0, bot=0, right=0, left=0;
  int               imgrows=pic_info->height;
  int               imgcols=pic_info->width;
  int               bytes_per_line=pic_info->bytes_per_line;
  int               bytes_per_pixel=pic_info->bytes_per_pixel;
  BYTE              *data=pic_info->data;
  VIS_color_process cal_obj=structs->objects[0];

  /* Find pixels */
  for (row=0; row<imgrows; row++) {
    int col;
    for (col=0; col<imgcols; col++) {
      int R=0, G=0, B=0, L=0;
      int offset=0;

      /* Get R,G and B values, calculate L */
      offset=(row*bytes_per_line)+(col*bytes_per_pixel);
      R=data[offset]; G=data[offset+1]; B=data[offset+2]; L=R+G+B;

      /* Fill in binary table */
      offset=row*imgcols+col;
      if (RGB==1) { cal_obj.table[offset]=VIS_Cal_find_pixels(mn, R, L);}
      else { cal_obj.table[offset]=VIS_Cal_find_pixels(mn, B, L);}
    }
  }

  /* Display undenoised result */
  VIS_Compute_bw(structs, cal_obj.table, pic_info);
  LCDPutImage(structs->LCD_info->data);

  /* Denoise */
  cal_obj.index=VIS_Denoise(2, &cal_obj, pic_info, structs->uplimit);

  /* Initialisation of variables top, bot, right, left */
  top=imgrows+1; bot=-1; right=-1; left=imgcols+1;
  VIS_Find_limits(&top, &bot, &right, &left, &cal_obj);

  /* Update result struct */
  goal->coord.top=top; goal->coord.bot=bot; goal->coord.right=right; goal->coord.left=left;
}
                     /***************************/
                     /* Second level processing */
                     /***************************/
static int VIS_Cal_second_process(VIS_structs *structs, Picture *pic_info, int R_level, int mn)
{
  int               row=0;
  int               top=0, bot=0, right=0, left=0;
  int               imgrows=pic_info->height, imgcols=pic_info->width;
  int               bytes_per_line=pic_info->bytes_per_line, bytes_per_pixel=pic_info->bytes_per_pixel;
  BYTE              *data=pic_info->data;
  VIS_color_process cal_obj=structs->objects[0];

  /* Find pixels */
  for (row=0; row<imgrows; row++) {
    int col;
    for (col=0; col<imgcols; col++) {
      int R=0, G=0, B=0, L=0;
      int offset=0;

      /* Get R,G and B values, calculate L */
      offset=(row*bytes_per_line)+(col*bytes_per_pixel);
      R=data[offset]; G=data[offset+1]; B=data[offset+2]; L=R+G+B;

      /* Fill in binary table */
      offset=row*imgcols+col;
      cal_obj.table[offset]=VIS_Cal_find_pixels2(R_level, mn, R, G, L);
    }
  }

  /* Display undenoised result */
  VIS_Compute_bw(structs, cal_obj.table, pic_info);
  LCDPutImage(structs->LCD_info->data);

  /* Denoise */
  cal_obj.index=VIS_Denoise(2, &cal_obj, pic_info, structs->uplimit);

  /* Initialisation of variables top, bot, right, left */
  top=imgrows+1; bot=-1; right=-1; left=imgcols+1;
  VIS_Find_limits(&top, &bot, &right, &left, &cal_obj);

  return (right-left);
}
                     /***************************/
                     /* Calibration find pixels */
                     /***************************/
static int VIS_Cal_find_pixels(int mn, int level, int L)
{
  if (100*level>=mn*L) {
    return 1;
  }
  return 0;
}
                     /******************************************************/
                     /* Calibration find pixels 2 i.e. with two conditions */
                     /******************************************************/
static int VIS_Cal_find_pixels2(int mn1, int mn2, int level1, int level2, int L)
{
  if ((100*level1>=mn1*L) && (100*level2>=mn2*L)) {
    return 1;
  }
  return 0;
}
                     /*************/
                     /* Find blue */
                     /*************/
static int VIS_Find_blue(int mn, int B, int L)
{
  if (100*B>=mn*L) {
    return VIS_BGOAL;
  }
  return VIS_NOTHING;
}
                     /*********************************************/
                     /* Find red i.e. yellow goal and orange ball */
                     /*********************************************/
static int VIS_Find_red(int r_mn, int g_mn, int R, int G, int L)
{
  int r_test=(100*R>=r_mn*L), g100=100*G, g_mnxL=g_mn*L;

  if (r_test && (g100>g_mnxL)) {
    return VIS_YGOAL;
  }
  else if (r_test && (g100<g_mnxL)) {
    return VIS_BALL;
  }
  return VIS_NOTHING;
}

/****f* libvision/VIS_cameraCapture
 * SYNOPSIS
 *   Picture *VIS_cameraCapture(void *capturedata)
 * DESCRIPTION
 *   Initialise the Picture structure for the camera at the first call.
 *   Fetch one image from the camera the following calls.
 * SEE ALSO
 *   N/A
 ****/
Picture *VIS_cameraCapture(/*@unused@*/void *cameradata) /* @...@ lclint comment */
{
static int      cam_firsttime=1;
static Picture  cam_pic;
static colimage cam_img;

  if (1==cam_firsttime) {
      picture_init(&cam_pic, imagecolumns, imagerows, pix_rgb24, 3, &cam_img[0][0][0], sizeof(cam_img));
      cam_firsttime=0;
      cameradata=NULL; /* Just to avoid cameradata unused warning... */
    }

  if (0!=CAMGetColFrame(&cam_img, 0)) {
      return NULL;
    }
  else {
    return &cam_pic;
  }
}
