/***********************************************************************************************************/
/* Header file for the vision library                                                                      */
/* Declaration of constants, structures and user functions                                                 */
/***********************************************************************************************************/

/****h* libvision/libvision
 *DESCRIPTION
 * The format for error messages is :
 * ----------------------------------
 *     -1 for wrong external structures to the libvision,
 *      0 for no errors,
 *     >0 for internal libvision errors.
 *
 *
 * To change the algorithm parameters, a "VIS_Algo temp_algo;" is needed :
 * -----------------------------------------------------------------------
 *     typedef struct {
 *        int depth; depth to denoise
 *        int fast;  use fast algorithm, only for depth=1
 *        int step;  process every dots or every two dots only
 *        int loop;  shift for loops, i.e. loop=step-1, used INTERNALLY only i.e. no need to modify.
 *     } VIS_algo;
 *
 *
 * The classes are defined for the robot soccer program as :
 * ---------------------------------------------------------
 *     #define VIS_NOTHING 0 Object non seen on the picture
 *     #define VIS_BALL    1 Ball
 *     #define VIS_YGOAL   2 Yellow goal
 *     #define VIS_BGOAL   4 Blue goal
 *
 *
 * The objects limits and class is returned throuh a pointer to a "VIS_color_region *result;" structure.
 * -----------------------------------------------------------------------------------------------------
 *     typedef struct {
 *        int       color; Class (i.e. color) of the object
 *        VIS_coord coord; Rectangular coordinate of the object
 *     } VIS_color_region;
 *
 *   Where the coordinate structure is defined as :
 *     typedef struct {
 *        int top, bot, right, left; Rectangular coordinate of the object
 *     } VIS_coord;
 *
 *
 * The distance to the ball is returned through a "VIS_distance distance;" structure.
 * ----------------------------------------------------------------------------------
 *     typedef struct {
 *        int d_row; depth distance
 *        int d_col; side distance (<0 left and >0 right)
 *        int dist;  direct distance : dist^2=d_row^2+d_col^2
 *     } VIS_distance;
 *
 *
 * The USE_MALLOC.
 * ---------------
 *     If USE_MALLOC is defined, then the program uses dynamic memory allocation, i.e. malloc(), realloc()
 *   and free() functions.
 *     If USE_MALLOC is not defined, then the program uses global variables with maximum memory needs as
 *   realloc() can not be used.
 *     Avoids the use of dynamic memory allocation on EyeBot.
 *
 *
 * The VIS_SIM.
 * ------------
 *     If you are using the simulator, then use VIS_SIM==1, if you use the Eyebot, use VIS_SIM==0.
 ****/

#ifndef VISION_H
#define VISION_H

#ifdef __cplusplus
extern "C" {
#endif

  /* C libraries */
#include <math.h>
#include <stdlib.h>
#include <string.h>

  /* EyeBot libraries */
#include <eyebot.h>
#include "picproc.h"
#include "VIS_Constants.h" /* Local user's "loadable" variables, part of LIBVISION */

  /*********************************************************************************************************/
  /* CONSTANTS */
  /*************/
                /* Color classes values, use power of 2 for new class */
#define VIS_NOTHING 0 /* Object non seen on the picture */
#define VIS_BALL    1 /* Ball */
#define VIS_YGOAL   2 /* Yellow goal */
#define VIS_BGOAL   4 /* Blue goal */

                /* Distances lookup tables : normal table, border table, angle tables */
#define VIS_ROW    0
#define VIS_COL    1
#define VIS_ALPHA  0
#define VIS_BETA   1
#define VIS_NORMAL 0
#define VIS_BORDER 1

  /*********************************************************************************************************/
  /* STRUCTURES */
  /**************/
                 /* Type of the framegrabber function pointer  */
  typedef Picture * (*VIS_cameraCapture_t)(void *);

                 /* Image processing struct : table -> undenoised class, index, X and Y_list denoised data */
  typedef struct {
    int size;
    BYTE *table;  /* Binary table */
    BYTE *X_list; /* X positions list */
    BYTE *Y_list; /* Y positions list */
    int  index;   /* Number of elements in X_list and Y_list */
  } VIS_color_process;

                 /* Coordinates structure */
  typedef struct {
    int top, bot, right, left; /* Rectangular coordinate of the object */
  } VIS_coord;

                 /* Color region struct : color and coordinates of the box */
  typedef struct {
    int       color; /* Color of the object */
    VIS_coord coord; /* Rectangular coordinate of the object */
  } VIS_color_region;

                 /* Camera angles and positions offsets to calibrate */
  typedef struct {
    float angle;   /* Lens angle in radians */
    BYTE  alpha;   /* Servo alpha offset */
    float beta;    /* Camera beta offset */
    int   height;  /* Camera height */
    int   lenght;  /* Camera distance to the middle of the robot */
  } VIS_camera_offset;

                /* Calibrated color levels */
  typedef struct {
    int R_level, G_level, B_level; /* R, G and B "magic number" */
  } VIS_color_levels;

                /* Color definition : R, G and B levels and lookup table */
  typedef struct {
    VIS_color_levels RGB_levels; /* R, G and B levels structure */
    BYTE             RGB_class[VIS_VALUES][VIS_VALUES][VIS_VALUES]; /* Color classes lookup table */
  } VIS_color_def;

                /* Distances lookup tables : normal table, border table, angle tables */
  typedef struct {
    short Dist_dist[2][imagerows][imagecolumns];     /* Direct relative distance, border or normal */
    short Dist_coord[2][2][imagerows][imagecolumns]; /* Relative distance in x and y, border or normal */
    float Dist_angle[2][imagerows][imagecolumns];    /* Alpha and beta angles */
  } VIS_distance_tables;

                /* Distance structure */
  typedef struct {
    short d_row; /* depth distance */
    short d_col; /* side distance */
    short dist;  /* direct distance : dist^2=d_row^2+d_col^2 */
    float alpha; /* sideways angle */
    float beta;  /* up-down angle */
  } VIS_distance;

                /* Algorithm options struct */
  typedef struct {
    int depth; /* denoise one dot around */
    int fast;  /* use fast algorithm, only for depth=1 */
    int step;  /* process every dots or two dots only */
    int loop;  /* shift for loops */
  } VIS_algo;

                /* libvision structure */
  typedef struct {
    int                 uplimit;      /* Goal upper limit */
    BOOL                color_done;   /* Color classes lookup table updated */
    BOOL                dist_done;    /* Distance lookup table updated*/
    BOOL                init_success; /* Structure initialised successfully */
    Picture             *LCD_info;    /* LCD display */
    VIS_algo            *Algo_info;   /* Used algorythm parameters */
    VIS_color_def       *RGB_space;   /* Color classes lookup table */
    VIS_color_region    *region;      /* Region limits (ball, yellow and blue goals) */
    VIS_camera_offset   *CAM_offset;  /* Camera parameters */
    VIS_distance_tables *Dist_tables; /* Distance lookup table */
    VIS_color_process   *objects;     /* Objects list (ball, yellow and blue goals)*/
  } VIS_structs;

  /*********************************************************************************************************/
  /* GENERAL */
  /***********/
              /* Initialisation of the camera */
  int     VIS_Init_cam(int cammode);
              /* BW picture initialisation */
  int     VIS_Init_bwimg(Picture *LCD_info);
              /* Let time to camera for autobrightness */
  void    VIS_Autobrightness_delay(VIS_cameraCapture_t camera, void *cameradata, int number);
              /* Compute b&w 1D array from segmented 2D array */
  void    VIS_Compute_bw(VIS_structs *structs, BYTE *temp, Picture *pic_info);
              /* Draw object limits and center on bw picture */
  void    VIS_Draw_limits(BYTE *bwimg, VIS_color_region *region, int value);
              /* Draw border of the image */
  void    VIS_Draw_border(BYTE *bwimg, int rows, int cols);
              /* Malloc error display */
  void    VIS_Error_display(char title[16], int error, char msg[16]);
              /* Initialise the show progress of loop */
  void    VIS_Init_show_progress(char title[16]);
              /* Show progress of loop */
  void    VIS_Show_progress(int actual, int max);
              /* Fetching one image from the camera */
  Picture *VIS_cameraCapture(void *cameradata);

  /*********************************************************************************************************/
  /* SOCCER */
  /**********/
             /* Testing RGB Calibration for soccer */
  int  VIS_Soccer_test(void);
             /* Vision library init function */
  VIS_structs *VIS_Init_var(Picture *LCD_info, Picture *pic_info);
             /* Vision library release function */
  void VIS_Release_var(VIS_structs *structs);
             /* Checks if structure has been initialised properly */
  int  VIS_Check_init(VIS_structs *structs);
             /* Color and distance lookup tables calibration */
  int  VIS_Init_calibration(VIS_structs *structs, VIS_cameraCapture_t camera, void *cameradata);
             /* Init classes */
  int  VIS_Init_color_classes(VIS_structs *structs);
             /* Distance calibration */
  int  VIS_Init_distance(VIS_structs *structs, Picture *pic_info);
             /* Distance table coordinates */
  int  VIS_Get_position(VIS_structs *structs, VIS_distance *distance,
                        int objclass);
             /* Find classes */
  VIS_color_region* VIS_Find_classes(VIS_structs *structs, Picture *pic_info);
             /* Find robot team and ID */
  int  VIS_Team_robot(void);
             /* Modify algorithm values and reallocate processing memory */
  int  VIS_Modify_algo(VIS_structs *structs,  VIS_algo *new_algo, Picture *new_pic);

#ifdef __cplusplus
}
#endif

#endif
