#ifndef PLANECONST_H
#define PLANECONST_H
/****************************************************************************
planeconst.h - Created by Peter Mauger 20/05/01
modified 12/10/01

Simply defines all enumerated types and constants used by the automated flight
system. The top two groups of constants can be changed by the user, however
knowledge of the effects of these changes should be obtained before they are
made.
****************************************************************************/

/* These constants define the external state of the control surfaces. They are generally
*  defined by the settings on the horns (different settings cause different deflections).
*  They should be changed by the user in the event that the system is installed in a new 
*  aircraft or the current aircraft settings are changed.
*
*  To determine the new values:
*		First set the control surface to neutral (or 0 degrees deflection).
*		Then set it to the minimum setting possible (servo setting 0) and measure the deflection in degrees. 
*		Repeat for maximum setting (servo setting 255) and enter the values below.
*/
#define MAX_ANGLE_RUDDER 	15	/* Maximum angle rudder can be set to (equiv to 255 of servo)*/
#define MIN_ANGLE_RUDDER 	-20	/* Minimum angle rudder can be set to (equiv to 0 of servo)*/
#define MAX_ANGLE_AILERON 	5	/* Maximum angle aileron can be set to (equiv to 255 of servo)*/
#define MIN_ANGLE_AILERON 	-5	/* Minimum angle aileron can be set to (equiv to 0 of servo)*/
#define MAX_ANGLE_ELEVATOR	5	/* Maximum angle elevator can be set to (equiv to 255 of servo)*/
#define MIN_ANGLE_ELEVATOR	-5	/* Minimum angle elevator can be set to (equiv to 0 of servo)*/
#define MAX_SETTING_THROTTLE	70	/* Maximum setting (percentage of open throttle) that throttle can be set to (equiv to 255 of servo)*/
#define MIN_SETTING_THROTTLE	20	/* Minimum setting (percentage of open throttle) that throttle can be set to (equiv to 0 of servo)*/

/* These constants can be user defined, however the effects of change should be known */
#define EARTH_RADIUS	   6378140/* The radius of the Earth in metres */
#define MAGNETIC_DEVIATION -1.958 /* The magnetic deviation from true north */
#define DIST_TOL	   10	  /* The distance the plane can be from the waypoint (in metres) before it is not at the waypoint */
#define HEAD_TOL	   5	  /* Degrees of tolerance within which the plane is deemed to be heading towards the waypoint */

/* These are program specific and affect the memory usage (if set too high they may cause memory overflow!) */
#define MAX_WAYPOINTS	   100	  /* max number of waypoints allowed in one flight */
#define MAX_FILE_LENGTH	   10000  /* max file size of 10000 characters (approx. 100 waypoints) */

#define MAX_BURST_LENGTH   700	  /* max length of characters read from the GPS in one pass */
#define MAX_MESSAGE_LENGTH 100	  /* max number of characters for a GPS message */
#define HDR_LENGTH	   6	  /* Length of the header identifiers for GPS messages (ie $GPGLL) */
#define GLL_MESSAGE_LENGTH 35	  /* max number of characters for a GLL message */

#define MAX_LOG_SIZE	   300000 /* max number of characters required to upload log */
#define MAX_ERROR_MESGS	   3000   /* max number of events that can be logged */
#define MAX_EVENT_LENGTH   30	  /* max length of an event tag string */

/* These specify the communications protocols used */
#define GPS_PORTNUM	   SERIAL2    	/* GPS port setting */
#define PC_PORTNUM	   SERIAL1    	/* PC port number */
#define GPS_BAUDRATE	   SER9600    	/* GPS port speed setting */
#define PC_BAUDRATE	   SER57600	/* PC port baud rate */

#define GPS_TIMEOUT        10           /* number of timeouts before cancelling receive */
#define WPFILE_TIMEOUT	   10		/* number of timeouts before deciding wp file won't be sent */

/* Enumerated types for hardware related functions */
typedef enum {OFF, ON, UNDEFINED} switchstate;			/* switchstate is used to determine the state*/
								/* of the autopilot switch, and the light */
typedef enum {RUDDER, AILERON, ELEVATOR, THROTTLE} servoconsts;	/* servoconsts provides identifiers for */
								/* the control surfaces */
typedef enum {NORMAL, SET} anglesetmode;			/* anglesetmode is used to determine which limits to use */
								/* when setting the angle */
#define SWITCH		   2		/* Analogue channel that the switch is on */
#define SWITCH_THRESHHOLD  512		/* The AD converter value that separates the 2 switch levels */
#define SWITCH_MAX	   1024		/* The maximum value of the AD converter */

/* Timing constants */
#define VSHORT 	10			/* The number of hundredths of seconds for a very short pause */
#define SHORT 	50			/* The number of hundredths of seconds for a short pause */
#define LONG 	100			/* The number of hundredths of seconds for a long pause */
#define VLONG	1000			/* The number of hundredths of seconds for a very long pause */

/* Enumerated types for plane state related functions */
typedef enum {STRAIGHT, LEFT, RIGHT} turnstate;			/* the currently desired state of turn */
typedef enum {LEVEL, HIGHER, LOWER} altitudestate;		/* the currently desired change of altitude */
typedef enum {TIME, EVENT} upltype;				/* the type of log upload desired */

/* Enumerated types for event logging (and error response) */
typedef enum {
	NOERROR,				/* no error occured */
	SWITCHDETECT,				/* the switch cannot be detected */
	SERVOERROR,				/* an error has occured with a servo */
	INITERROR,				/* Initialisation procedure failed */
	IDERROR,				/* the control surface id was invalid */
	INVALIDSERVO,				/* the servo handle was invalid */
	LIMITERROR,				/* the angle is outside the set limits */
	LIMITCROSS,				/* min limit > max limit */
	ABSOLUTELIMITERROR,			/* the limit is set outside of the absolute values */
	COMPASSERROR,				/* the compass was not initialised correctly */
	COMPASSHEADINGERROR,			/* the heading could not be determined */
	NOHEADING,				/* no heading could be determined */
	GPSERROR,				/* the GPS was not initialised correctly */
	GPSPOSERROR,				/* the position could not be determined */
	MESGTOOLONG,				/* the gps message was too long */
	NOPOSITION,				/* no position could be determined */
	NOWPS,					/* no waypoints are currently stored */
	INVALIDWPNUM,				/* the waypoint number entered was greater than the number of waypoints */
	WPFILEERROR,				/* the waypoint file was not correctly received or parsed */
	DIVZERO,				/* a divide by zero was attempted */
	NEGVAL,					/* a negative value was entered in a non-negative variable */
	COMMERROR,				/* a communications error occured */
	SENDERROR,				/* the send failed */
	SWITCHMANUAL,				/* (Not an error) the switch changed state to manual */
	SWITCHAUTO,				/* (Not an error) the switch changed state to autopilot */
	LOGPOSITION,				/* (Not an error) data logging event */
	ATWP,					/* (Not an error) reached a waypoint */
	FLIGHTCOMPLETE				/* (Not an error) flight completed event */
} error;

typedef enum {FLIGHT, INITIALISE} errorstage;	/* identifies which part of the program */
						/* an error occured in */

/* Cardinal point constants */
#define NORTH 1					/* multiplies lat by +1 if in northern hemisphere */
#define SOUTH -1				/* multiplies lat by -1 if in southern hemisphere */
#define EAST 1					/* multiplies long by +1 if in eastern hemisphere */
#define WEST -1					/* multiplies long by -1 if in western hemisphere */

/* Enumerated type to determine the program state (usually set by user inputs) */
typedef enum {GO, TERMINATE} programstate;	/* defines the desired state of the program */

/* Boolean TRUE/FALSE */
typedef enum {FALSE, TRUE} bool;		/* defines the boolean variables required */
		
#endif








