#include <stdlib.h>
#include <stdio.h>
#include "Gyro.h"

// Number of samples needed to get a filtered inclinometer reading
#define INCLINE_SAMPLES  5
#define ANGLE_ERROR_SAMPLES 5

// variance threshold to recalibrate
#define RECALIBRATE_VARIANCE 0.05
#define MIN_TIME_BETWEEN_RECALIBRATE 100
#define MAX_RECALIBRATE_CORRECTION 4500

// These need to be global for the timer sample function to access...
int g_iZeroVelocity;			// Raw Gyro reading that is quated to 0deg/sec
int g_iAngularVelocity;			// Gyro estimate of angular velocity (last raw reading)
int g_iAngle;					// Gyro estimate of angle
int g_iSampleTime;				// System time (tics) sample last taken
int g_iInclineZero;				// The raw AD inclinometer reading at 0deg
int g_iTimeLastCalibrated;		// System time (tics) last calibrated
int g_iMaxAngle;				// The raw gyro estimate of 90deg

int g_iPWMInclinometer;
int g_iPWMInclineZero;

int g_iGyroAngleCorrection;			// After recalibration, this is the angle error to be corrected
int g_iGyroAngleCorrectionDelta;	// the error is corrected in this size chunks - not all at once

int g_iRecordInclineIdx;						// Last few inclinometer readings are stored,
int g_rgiRecordInclineTime[INCLINE_SAMPLES];	// in case we want to recalibrate the Gyroscope.
int g_rgiRecordIncline[INCLINE_SAMPLES];

// Prototypes
void OnlineRecalibrate(float fMeanIncline);
void DownloadResult();


CGyro::CGyro()
{
	int i;

	g_iRecordInclineIdx = 0;

	// Start with no timer...
	m_hSampleTimer = 0;

	// Need to initialise the SERVO driving Gyro...
	m_hServo = SERVOInit(SERVO8);
	SERVOSet(m_hServo, 127);

	// Initialise the gyroscope sensor
	accinit();

	// Release & Re-initialise gyroscope
	accrelease();
	accinit();

	// Initialise state values
	InitialiseState();
}

CGyro::~CGyro()
{
	// Release the gyroscope sensors
	accrelease();

	// Release the servo
	SERVORelease(SERVO8);

	// Release the timer
	OSDetachTimer( m_hSampleTimer );
}


/////////////////////////////////////////////////////////
//
// TIMER SAMPLE
//
// Handler function for sample timer - gets reading from
// Gyro and updates state variables
//
void CGyro::TimerSample()
{
	int iAngVel;
	int iPWMIncline;
	int iTimeNow, iElapsed;
	int iIncline;
	int i;
	int iRawADReading;
	float fMeanIncline;
	float fVarianceIncline;

	// Check if there is a new sample:
	iAngVel = accreadX();
//	iPWMIncline = accreadY();
	
	// Only do stuff if there was a sample
//	if (iPWMIncline > -1)
//	{
//		g_iPWMInclinometer = iPWMIncline - g_iPWMInclineZero;
//	}

	if (iAngVel > -1)
	{
		iAngVel = iAngVel;

		// Get the elapsed time
		iTimeNow = OSGetCount();
		iElapsed = iTimeNow - g_iSampleTime;

		// Correct elapsed time if rolled over!
		if (iElapsed < 0) iElapsed += 0xFFFFFFFF; // ROLL OVER

		// Correct the angular velocity
		iAngVel -= g_iZeroVelocity;

		// Calculate angular displacement
		g_iAngle += (g_iAngularVelocity * iElapsed);
		g_iAngularVelocity = -iAngVel;
		g_iSampleTime = iTimeNow;

		// Read inclinometer (a few times to drain residual values)
		iRawADReading = OSGetAD(INCLINE_CHANNEL);
		iRawADReading = OSGetAD(INCLINE_CHANNEL);


		// If we are recording, and we have started...store data
		if (g_iTimeLastCalibrated > 0)
		{
			g_rgiRecordInclineTime[g_iRecordInclineIdx] = g_iSampleTime;
			g_rgiRecordIncline[g_iRecordInclineIdx++]  = iRawADReading - g_iInclineZero;
			if (g_iRecordInclineIdx >= INCLINE_SAMPLES) 
			{
				g_iRecordInclineIdx = 0;

				// Work out mean of samples
				fMeanIncline = 0.0;
				for (i=0; i<INCLINE_SAMPLES; i++)
				{
					fMeanIncline += g_rgiRecordIncline[i];
				}
				fMeanIncline /= INCLINE_SAMPLES;

				// Work out variance
				fVarianceIncline = 0.0;
				for (i=0; i<INCLINE_SAMPLES; i++)
				{
					fVarianceIncline += ((g_rgiRecordIncline[i] - fMeanIncline) * (g_rgiRecordIncline[i] - fMeanIncline));
				}
				fVarianceIncline /= (INCLINE_SAMPLES-1);

				if (fVarianceIncline < RECALIBRATE_VARIANCE &&
					(g_iSampleTime - g_iTimeLastCalibrated) > MIN_TIME_BETWEEN_RECALIBRATE) 
					OnlineRecalibrate(fMeanIncline);
			}
		}
	}

	// If there is a correction factor remaining to apply, apply it!
	if (g_iGyroAngleCorrection > 0)
	{
		g_iGyroAngleCorrection -= g_iGyroAngleCorrectionDelta;
		g_iAngle -= g_iGyroAngleCorrectionDelta;
	}
}


//////////////////////////////////////////////////////////
//
// INITIALISE STATE
//
// Recalibrates zero state, and takes the first state
// sample, assuming angle is zero.
//
void CGyro::InitialiseState()
{
	// May need to release sample timer...
	if (m_hSampleTimer)
	{
		OSDetachTimer( m_hSampleTimer );
		m_hSampleTimer = 0;
	}

	// Pause for a bit
	LCDPrintf("Wait...\n");
	OSWait(INIT_DELAY);
	LCDPrintf("Calibrate...\n");

	// Initialise state values...
	g_iAngle = 0;
	g_iZeroVelocity = 0;
	g_iTimeLastCalibrated = 0;
//	g_iPWMInclinometer = 0;

	g_iGyroAngleCorrection = 0;

	ZeroAngVel();

	g_iSampleTime = OSGetCount();


	LCDPrintf("Sampling...\n");
	// Create the sampling timer:
	m_hSampleTimer = OSAttachTimer(1, (TimerFnc) (this->TimerSample));
}

///////////////////////////////////////////////////////////
//
// GET TIME SINCE CALIBRATED
//
// Returns the number of ticks elapsed since the last recalibration,
// or -1 if the Gyro has never been calibrated.
//
int CGyro::GetTimeSinceCalibration()
{
	int iTimeNow;

	// Get the elapsed time
	iTimeNow = OSGetCount();

	if (g_iTimeLastCalibrated>0)
	{
		return (iTimeNow - g_iTimeLastCalibrated);
	}
	else
	{
		return (-1);
	}
}

//////////////////////////////////////////////////////////
//
// RECALIBRATE
//
// Recalibrates zero state, and finds max and zero angles.
//
void CGyro::Recalibrate()
{
	int iKey;
	int i;

	iKey = 0;

	// Get MaxAngle & ZeroAngle
	LCDPrintf("Lie on face & KEY");
	iKey = KEYGet();
	InitialiseState();

	LCDPrintf("Stand up & KEY");
	iKey = KEYGet();

	g_iMaxAngle = -g_iAngle;
	g_iAngle = 0;

	// Wake up AD channel
	OSGetAD(INCLINE_CHANNEL);
	LCDPrintf("Reset Inclinometer...\n");

	// Get the zero-inclinometer reading:
	g_iInclineZero = 0;
	for (i=0;i<INCLINE_SAMPLES; i++)
	{
		g_iInclineZero += OSGetAD(INCLINE_CHANNEL);
		LCDPrintf(".");
	}
	LCDPrintf("\n");

	g_iInclineZero /= INCLINE_SAMPLES; // Inclinometer units


	// Remember time calibrated:
	g_iTimeLastCalibrated = OSGetCount();

}

void CGyro::AdjustMaxAngle(int iGUnits)
{
	g_iMaxAngle += iGUnits;
	LCDPrintf(".%d.\n", g_iMaxAngle);
}

//////////////////////////////////////////////////////////
//
// ONLINE RECALIBRATE
//
// Recalibrates gyro, based on inclinometer readings
//
void OnlineRecalibrate( float fMeanIncline )
{
	int i;
	int iIncline;
	int iGyroAngleError;
	int iGyroAngleVelError;
	int iMaxIncline;
	int iMinIncline;

	int iTimeElapsed;

	if (g_iTimeLastCalibrated > 0)
	{
		// Determine elapsed time:
		iTimeElapsed = g_iTimeLastCalibrated;

		// Remember time calibrated:
		g_iTimeLastCalibrated = OSGetCount();
		
		iTimeElapsed = g_iTimeLastCalibrated - iTimeElapsed;

		// g_iAngle is the gyro measureed angle, it should be what the inclinometer says!
		iIncline = (fMeanIncline * g_iMaxAngle) / 123;
		iGyroAngleError = g_iAngle - iIncline;

		// Error in angular velocity is due to an incorrect g_iZeroVelocity
		iGyroAngleVelError = iGyroAngleError / iTimeElapsed;

		g_iZeroVelocity -= iGyroAngleVelError / 8;
		
		// Set angle to correct value
		if (iGyroAngleError >  MAX_RECALIBRATE_CORRECTION) iGyroAngleError = MAX_RECALIBRATE_CORRECTION;
		if (iGyroAngleError < -MAX_RECALIBRATE_CORRECTION) iGyroAngleError = -MAX_RECALIBRATE_CORRECTION;

		g_iGyroAngleCorrectionDelta = iGyroAngleError / 20;
		g_iGyroAngleCorrection = iGyroAngleError;
	}
}


///////////////////////////////////////////////////////////
//
// ZERO ANG VEL
//
// Records the current reading of angular velocity from the
// Gyro, and assumes this value corresponds to 0 rad/sec.
//
// Function also "zero's" the inclinometer reading of angle
//
void CGyro::ZeroAngVel()
{
	int i, iRawVel, iPWMIncline;

	int cPWM6, cPWM7;

	cPWM6 = 0;
	cPWM7 = 0;
	g_iPWMInclinometer = 0;
	g_iZeroVelocity = 0;

	// Take samples until one of the PWM channels has enough...
	while (cPWM6 < INIT_SAMPLES && cPWM7<INIT_SAMPLES)
	{
		iRawVel = accreadX();
//		iPWMIncline = accreadY();

		if (iRawVel > -1)
		{
			cPWM6++;
			g_iZeroVelocity += iRawVel;
		}
//		if (iPWMIncline > -1)
//		{
//			cPWM7++;
//			g_iPWMInclineZero += iPWMIncline;
//		}
	}

	// Top up CHN6...
	for (i=cPWM6; i<INIT_SAMPLES; i++)
	{
		// Get a sample
		do {iRawVel = accreadX();} while (iRawVel == -1);

		g_iZeroVelocity += iRawVel;
	}

	// Top up CHN7...
//	for (i=cPWM7; i<INIT_SAMPLES; i++)
//	{
//		// Get a sample
//		do {iPWMIncline = accreadY();} while (iPWMIncline == -1);
//
//		g_iPWMInclineZero += iPWMIncline;
//	}
//
//	g_iPWMInclineZero = g_iPWMInclineZero / INIT_SAMPLES;
	g_iZeroVelocity = g_iZeroVelocity / INIT_SAMPLES;
	g_iAngularVelocity = 0;
}

/////////////////////////////////////////////////////////
//
// GET GYRO STATE FUNCTIONS
//

///////////////////////////////////////////////////////////
//
// GET ANGLE
//
// If the sensor has been calibrated
//   GetAngle returns the Gyro estimate of Angle in RADIANS
// Else
//   GetAngle returns the Gyro estimate of Angle in "GUnits" from Zero
//
float CGyro::GetAngle()
{
	if (g_iTimeLastCalibrated > 0)
	{
		return ((g_iAngle * PI) / (g_iMaxAngle * 2.0));
	}
	else
	{
		return ((float)g_iAngle);
	}
}

///////////////////////////////////////////////////////////
//
// GET ANGLULARVEL
//
// If the sensor has been calibrated
//   GetAngle returns the Gyro estimate of Angular Velocity in RADIANS/Second
// Else
//   GetAngle returns the Gyro estimate of Angular Velocity in "GUnits/Tick"
//
float CGyro::GetAngularVel()
{
	float fResult;

	if (g_iTimeLastCalibrated > 0)
	{						// Note :   g_iAngularVelocity = GUnits/Tick
		fResult = (float)g_iAngularVelocity*(float)100.0;		  // GUnits/Second
		fResult = ((fResult * PI) / (g_iMaxAngle * (float)2.0)); // Radians/Second
		return (fResult);
	}
	else
	{
		return ((float)g_iAngularVelocity);
	}
}

void CGyro::DisplayAngularVelocity()
{
	LCDPrintf(" %d\n", g_iAngularVelocity);
}
//
//////////////////////////////////////////////////////////


///////////////////////////////////////////////////
//
// DOWNLOAD RESULT
// use this to take recordings of sensor samples.
// the values to download must be first stored in global arrays
//
//
void DownloadResult()
{
/*
	int i;
	char sz[80];

	// Ready?
	LCDPrintf("ready to send?");
	KEYGet();

	// Open serial port
	OSInitRS232(SER115200, RTSCTS, SERIAL1);

	// Download gyro angle
	TransmitLine("GYRO_ANGLE");
	for (i=0; i<g_iARRAYSIZE; i++)\
	{
		sprintf(sz, "...
		TransmitLine(sz);
	}

	LCDPrintf("Done.\n");
	KEYGet();
*/
}

///////////////////////////////////////////////////
//
// TRANSMIT SAMPLE
//
//
int TransmitSample(int iTime, int iIncline, int iAngle, int iAngleVel)
{
	static int s_iOpenPort=0;

	char sz[256]; // a line
	int iRow, iCol;
	int ierr;

	// May need to open the port (only the first time...)
	if (s_iOpenPort == 0)
	{
		if ( ierr = OSInitRS232(SER115200, RTSCTS, SERIAL1) ) return ierr;
		s_iOpenPort = 1;

		// Send the verison number
		sprintf(sz, "Inclinometer - no motion, 0 deg\n");
		if (ierr = TransmitLine(sz)) return ierr;
	}

	// Output sample to RS232 & screen...
	sprintf(sz, "%d\t%d\t%d\t%d\0", iTime, iIncline, iAngle, iAngleVel);
	if (ierr = TransmitLine(sz)) return ierr;
	return 0;
}


///////////////////////////////////////////////////
//
// Send data back to host computer for processing
//

int TransmitLine(char * sz)
{
	int i;
	int c;
	int ierr;

	c = strlen(sz);
	for (i=0; i<c; i++)
	{
		if (ierr = OSSendCharRS232(sz[i], SERIAL1)) break;
	}
	if (!ierr) ierr = OSSendCharRS232('\n', SERIAL1);
	return ierr;
}


