/* *************************************************************************
 * matrix.h declares the template class matrix used to have a simple access
 * to matix based data structures.
 * ************************************************************************* */

#ifndef _MATRIX_H_
#define _MATRIX_H_

#include <stdexcept>
#include <fstream>
#include <sstream>
#include <limits>

#include "StringifiedException.h"

#ifndef SPECIAL_PETER_VERSION
#include "string_helpers.h"
#endif /* SPECIAL_PETER_VERSION */

/** The matrix_base class is a template class representing data in a
 * matrix oriented way. like in an image with columns and rows. You
 * can access the data throw several access operators. Additionaly
 * exists functionality to load and save the data from/to disk. */
template<class T>
class matrix_base
{
public:
	typedef enum { ROW_MAJOR = 0, COL_MAJOR = 1 } storage_order_t;

private:
	T* _buffer;
	size_t _first_dim;
	size_t _second_dim;
	bool _owned;
	storage_order_t _storage_order;

	template<class U> void copy_from( const matrix_base<U>& m )
	{
		_first_dim = m.first_dim();
		_second_dim = m.second_dim();
		if( _storage_order != m.storage_order() )
			std::swap( _first_dim, _second_dim );
		allocate( _first_dim, _second_dim );
		for( size_t j=0; j < _second_dim; j++ )
			for( size_t i=0; i < _first_dim; i++ )
				operator()( i, j ) = m( i, j, _storage_order );
	}

public:
	typedef T value_type;
	typedef T* iterator;
	typedef const T* const_iterator;

	inline size_t first_dim() const { return _first_dim; }
	inline size_t second_dim() const { return _second_dim; }
	inline storage_order_t storage_order() const { return _storage_order; }
	inline size_t width() const
		{ return _storage_order == ROW_MAJOR ? _first_dim : _second_dim; }
	inline size_t height() const
		{ return _storage_order == ROW_MAJOR ? _second_dim : _first_dim; }

	/** Copy constructor */
	matrix_base( const matrix_base<T>& m ) :
		_buffer(0), _owned(false), _storage_order(m.storage_order())
	{
		copy_from(m);
	}

	matrix_base( size_t first_dim, size_t second_dim,
		storage_order_t storage_order ) :
		_buffer(0), _owned(false), _storage_order(storage_order)
	{	allocate( first_dim, second_dim ); }

	matrix_base( T* buffer, size_t first_dim, size_t second_dim,
		storage_order_t storage_order, bool copy ):
		_buffer(buffer), _first_dim(first_dim), _second_dim(second_dim), 
		_owned(false), _storage_order(storage_order)
	{
		if( copy )
		{
			allocate( first_dim, second_dim );
			::memcpy( _buffer, buffer, first_dim * second_dim * sizeof(T) );
		}
	}

	matrix_base( const char* fname, size_t first_dim, size_t second_dim,
		storage_order_t storage_order ):
		_buffer(0), _first_dim(first_dim), _second_dim(second_dim), 
		_owned(false), _storage_order(storage_order)
	{
		allocate(first_dim, second_dim);

		load( fname );
	}

	virtual ~matrix_base() { clear(); }

	void clear()
	{
		if( _buffer )
		{
			if( _owned )
				delete [] _buffer;
			_buffer = 0;
			_first_dim = 0;
			_second_dim = 0;
		}
	}

	template<class U>
	const matrix_base<T>& operator=( const matrix_base<U>& m )
	{
		copy_from(m);

		return *this;
	}

	inline T& operator()( size_t first_dim_coordinate, size_t second_dim_coordinate )
	{
#ifdef DEBUG
		if( (first_dim_coordinate >= _first_dim) ||
			(second_dim_coordinate >= _second_dim) )
			throw std::out_of_range("Access violation in matrix_base<T>().");
#endif /* DEBUG */

		return _buffer[first_dim_coordinate + _first_dim * second_dim_coordinate];
	}

	inline T& operator()( size_t first_dim_coordinate,
		size_t second_dim_coordinate,
		storage_order_t storage_order )
	{
		if( storage_order != _storage_order )
			return operator()( second_dim_coordinate, first_dim_coordinate );
		else
			return operator()( first_dim_coordinate, second_dim_coordinate );
	}

	inline T& operator[]( size_t pos )
	{
#ifdef DEBUG
		if( pos >= size() )
			throw std::out_of_range("Access violation in matrix_base<T>[].");
#endif /* DEBUG */

		return _buffer[ pos ];
	}

	/* These const_cast hacks exists just because i am to lazy to copy the
	 * code of the non const versions */ 
	inline const T& operator()( size_t first_dim_coordinate, size_t second_dim_coordinate ) const
		{ return const_cast< matrix_base<T>& >(*this)( first_dim_coordinate, second_dim_coordinate );	}
	inline const T& operator()( size_t first_dim_coordinate, size_t second_dim_coordinate, storage_order_t storage_order ) const
		{ return const_cast< matrix_base<T>& >(*this)( first_dim_coordinate, second_dim_coordinate, storage_order );	}
	inline const T& operator[]( size_t pos ) const
		{ return const_cast< matrix_base<T>& >(*this)[pos]; }

	iterator begin() const { return _buffer; }
	iterator end() const { return _buffer + size(); }

	size_t size() const { return _first_dim * _second_dim; }
	size_t size_in_bytes() const { return _first_dim * _second_dim * sizeof(T); }
	bool empty() const { return _buffer == 0; }

	void* buffer() { return _buffer; }

	/** This method resizes the underlying memory buffer. Old data
	 * will be preserved but in old storage order, that means if the
	 * first dimension changes, the data will me mixed up. If this
	 * matrix_base is not owner of its memory block then a new,
	 * uninitialized block is allocated. */
	void resize( size_t first_dim, size_t second_dim = 1 )
	{
		if( _buffer )
		{
			_buffer = (T*)::realloc( _buffer,
				first_dim * second_dim * sizeof(T) );
		}
		else
		{
			_buffer = first_dim * second_dim > 0 ?
				new T[first_dim * second_dim] : 0;
		}
		_owned = true;
		_first_dim = first_dim;
		_second_dim = second_dim;
	}

	/** This method just frees possibly allocated data and allocates
	 * new according to passed dimension arguments. */ 
	void allocate( size_t first_dim, size_t second_dim = 1 )
	{
		clear(); /* this is neccessary if we don't own the buffer */
		resize( first_dim, second_dim );
	}

	/** This method simply calls memset with this buffer, the given
	 * char c and the size of this buffer. */
	void memset(int c)
		{ ::memset( _buffer, c, size_in_bytes() ); }

#ifndef SPECIAL_PETER_VERSION
	/** This method loads data from a file. If there was some memory
	 * assigned previous to this call then only the assigned amount of
	 * data will be loaded.  Otherwise the whole file will be loaded
	 * (if it contains an integer multiple of sizeof(T). */
	void load( const char* fname )
	{
		using namespace std;

		try
		{
			ifstream f(fname, ios::in|ios::binary);
			if( ! f )
				throw EIosFailure("can not open file");
			f.exceptions( ios::badbit|ios::failbit|ios::eofbit );

			if( ! _buffer )
			{
				f.seekg( 0, ios::end );
				size_t file_size = f.tellg();
				f.seekg( 0, ios::beg );

				if( file_size % sizeof(T) != 0 )
					throw EIosFailure("file has an invalid size");
				allocate( file_size / sizeof(T) );
			}

			f.read( (char*)_buffer, size_in_bytes() );
			f.close();
		}
		catch( exception& e ) {
			throw EIosFailure( EMsgE()
				<< "Error loading file '" << fname << "':\n" << e.what() << "." );
		}
	}

	/** This method simply saves the data to the specified file. */
	void save( const char* fname ) const
	{
		using namespace std;

		try
		{
			ofstream f(fname, ios::out|ios::trunc|ios::binary);
			if( ! f )
				throw EIosFailure("can not open file");
			f.exceptions( ios::badbit | ios::failbit | ios::eofbit );
			f.write( (char*)_buffer, size_in_bytes() );
			f.close();

#ifdef DEBUG_DUMP_TEXT
			ofstream g( (string(fname) + string(".txt")).c_str() );
			g << "Width  : " << _second_dim << endl
			  << "Height : " << _first_dim << endl;
#endif /* DEBUG_DUMP_TEXT */
		}
		catch( exception& e ) {
			throw EIosFailure( EMsgE()
				<< "Error saving file '" << fname << "':\n"
				<< e.what() << "." );
		}
	}


	void load_text( const char* fname )
	{
		using namespace std;

		try
		{
			ifstream f( fname );
			if( ! f )
				throw EIosFailure("can not open file");
			f.exceptions( ios::badbit );
			string s;
			getline( f, s );
			s = trim(s);
			istringstream is( s );

			size_t h = 0, w = 0;
			T dummy;
			while( ! is.eof() )
			{
				is >> dummy;
				w++;
			}

			while( ! f.eof() && ! s.empty() )
			{
				h++;
				getline( f, s );
			}

			allocate( h, w );

//BUG: seekg( 0, ios::beg ) seemed not to work ???
			f.close();
			f.open( fname );

			for( size_t r = 0; r < _first_dim; r++ )
				for( size_t c = 0; c < _second_dim; c++ )
				{
					if( f.eof() )
						throw EIosFailure("can not read file completely");
					f >> operator()( r, c );
				}
		}
		catch( exception& e ) {
			throw EIosFailure( EMsgE()
				<< "Error loading file '" << fname << "':\n"
				<< e.what() << "." );
		}
	}

	/** This method saves the data shifted and converted to U.*/
	template<class U> void save( const char* fname, int shift ) const
	{
		matrix_base<U> m(size());

		for( size_t i=0; i<size(); i++ )
			m[i] = _buffer[i] >> shift;

		m.save( fname );
	}

	/** This method saves the data as an ASCII file.  It uses the
	 * standard ostream operator << of the U. */
	template<class U> void save_text( const char* fname, U = U(0) ) const
	{
		using namespace std;

		try
		{
			ofstream f( fname, ios::out|ios::trunc /*|~ios::binary*/ );
			if( ! f )
				throw EIosFailure("can not open file");
			f.exceptions( ios::badbit | ios::failbit | ios::eofbit );

			for( size_t i=0; i < _first_dim; i++ )
			{
				for( size_t j=0; j < _second_dim; j++ )
					f << U( operator()(i, j) ) << "\t";
				f << std::endl;
			}
		}
		catch( exception& e ) {
			throw EIosFailure( EMsgE()
				<< "Error saving file '" << fname << "':\n"
				<< e.what() << "." );
		}
	}

	void save_text( const char* fname ) const
	{
		save_text<T>( fname );
	}

	template<class U> inline U sum() const
	{
		return sum<U>( 0, 0, _first_dim-1, _second_dim-1 );
	}

	template<class U> inline U average() const
	{
		return average<U>( 0, 0, _first_dim-1, _second_dim-1 );
	}

	template<class U> inline U sigma() const
	{
		return sigma<U>( 0, 0, _first_dim - 1, _second_dim - 1 );
	}

	inline T min() const
	{
		return min( 0, 0, _first_dim - 1, _second_dim - 1 );
	}

	inline T max() const
	{
		return max( 0, 0, _first_dim - 1, _second_dim - 1 );
	}

	template<class U>
	U sum( size_t first_d1, size_t first_d2, size_t last_d1, size_t last_d2 ) const
	{
		U sum = 0;

		for( size_t i=first_d2; i <= last_d2; i++ )
			for( size_t j=first_d1; j <= last_d1; j++ )
				sum += (*this)(j,i);

		return sum;
	}

	template<class U>
	U average( size_t first_d1, size_t first_d2, size_t last_d1, size_t last_d2 ) const
	{
		return sum<U>(first_d1, first_d2, last_d1, last_d2) /
			((last_d2-first_d2+1) * (last_d1-first_d1+1));
	}

	template<class U>
	U sigma( size_t first_d1, size_t first_d2, size_t last_d1, size_t last_d2 ) const
	{
		return sigma<U>( first_d1, first_d2, last_d1, last_d2,
			average<U>( first_d1, first_d2, last_d1, last_d2 ) );
	}

	template<class U>
	U sigma( size_t first_d1, size_t first_d2, size_t last_d1, size_t last_d2, U avr ) const
	{
		U sum = 0;
		U b = 0;

		for( size_t i=first_d2; i <= last_d2; i++ )
			for( size_t j=first_d1; j <= last_d1; j++ )
			{
				b = (*this)(j,i);
				sum += (b-avr)*(b-avr);
			}

		return sqrt( sum / ((last_d2-first_d2+1) * (last_d1-first_d1+1) - 1) );
	}

	T min( size_t first_d1, size_t first_d2, size_t last_d1, size_t last_d2 ) const
	{
		T min = (*this)(first_d1, first_d2);

		for( size_t i=first_d2; i <= last_d2; i++ )
			for( size_t j=first_d1; j <= last_d1; j++ )
				min = min_elements( min, (*this)(j,i) );

		return min;
	}

	T max( size_t first_d1, size_t first_d2, size_t last_d1, size_t last_d2 ) const
	{
		T max = (*this)(first_d1, first_d2);

		for( size_t i=first_d2; i <= last_d2; i++ )
			for( size_t j=first_d1; j <= last_d1; j++ )
				max = max_elements( max, (*this)(j,i) );

		return max;
	}
#endif /* SPECIAL_PETER_VERSION */
};

template<class T>
class matrix : public matrix_base<T>
{
public:
	matrix( const matrix<T>& m ) : matrix_base<T>( m ) {}

	matrix( size_t height = 0, size_t width = 1 ) :
		matrix_base<T>( height, width, COL_MAJOR ) {}

	matrix( T* buffer, size_t height, size_t width = 1, bool copy = false ) :
		matrix_base<T>( buffer, height, width, COL_MAJOR, copy ) {}

	matrix( const char* fname, size_t height = 0, size_t width = 1 ) :
		matrix_base<T>( fname, height, width, COL_MAJOR ) {}
};

template<class T>
class cmatrix : public matrix_base<T>
{
public:
	cmatrix( const matrix<T>& m ) : matrix_base<T>( m ) {}

	cmatrix( size_t width = 0, size_t height = 1 ) :
		matrix_base<T>( width, height, ROW_MAJOR ) {}

	cmatrix( T* buffer, size_t width, size_t height = 1, bool copy = false ) :
		matrix_base<T>( buffer, width, height, ROW_MAJOR, copy ) {}

	cmatrix( const char* fname, size_t width = 0, size_t height = 1 ) :
		matrix_base<T>( fname, width, height, ROW_MAJOR ) {}
};

#endif /* _MATRIX_H_ */
