/* *************************************************************************
 * Triple.h declares the template class triple used to have a simple
 * access to RGB based data structures.
 * ************************************************************************* */

#ifndef _TRIPLE_H_
#define _TRIPLE_H_

#include <iostream>
#include <limits>
#include <algorithm>
#include <cmath>
#include <cstdlib> /* e.g. abs(int) */

template<class T>
struct triple
{
	typedef T value_type;

	T r; T g; T b;

	inline triple( const T& val = T() ): r(val), g(val), b(val) {};
	explicit inline triple( const T& r, const T& g, const T& b ): r(r), g(g), b(b) {};
	template<class U> inline triple( const triple<U>& rh )
		{ r = T(rh.r); g = T(rh.g); b = T(rh.b); }

	/** index operator: 0 = r, 1 = g, 2 = b */
	inline T& operator[]( int index )
		{ return reinterpret_cast<T*>(this)[index]; }
	inline const T& operator[]( int index ) const
		{ return reinterpret_cast<const T*>(this)[index]; }

	/** @name special access operators
	 * attention: these methods are not neccessarily a disjunct dissection
	 * of the triple. (imagine r == g == b, the result would be three times
	 * &b, i think.) */
	//@{
	inline T max_element() const
		{ return (((r > g)? r : g) > b)? ((r > g)? r : g) : b; }
	inline T mid_element() const
		{ return ((r!=max_element()) && (r!=min_element()))? r : (((g!=max_element()) && (g!=min_element()))? g : b); }
	inline T min_element() const
		{ return (((r < g)? r : g) < b)? ((r < g)? r : g) : b; }
	//@}

	inline T average() const
		{ return (r+b+g)/3; }

	inline T intensity() const
		{ return T(sqrt((r*r+b*b+g*g)/3.0)); }

	inline void swap_rb( bool swap = true )
		{ if( swap ) std::swap( r, b ); }

	/** @name assignment operators */
	//@{
   	template<class U> inline triple<T>& operator=( const triple<U>& rh )
		{ r = T(rh.r); g = T(rh.g); b = T(rh.b); return *this; }
   	template<class U> inline triple<T>& operator=( const U& rh )
		{ r = T(rh); g = T(rh); b = T(rh); return *this; }

	inline triple<T>& operator*=( const triple<T>& rh )
		{ r *= rh.r; g *= rh.g; b *= rh.b; return *this; }
	inline triple<T>& operator*=( const T& rh )
		{ r *= rh; g *= rh; b *= rh; return *this; }
	template<class U> inline triple<T>& operator/=( const U& rh )
		{ r /= rh; g /= rh; b /= rh; return *this; }
	template<class U> inline triple<T>& operator+=( const triple<U>& rh )
		{ r += rh.r; g += rh.g; b += rh.b; return *this; }
	template<class U> inline triple<T>& operator-=( const triple<U>& rh )
		{ r -= rh.r; g -= rh.g; b -= rh.b; return *this; }

	inline triple<T>& operator<<=( const int& i )
		{ r <<= i; g <<= i; b <<= i; return *this; }
	inline triple<T>& operator>>=( const int& i )
		{ r >>= i; g >>= i; b >>= i; return *this; }
	//@}

	inline triple<T>& operator++()
		{ ++r; ++g; ++b; return *this; }
	inline triple<T>& operator--()
		{ --r; --g; --b; return *this; }
};

/** @name triple operators
 * These are the standard operators except min- and max_values. */
//@{

#define DEF_ARITHMETIC_OPERATOR_FOR_TRIPLE( OP ) \
template<class T> \
inline triple<T> operator OP( const triple<T>& lh, const triple<T>& rh ) \
{	return triple<T>(lh.r OP rh.r, lh.g OP rh.g, lh.b OP rh.b); }
DEF_ARITHMETIC_OPERATOR_FOR_TRIPLE(+)
DEF_ARITHMETIC_OPERATOR_FOR_TRIPLE(-)
DEF_ARITHMETIC_OPERATOR_FOR_TRIPLE(*)
DEF_ARITHMETIC_OPERATOR_FOR_TRIPLE(/)

template<class T>
inline triple<T> operator*( const triple<T>& lh, const T& rh )
{	return triple<T>(lh.r * rh, lh.g * rh, lh.b * rh); }
template<class T>
inline triple<T> operator/( const triple<T>& lh, const double& rh )
{	return triple<T>((double)lh.r / rh, (double)lh.g / rh, (double)lh.b / rh); }

template<class T>
inline triple<T> operator<< ( const triple<T>& tr, const int& i )
{	return triple<T>(tr.r<<i, tr.g<<i, tr.b<<i); }
template<class T>
inline triple<T> operator>> ( const triple<T>& tr, const int& i )
{	return triple<T>(tr.r>>i, tr.g>>i, tr.b>>i); }


#define DEF_RELATIONAL_OPERATOR_FOR_TRIPLE( OP ) \
template<class T, class U> \
inline bool operator OP( const triple<T>& lh, const triple<U>& rh ) \
{	return (lh.r OP rh.r) && (lh.g OP rh.g) && (lh.b OP rh.b); }

/** A is less than B if every color of A is less then the corresponding in B */
DEF_RELATIONAL_OPERATOR_FOR_TRIPLE(<)
/** A is greater than B if every color of A is greater then the corresponding in B */
DEF_RELATIONAL_OPERATOR_FOR_TRIPLE(>)
DEF_RELATIONAL_OPERATOR_FOR_TRIPLE(<=)
DEF_RELATIONAL_OPERATOR_FOR_TRIPLE(>=)
DEF_RELATIONAL_OPERATOR_FOR_TRIPLE(==)

/** A is not equal to be if any color of A is unqual to the corresponding one of B */
template<class T, class U>
inline bool operator!=( const triple<T>& lh, const triple<U>& rh )
{	return (lh.r != rh.r) || (lh.g != rh.g) || (lh.b != rh.b); }


/** max_elements returns a new triple with the maximum red, green and blue
 * value of the two arguments. */
template<class T>
inline triple<T> max_elements( const triple<T>& t, const triple<T>& u )
{	return triple<T>( std::max(t.r, u.r), std::max(t.g, u.g), std::max(t.b, u.b) ); }

/** min_elements returns a new triple with the minimum red, green and blue
 * value of the two arguments. */
template<class T>
inline triple<T> min_elements( const triple<T>& t, const triple<T>& u )
{	return triple<T>( std::min(t.r, u.r), std::min(t.g, u.g), std::min(t.b, u.b) ); }


template<class T>
inline std::ostream& operator<< ( std::ostream& os, const triple<T>& tr )
{
	return os << tr.r << '\t' << tr.g << '\t' << tr.b;
}

template<class T>
inline std::istream& operator>> ( std::istream& is, triple<T>& tr )
{
	return is >> tr.r >> tr.g >> tr.b;
}

template<class T>
inline triple<T> pow( const triple<T>& base, double exponent )
{	return triple<T>( std::pow(base.r, exponent), std::pow(base.g, exponent), std::pow(base.b, exponent) ); }
template<class T>
inline triple<T> atan2( const triple<T>& lh, const triple<T>& rh )
{	return triple<T>( std::atan2(lh.r, rh.r), std::atan2(lh.g, rh.g), std::atan2(lh.b, rh.b) ); }

#define DEF_EXT_OF_STD_FUNC_FOR_TRIPLE( func )\
template<class T> \
inline triple<T> func( const triple<T>& tr ) \
{	return triple<T>( std::func(tr.r), std::func(tr.g), std::func(tr.b) ); }

DEF_EXT_OF_STD_FUNC_FOR_TRIPLE(sqrt)
DEF_EXT_OF_STD_FUNC_FOR_TRIPLE(log)
DEF_EXT_OF_STD_FUNC_FOR_TRIPLE(exp)
DEF_EXT_OF_STD_FUNC_FOR_TRIPLE(abs)
DEF_EXT_OF_STD_FUNC_FOR_TRIPLE(tan)
DEF_EXT_OF_STD_FUNC_FOR_TRIPLE(sin)
DEF_EXT_OF_STD_FUNC_FOR_TRIPLE(cos)
DEF_EXT_OF_STD_FUNC_FOR_TRIPLE(atan)
DEF_EXT_OF_STD_FUNC_FOR_TRIPLE(asin)
DEF_EXT_OF_STD_FUNC_FOR_TRIPLE(acos)

#ifndef __VC__
#if defined __CBUILDER__
template<>
#endif
inline triple<float> abs( const triple<float>& tr )
	{ return triple<float>( std::fabs(tr.r), std::fabs(tr.g), std::fabs(tr.b) ); }
#if defined __CBUILDER__
template<>
#endif
inline triple<double> abs( const triple<double>& tr )
	{ return triple<double>( std::fabs(tr.r), std::fabs(tr.g), std::fabs(tr.b) ); }
#endif

//@}

/** These functional objects are used in the CPixelCorrectorFilter for
 * example. They should help making code duplication for all components
 * red, green, blue of a triple unnecessary. */
struct triple_r { template<class T> inline T& operator()(triple<T>& t) const { return t.r; } };
struct triple_g { template<class T> inline T& operator()(triple<T>& t) const { return t.g; } };
struct triple_b { template<class T> inline T& operator()(triple<T>& t) const { return t.b; } };

/** This is a special class for the white balance algorithm.
 * It saves the order of a triples values. */
class triple_order
{
private:
	typedef enum { red, green, blue } selector_t_;

	selector_t_ _first;
	selector_t_ _second;
	selector_t_ _third;

public:
	/** The default constructor sets the order red, green, blue. */
	triple_order() :
		_first(red), _second(green), _third(blue)
	{}

	/** This constructor simply calls get_order. */
	template<class T> triple_order( triple<T>& tr )
		{ get_order( tr ); }

	/** get_order examines the color order out of the passed triple and
	 * stores it. */
	template<class T> void get_order( triple<T>& tr )
	{
		if( tr.max_element() == tr.r )
		{
			_first = red;
			_second = green;
			_third = blue;
		}
		else if( tr.max_element() == tr.g )
		{
			_first = green;
			_second = red;
			_third = blue;
		}
		else
		{
			_first = blue;
			_second = green;
			_third = red;
		}
	}

	/** This method inverts the order depending on the value of the passed
	 * bool argument. */
	void invert( bool val = true )
		{ if( val ) std::swap( _first, _third ); }

	/** These methods return a reference to the stored color value of
	 * the passed triple.
	 * @name order access methods*/
	//@{
	template<class T> T& first( triple<T>& tr )
		{ return (_first == red)? tr.r : ((_first == green)? tr.g : tr.b); }
	template<class T> T& second( triple<T>& tr )
		{ return (_second == red)? tr.r : ((_second == green)? tr.g : tr.b); }
	template<class T> T& third( triple<T>& tr )
		{ return (_third == red)? tr.r : ((_third == green)? tr.g : tr.b); }
	//@}
};

/** @name triple types
 * These typedefs are just there to shorten the source code */
//@{
typedef triple<unsigned char>	uchar_triple;
typedef triple<unsigned short>	ushort_triple;
typedef triple<unsigned long>	ulong_triple;
typedef triple<short>			short_triple;
typedef triple<int>				int_triple;
typedef triple<long>			long_triple;
typedef triple<float>			float_triple;
typedef triple<double>			double_triple;
//@}


namespace std {

#ifndef __VC__

/** This is a specialization of the std::numeric_limits class for triples. */
template<class T>
class numeric_limits< triple<T> >
{
public:
	static triple<T> max()
	{
		return triple<T>( numeric_limits<T>::max() );
	}
};

#else

template<>
inline uchar_triple std::numeric_limits<uchar_triple>::max()
{	return uchar_triple(numeric_limits<unsigned char>::max()); }

template<>
inline ushort_triple std::numeric_limits<ushort_triple>::max()
{	return ushort_triple(numeric_limits<unsigned short>::max()); }

#endif /* __VC__ */

} /* namespace std */


#endif /* _TRIPLE_H_ */
