#include "eyebot.h"
#include "thread_adapter.h"
#include <algorithm>

void LCD::setCursor( int row, int col )
{
	if( row < 0 || row >= textBuf.height() ||
		col < 0 || col >= textBuf.width()-1 )
		throw EInvalidParameter(
			EMsgE() << "invalid cursor pos in LCD: ("
			<< row << ", " << col << ")" );
	r = row;
	c = col;
}

void LCD::clear( size_t row, size_t col )
{
	char* b = &textBuf( col, row );
	char* e = &textBuf( textBuf.width()-1, row );
	fill( b, e, ' ' );
	*e = '\n';

	dirtyText = true;
}

void LCD::incCursorRow()
{
	++r;
	c = 0;
	
	if( r == textBuf.height()-1 )
	{
		if( scrolling )
		{
			memmove( textBuf.begin(), textBuf.begin() + textBuf.width(),
					 textBuf.width() * (textBuf.height()-2) );
			clear( textBuf.height()-2, 0 );
			--r;
		}
		else
			r = 0;
	}
}

void LCD::putString( const char* s )
{
	char* b = &textBuf( c, r );

	while( *s != '\0' )
	{
		if( *s == '\n' )
		{
			clear( r, c );
			incCursorRow();
			b = &textBuf( c, r );
			++s;
		}
		else
		{
			*b = *s;

			++s;
			++c;
			++b;

			if( c == textBuf.width()-1 )
			{
				incCursorRow();
				b = &textBuf( c, r );
			}
		}
	}

	dirtyText = true;
}

void LCD::clear()
{
	for( size_t i = 0; i < textBuf.height(); ++i )
		clear( i, 0 );
	imageBuf.memset(255);
	dirtyImage = true;
	roi = uchar_pointxy(0);
}

void LCD::setMenu( int pos, const char* s )
{
	if( pos < 1 || pos > 4 )
		throw EInvalidParameter(
			EMsgE() << "Invalid pos in LCD::setMenu: " << pos );

	if( s == 0 || *s == '\0' )
		return;

	char* b = &textBuf( (pos-1) * menuWidth, textBuf.height()-1 );
	char* e = b + menuWidth;

	while( b != e && *s != '\0' && *s != '\n' && *s != '\t' )
	{
		*b = *s;
		++b;
		++s;
	}

	fill( b, e, ' ' );

	dirtyText = true;
}

string LCD::text()
{
	dirtyText = false;
	return string( textBuf.begin(), textBuf.size_in_bytes() );
}

void LCD::extendROI( size_t x, size_t y )
{
	if( roi.x < x )
		roi.x = x;
	if( roi.y < y )
		roi.y = y;
}

void LCD::putImage( robios::colimage* buf )
{
	for( size_t r = 0; r < imagerows; ++r )
		memcpy( &imageBuf(0, r), &(*buf)[r][0][0], imagecolumns*3 );
	extendROI( imagecolumns, imagerows );
	dirtyImage = true;
}

void LCD::putImage( robios::image* buf )
{
	for( size_t r = 0; r < imagerows; ++r )
		for( size_t c = 0; c < imagerows; ++c )
			imageBuf(c, r) = uchar_triple( (*buf)[r][c] );
	extendROI( imagecolumns, imagerows );
	dirtyImage = true;
}

void LCD::putImage( robios::BYTE* buf )
{
	for( size_t r = 0; r < 64; ++r )
		for( size_t c = 0; c < 128; ++c )
			imageBuf(c, r) = uchar_triple( *(buf + r*128 + c) );
	extendROI( 128, 64 );
	dirtyImage = true;
}

void LCD::setPixel( int x, int y, int c )
{
	if( (size_t)x >= imageBuf.width() || (size_t)y >= imageBuf.height() )
		throw EInvalidParameter(
			EMsgD() << "in LCD::setPixel - x: " << x << ", y: " << y );

	uchar_triple& t = imageBuf( x, y );
	switch( c )
	{
	case 0: t = uchar_triple(255); break;
	case 1: t = uchar_triple(0); break;
	case 2: t = uchar_triple(255) - t; break;
	default:
		throw EInvalidParameter( EMsgD() << "in LCD::setPixel - c: " << c );
	}

	extendROI( x+1, y+1 );
	dirtyImage = true;
}

struct LCDCursorBackup
{
	int old_r, old_c;
	LCD& lcd;

	LCDCursorBackup( LCD& lcd, int row, int col ) :
		lcd(lcd), old_r(lcd.r), old_c(lcd.c)
	{
		lcd.setCursor( row, col );
	}
	~LCDCursorBackup()
	{
		lcd.setCursor( old_r, old_c );
	}
};

namespace robios {

#define ENTER_ROBIOS_LCD \
	ENTER_ROBIOS; \

int LCDPrintf( const char* fmt, ... )
{
	ENTER_ROBIOS_LCD;

#if __GNUC__ == 2
	int count;
	char* s;
	va_list vargs;

	va_start( vargs, fmt );
	count = vasprintf( &s, fmt, vargs );
	va_end( vargs );

	try{ robi.lcd.putString( s ); }
	catch( ... ) { free( s ); throw; }
	free( s );
#endif

	return 0;
}

int LCDSetPrintf( int row, int col, const char* fmt, ... )
{
	ENTER_ROBIOS_LCD;
	LCDCursorBackup( robi.lcd, row, col );

#if __GNUC__ == 2
	int count;
	char* s;
	va_list vargs;

	va_start( vargs, fmt );
	count = vasprintf( &s, fmt, vargs );
	va_end( vargs );

	try{ robi.lcd.putString( s ); }
	catch( ... ) { free( s ); throw; }
	free( s );
#endif

	return 0;
}

int LCDClear( void )
{
	ENTER_ROBIOS_LCD;
	robi.lcd.clear();
	return 0;
}

int LCDPutChar( char c )
{
	ENTER_ROBIOS_LCD;
	robi.lcd.putString( string( &c, 1 ).c_str() );
	return 0;
}

int LCDSetChar( int row, int col, char c )
{
	ENTER_ROBIOS_LCD;
	LCDCursorBackup( robi.lcd, row, col );
	robi.lcd.putString( string( &c, 1 ).c_str() );
	return 0;
}

int LCDPutString( char* s )
{
	ENTER_ROBIOS_LCD;
	robi.lcd.putString( s );
	return 0;
}

int LCDSetString( int row, int col, char* s )
{
	ENTER_ROBIOS_LCD;
	LCDCursorBackup( robi.lcd, row, col );
	robi.lcd.putString( s );
	return 0;
}

int LCDMode( int mode )
{
	ENTER_ROBIOS_LCD;
	robi.lcd.scrolling = mode & SCROLLING;
	return 0;
}

int LCDGetPos( int* row, int* col )
{
	ENTER_ROBIOS_LCD;
	*row = robi.lcd.r;
	*col = robi.lcd.c;
	return 0;
}

int LCDSetPos( int row, int col )
{
	ENTER_ROBIOS_LCD;
	robi.lcd.setCursor( row, col );
	return 0;
}

int LCDMenu( char* s1, char* s2, char* s3, char* s4 )
{
	ENTER_ROBIOS_LCD;
	robi.lcd.setMenu( 1, s1 );
	robi.lcd.setMenu( 2, s2 );
	robi.lcd.setMenu( 3, s3 );
	robi.lcd.setMenu( 4, s4 );
	return 0;
}

int LCDMenuI( int pos, char* s )
{
	ENTER_ROBIOS_LCD;
	robi.lcd.setMenu( pos, s );
	return 0;
}

int LCDPutColorGraphic( colimage *buf )
{
	ENTER_ROBIOS_LCD;
	robi.lcd.putImage( buf );
	return 0;
}

int LCDPutGraphic( image* buf )
{
	ENTER_ROBIOS_LCD;
	robi.lcd.putImage( buf );
	return 0;
}

int LCDPutImage( BYTE* buf )
{
	ENTER_ROBIOS_LCD;
	robi.lcd.putImage( buf );
	return 0;
}

int LCDSetPixel( int row, int col, BOOL b )
{
	ENTER_ROBIOS_LCD;
	try { robi.lcd.setPixel( col, row, b ? 1 : 0 ); } catch(...) {}
	return 0;
}

int LCDInvertPixel( int row, int col )
{
	ENTER_ROBIOS_LCD;
	try { robi.lcd.setPixel( col, row, 2 ); } catch(...) {}
	return 0;
}

int LCDGetPixel( int row, int col )
{
	ENTER_ROBIOS_LCD;
	return robi.lcd.imageBuf( col, row ) == uchar_triple(0) ? 0 : 1;
}

int LCDLine(int x1, int y1, int x2, int y2, int c)
{
	ENTER_ROBIOS_LCD;

	if( y1 == y2 || x1 == x2 )
	{
		for( int x = x1; x <= x2; ++x )
			for( int y = y1; y <= y2; ++y )
				robi.lcd.setPixel( x, y, c );
	}
	else
	{
		int x = x1; int y = y1;      /* starting point */
		int error=0;
		int deltax = x2-x1;  /* difference */
		int deltay = y2-y1;
		int xchange, ychange;
		
		if (deltax < 0) { xchange = -1; deltax = -deltax; } else xchange = 1;
		if (deltay < 0) { ychange = -1; deltay = -deltay; } else ychange = 1;

		if (deltax < deltay)
		{
			for (int i=0; i < deltay+1; i++)
			{
				robi.lcd.setPixel( x, y, c );
				y     += ychange;
				error += deltax;
				if (error > deltay) { x += xchange; error -= deltay; }
			}
		}
		else
		{
			for (int i=0; i < deltax+1; i++)
			{
				robi.lcd.setPixel( x, y, c );
				x     += xchange;
				error += deltay;
				if (error > deltax) { y += ychange; error -= deltax; }
			}
		}
		robi.lcd.setPixel( x2, y2, c ); /* add last pixel to line */
	}

	return 0;
}

int LCDArea(int x1, int y1, int x2, int y2, int c)
{
	ENTER_ROBIOS_LCD;
	try {
	for( int x = x1; x <= x2; ++x )
		for( int y = y1; y <= y2; ++y )
			robi.lcd.setPixel( x, y, c );
	} catch(...) {}
	return 0;
}

} /* namespace robios */
