#include "eyebot.h"
#include "thread_adapter.h"

RADIOMessage::RADIOMessage( robios::BYTE rec, robios::BYTE snd,
							unsigned char* b, size_t s, const Time& t ) :
	receiver(rec), sender(snd), size(s), sendingTime(t)
{
	buffer = new unsigned char[size];
	memcpy( buffer, b, size );
}

void RADIOMessageQueue::deliver( const RADIOMessage& message )
{
	CEnterCriticalSection lock_watch( lock );
//BUG: TODO: insert the message at the write place, to preserve
//ascending order of receptionTimes
	messages.push( message );
}

int RADIOMessageQueue::check()
{
	CEnterCriticalSection lock_watch( lock );
	return messages.size();
}

void RADIOMessageQueue::receive( LocalRobi& robi, RADIOMessage& message )
{
	CEnterCriticalSection lock_watch( lock );

//BUG:
	/* this buisy loop implementation is basicaly the same than the
	 * keyboard wait method, if anything is wrong with this it should
	 * be fixed in both places. */
	if( messages.empty() )
	{
		if( robi.mtMode == NOTASK )
		{
			/* this is a buisy waiting loop, which is bad, of course
			 * but I am runnging out of time and this is a simpler
			 * solution than a "proper" one. */
			bool stop = false;
			do
			{
				/* increment the virtualTime to the next
				 * synchronization time and synchronize to this time */
				robi.virtualTime.steps( robi.virtualTime.steps() + 1 );
				
				lock_watch.leave();
				theCoreAdapter.synchronize( robi.virtualTime );
				robi.lastSynchronization.steps( robi.virtualTime.steps() );
				lock_watch.enter();

				/* To ensure, that we get the messages in the correct
				 * order, we actualy can only process messages with a
				 * receptionTime less than the last synchronization
				 * time. */
				stop = check() != 0 &&
					messages.front().receptionTime() < robi.virtualTime;

//				lock_watch.enter();
			}
			/* until a message was received */
			while( ! stop );

			message = messages.front();
			messages.pop();

			/* trick: set the current virtual time to the reception
			 * time of the received message, i.e. to a smaller value
			 * than we already synchronized to. */
			robi.virtualTime = message.receptionTime();

//BUG: it would be necessary to keep track of the timers properly, but
// as I said, I am short with time, and a problem only occurs for the case,
// that we have timers and no threads and blocking message receive calls...
		}
		else
		{
			/* in this case (multi-threading), we don't actively poll
			 * for messages, instead we let the waiting threads sleep
			 * in a similar way, as if they waited for a semaphore */
			
			lock_watch.leave();
			
			waitingThreads.push( *robi.currentThread );
			(*robi.currentThread)->semLocked = true;
			dbg_msg( "reschedule due to blocking message wait\n" );
			robi.reschedule();

			lock_watch.enter();

			message = messages.front();
			messages.pop();

//BUG: if the only currently executable thread blocks while waiting
// for a message, the main thread will be rescheduled, which is
// probably not what is supposed to happen.
		}
	}
	else
	{
		message = messages.front();
		messages.pop();
	}
}

void RADIOMessageQueue::processWaitingThreads( LocalRobi& robi )
{
//BUG: the order of the messages could get mixed up randomly inside
//one Time::delta time segment, depending on the system scheduler,
//since the messages get delivered instantly. This could possibly be
//improved in a similar way than for the non-mt part, but you know,
//the time...

	if( ! initialized || waitingThreads.empty() )
		return;

	CEnterCriticalSection lock_watch( lock );
	
	/* if we have a message and the current virtualTime is at least
	 * this message's receptionTime, we unlock the first waiting
	 * thread. */
	if( ! messages.empty() &&
		messages.front().receptionTime() <= robi.virtualTime )
	{
		waitingThreads.front()->semLocked = false;
		waitingThreads.pop();
		dbg_msg( "releasing thread, that waited for a message\n" );		
	}
}


#define ENTER_ROBIOS_RADIO \
	ENTER_ROBIOS; \

namespace robios {

int RADIOInit()
{
	ENTER_ROBIOS_RADIO;
	robi.radio.initialized = true;
	return 0;
}

int RADIOTerm()
{
	ENTER_ROBIOS_RADIO;
	robi.radio.initialized = false;
	return 0;
}

int RADIOCheck()
{
	ENTER_ROBIOS_RADIO;
	return robi.radio.check();
}

int RADIORecv(BYTE* id, int* bytesReceived, BYTE* buffer)
{
	ENTER_ROBIOS_RADIO;
	RADIOMessage m;
	robi.radio.receive( robi, m );
//BUG:
	*id = m.sender + 1;
	*bytesReceived = m.size;
	memcpy( buffer, m.buffer, m.size );
	delete [] m.buffer;
	return 0;
}

int RADIOSend(BYTE id, int byteCount, BYTE* buffer)
{
	ENTER_ROBIOS_RADIO;
//BUG:
	RADIOMessage m( id - 1, robi.id, buffer, byteCount,	robi.virtualTime );
	theCoreAdapter.sendRADIOMessage( m );
	return 0;
}

int RADIOSetIoctl(RadioIOParameters radioParams)
{
	ENTER_ROBIOS_RADIO;
	return 0;
}

int RADIOGetIoctl(RadioIOParameters* radioParams)
{
	ENTER_ROBIOS_RADIO;
	return 0;
}

int RADIOGetStatus(RadioStatus *s)
{
	ENTER_ROBIOS_RADIO;
	return 0;
}


} /* namespace radio */
