#ifndef _world_h_
#define _world_h_

#include "sim.h"

struct Segment
{
	double_pointxy p, q, n;
	double a;

	Segment() {}
	Segment( const double_pointxy& p, const double_pointxy& q ) : p(p), q(q)
		{ hesse(); }

	void hesse()
	{
		// calculate the hesse normal form for easy distance determination
		n.x = p.y - q.y;
		n.y = q.x - p.x;
		n /= n.norm();
		a = n * p;
	}

	inline double_pointxy center() const
		{ return (p+q)/2; }
	inline double_pointxy p2q() const
		{ return  q-p; }

	inline bool isInsideBoundingbox( const Position& pos, double r ) const
	{
		return !( max(p.x, q.x) + r < pos.x || pos.x < min(p.x, q.x) - r ||
				  max(p.y, q.y) + r < pos.y || pos.y < min(p.y, q.y) - r );
	}

	inline double distance( const double_pointxy& pos ) const
		{ return fabs( n * pos - a); }

	inline double signedDistance( const double_pointxy& pos ) const
		{ return n * pos - a; }

	inline bool intersecting( const Position& pos, double r ) const
		{ return isInsideBoundingbox( pos, r ) && distance( pos ) <= r;	}
};

inline istream& operator>>( istream& is, Segment& seg )
{ return is >> seg.p >> seg.q; }
inline ostream& operator<<( ostream& os, const Segment& seg )
{ return os << seg.p << " \t" << seg.q; }

/** this function checks if the two given segments intersect and
 * returns the distance of the intersection point from the point a.p
 * in proportion to the distance between a.p and a.q. if the segments
 * don't intersect, 1.0 is returned. */
double intersectionDistance( const Segment& a, const Segment& b );

struct OnWorldUpdateHandler
{
	virtual void onUpdate() = 0;
};

struct World
{
	double width, height;

	vector<Segment> segments;
	vector<GlobalRobi> robis;
	vector<Ball> balls;
	Segment boundingBox;
	OnWorldUpdateHandler* onUpdateHandler;

	World() : onUpdateHandler(0) {}

	void findBoundingBox();
	void readWorld( const char* fileName );
	void readMaze( const char* fileName );

	void update( const Time& newTime, const Time& lastTime );
};

extern World theWorld;

#endif /* _world_h_ */
