#ifdef __WIN32__
#include <windows.h>

#include "Synchronization.h"

/* CCriticalSection */

CCriticalSection::CCriticalSection( bool enter )
{
	impl_ = new CRITICAL_SECTION;
	InitializeCriticalSection( static_cast<CRITICAL_SECTION*>(impl_) );
	if( enter )
		this->enter();
}

CCriticalSection::~CCriticalSection()
{
	DeleteCriticalSection( static_cast<CRITICAL_SECTION*>(impl_) );
	delete impl_;
}

void CCriticalSection::enter()
{
	EnterCriticalSection( static_cast<CRITICAL_SECTION*>(impl_) );
}

void CCriticalSection::leave()
{
	LeaveCriticalSection( static_cast<CRITICAL_SECTION*>(impl_) );
}


/* CSemaphore */

CSemaphore::CSemaphore( long initial_value )
{
	//BUG: max count
	impl_ = CreateSemaphore( NULL, initial_value, 0xffff, NULL );
	if( impl_ == NULL )
		throw 3.1415;
}

CSemaphore::~CSemaphore()
{
	CloseHandle( impl_ );
}

bool CSemaphore::wait( unsigned long timeout )
{
	if( timeout == (unsigned long)-1 )
		timeout = INFINITE;

	DWORD res = WaitForSingleObject( impl_, timeout );
	if( res == WAIT_OBJECT_0 )
		return true;
	if( res == WAIT_TIMEOUT )
		return false;

	throw ESynchronization( EMsgD()
		<< "WaitForSingleObject failed: " << GetLastError() );
}

void CSemaphore::release()
{
	if( ! ReleaseSemaphore( impl_, 1, NULL ) )
		throw 3.1415;
}

bool CSemaphore::locked()
{
	bool l = WaitForSingleObject( impl_, 0 ) == WAIT_TIMEOUT;
	if( ! l )
		release();
	return l;
}

#endif /* WIN32 */

#if defined(__APPLE__) || defined(linux)

#include <pthread.h> /* pthread_mutex_... */
#include "Synchronization.h"

/* CCriticalSection */

CCriticalSection::CCriticalSection( bool enter )
{
//BUG: this hack was neccesary, because the appropriate mutexattr_..
// function was missing. This might not work under other systems than linux.
	pthread_mutex_t temp =
		/* PTHREAD_RECURSIVE_MUTEX_INITIALIZER_NP; */
		{0, 0, 0, PTHREAD_MUTEX_RECURSIVE_NP, __LOCK_INITIALIZER};
	impl_ = new pthread_mutex_t( temp );

	if( enter )
		this->enter();
}

CCriticalSection::~CCriticalSection()
{
	pthread_mutex_destroy( reinterpret_cast<pthread_mutex_t*>(impl_) );
	delete reinterpret_cast<pthread_mutex_t*>(impl_);
}

void CCriticalSection::enter()
{
	pthread_mutex_lock( reinterpret_cast<pthread_mutex_t*>(impl_) );
}

void CCriticalSection::leave()
{
	pthread_mutex_unlock( reinterpret_cast<pthread_mutex_t*>(impl_) );
}


/* CSemaphore */

#include <semaphore.h>
#include <errno.h>

void errnoCheck( int result )
{
	if( result )
		throw Eyae( EMsgE() << strerror(errno) );
}

CSemaphore::CSemaphore( long initial_value )
{
	impl_ = malloc(sizeof(sem_t));
	errnoCheck( sem_init( reinterpret_cast<sem_t*>(impl_), 0, initial_value ) );
}

CSemaphore::CSemaphore( const CSemaphore& )
{
	impl_ = malloc(sizeof(sem_t));
	errnoCheck( sem_init( reinterpret_cast<sem_t*>(impl_), 0, 0 ) );
}

CSemaphore::~CSemaphore()
{
	errnoCheck( sem_destroy( reinterpret_cast<sem_t*>(impl_) ) );
	free( impl_ );
}

bool CSemaphore::wait( unsigned long timeout )
{
	switch( timeout )
	{
	case  0 :
		return sem_trywait( reinterpret_cast<sem_t*>(impl_) ) == 0;
	case -1 :
		sem_wait( reinterpret_cast<sem_t*>(impl_) );
		return true;
	default:
		throw EInvalidParameter( EMsgD()
			<< "not timeout for linux semaphores supported" );
	}
}

void CSemaphore::release()
{
	errnoCheck( sem_post( reinterpret_cast<sem_t*>(impl_) ) );
}

bool CSemaphore::locked()
{
	int val;
	errnoCheck( sem_getvalue( reinterpret_cast<sem_t*>(impl_), &val ) );
	return val == 0;
}

#endif /* pthreads */


CMultiReadExclusiveWrite::CMultiReadExclusiveWrite()
	: counter_(0)
{}

void CMultiReadExclusiveWrite::enterRead()
{
	CEnterCriticalSection lock( lokal_ );
	if( counter_ == 0 )
		resource_.enter();
	counter_++;
}

void CMultiReadExclusiveWrite::leaveRead()
{
	CEnterCriticalSection lock( lokal_ );
	if( counter_ == 0 )
		throw ESynchronization();
	counter_--;
	if( counter_ == 0 )
		resource_.leave();
}

void CMultiReadExclusiveWrite::enterWrite()
{
	resource_.enter();
}

void CMultiReadExclusiveWrite::leaveWrite()
{
	resource_.leave();
}
