#ifndef _StringifiedException_H_
#define _StringifiedException_H_

#include <exception>
#include <sstream>
#include <string>

class EMsgBase
{
protected:
	std::ostringstream os_;

public:
	const char* c_str() const
		{ return const_cast<std::ostringstream&>(os_).str().c_str(); }
};

/** The exception message generated with a EMsgD
 * (ExceptionMessageDefault) object contains the default
 * message string plus the optional specified
 * message string. */
struct EMsgE : public EMsgBase
{
	template<class T> EMsgE& operator<<( const T& val )
		{ os_ << val; return *this;	}
};

/** The exception message generated with a EMsgE
 * (ExceptionMessageEmpty) object contains only the
 * specified message string. */
struct EMsgD : public EMsgBase
{
	template<class T> EMsgD& operator<<( const T& val )
		{ os_ << val; return *this; }
};

/** This is the base class for all stringified exceptions
 * that means exceptions wich contain a message string
 * specified at construction time of the exception.
 * Use the DEF_EXCEPTION and DEF_EXCEPTION_WDM makros
 * to define derived exception classes. */
class StringifiedException: public std::exception
{
protected:
	std::string msg_;

public:
	StringifiedException( const char* msg = 0 )
		{ if( msg ) msg_.assign( msg ); }
	StringifiedException( const EMsgBase& msg ) :
		msg_( msg.c_str() ) {}
	~StringifiedException() throw() {} /* only provided for g++-3.0 */

	const char* what() const throw()
		{ return msg_.c_str(); }
};

/* stringified exceptions without default message */
#define DEF_EXCEPTION( name, base )\
	class name : public base \
	{ \
	public: \
		name( const char* msg ) 		: base( msg ) {} \
		name( const std::string& msg ) 	: base( msg.c_str() ) {} \
		name( const EMsgE& msg ) 		: base( msg ) {} \
	};

/* stringified exceptions with default message (WDM) */
#define DEF_EXCEPTION_WDM( name, default_msg, base )\
	class name : public base \
	{ \
	public: \
		name( const char* msg = default_msg "." ) : base( msg ) {} \
		name( const std::string& msg ) : base( msg.c_str() ) {} \
		name( const EMsgE& msg ) : base( msg ) {} \
		name( const EMsgD& msg ) : \
			base( EMsgE() << default_msg << ": " << msg.c_str() << "." ) {} \
	};


/* Here are some commonly used exception classes */
DEF_EXCEPTION( Eyae, StringifiedException )
DEF_EXCEPTION_WDM( EInvalidParameter, "Invalid parameter", StringifiedException )
DEF_EXCEPTION_WDM( EIosFailure, "Io stream failure", StringifiedException )
DEF_EXCEPTION_WDM( EAccessViolation, "Access violation", StringifiedException );

#endif /* _StringifiedException_H_ */
