#ifndef _POINTXY_H_
#define _POINTXY_H_

#include <iostream>
#include <limits>
#include <algorithm>
#include <cmath>
#include <cstdlib> /* e.g. abs(int) */

template<class T>
struct pointxy
{
	typedef T value_type;

	T x, y;

	inline pointxy( const T& val = T() ): x(val), y(val) {};
	explicit inline pointxy( const T& x, const T& y ): x(x), y(y) {};
	template<class U> inline pointxy( const pointxy<U>& rh )
		{ x = T(rh.x); y = T(rh.y);  }

	/** index operator: 0 = x, 1 = y */
	inline T& operator[]( int index )
		{ return reinterpret_cast<T*>(this)[index]; }
	inline const T& operator[]( int index ) const
		{ return reinterpret_cast<const T*>(this)[index]; }

	/** @name assignment operators */
	//@{
   	template<class U> inline pointxy<T>& operator=( const pointxy<U>& rh )
		{ x = T(rh.x); y = T(rh.y); return *this; }

	inline pointxy<T>& operator*=( const T& rh )
		{ x *= rh; y *= rh; return *this; }
	inline pointxy<T>& operator/=( const T& rh )
		{ x /= rh; y /= rh; return *this; }
	template<class U> inline pointxy<T>& operator+=( const pointxy<U>& rh )
		{ x += rh.x; y += rh.y; return *this; }
	template<class U> inline pointxy<T>& operator-=( const pointxy<U>& rh )
		{ x -= rh.x; y -= rh.y; return *this; }

	inline pointxy<T>& operator<<=( const int& i )
		{ x <<= i; y <<= i; return *this; }
	inline pointxy<T>& operator>>=( const int& i )
		{ x >>= i; y >>= i; return *this; }
	//@}

	inline T norm() const { return sqrt(x*x + y*y); }
	inline T norm2() const { return x*x + y*y; }
	inline T phi() const { return std::atan2( x, y ); }
	inline pointxy<T> normalized() const { return *this / norm(); }
};

template<typename T>
inline pointxy<T> pointxy_from_polar( const T& r, const T& phi )
{ return pointxy<T>( r * std::cos( phi ), r * std::sin( phi ) ); }

template<typename T>
inline pointxy<T> rotate( const pointxy<T>& p, const T& angle )
{
	T s = std::sin( angle );
	T c = std::cos( angle );

	return pointxy<T>(
		c * p.x + -s * p.y,
		s * p.x +  c * p.y
	);
}

/** @name pointxy operators
 * These are the standard operators except min- and max_values. */
//@{

#define DEF_ARITHMETIC_OPERATOR_FOR_POINTXY( OP ) \
template<class T> \
inline pointxy<T> operator OP( const pointxy<T>& lh, const pointxy<T>& rh ) \
{	return pointxy<T>(lh.x OP rh.x, lh.y OP rh.y); }
DEF_ARITHMETIC_OPERATOR_FOR_POINTXY(+)
DEF_ARITHMETIC_OPERATOR_FOR_POINTXY(-)
//DEF_ARITHMETIC_OPERATOR_FOR_POINTXY(*)
//DEF_ARITHMETIC_OPERATOR_FOR_POINTXY(/)

/* the dot product */
template<class T>
inline T operator*( const pointxy<T>& lh, const pointxy<T>& rh )
{ return lh.x * rh.x + lh.y * rh.y; }

template<class T>
inline pointxy<T> operator*( const pointxy<T>& lh, const T& rh )
{ return pointxy<T>(lh.x * rh, lh.y * rh); }
template<class T>
inline pointxy<T> operator*( const T& lh, const pointxy<T>& rh )
{ return pointxy<T>(lh * rh.x, lh * rh.y); }
template<class T>
inline pointxy<T> operator/( const pointxy<T>& lh, const double& rh )
{ return pointxy<T>((double)lh.x / rh, (double)lh.y / rh); }

template<class T>
inline pointxy<T> operator<< ( const pointxy<T>& p, const int& i )
{ return pointxy<T>(p.x<<i, p.y<<i); }
template<class T>
inline pointxy<T> operator>> ( const pointxy<T>& p, const int& i )
{ return pointxy<T>(p.x>>i, p.y>>i); }


#define DEF_RELATIONAL_OPERATOR_FOR_POINTXY( OP ) \
template<class T, class U> \
inline bool operator OP( const pointxy<T>& lh, const pointxy<U>& rh ) \
{ return (lh.x OP rh.x) && (lh.y OP rh.y); }

DEF_RELATIONAL_OPERATOR_FOR_POINTXY(==)

template<class T, class U>
inline bool operator!=( const pointxy<T>& lh, const pointxy<U>& rh )
{ return (lh.x != rh.x) || (lh.y != rh.y); }


template<class T>
std::ostream& operator<< ( std::ostream& os, const pointxy<T>& p )
{ return os << p.x << " \t" << p.y; }

template<class T>
std::istream& operator>> ( std::istream& is, pointxy<T>& p )
{ return is >> p.x >> p.y; }


/** @name pointxy types
 * These typedefs are just there to shorten the source code */
//@{
typedef pointxy<unsigned char>	uchar_pointxy;
typedef pointxy<unsigned short>	ushort_pointxy;
typedef pointxy<unsigned long>	ulong_pointxy;
typedef pointxy<short>			short_pointxy;
typedef pointxy<int>			int_pointxy;
typedef pointxy<long>			long_pointxy;
typedef pointxy<float>			float_pointxy;
typedef pointxy<double>			double_pointxy;
//@}


namespace std {

#ifndef __VC__

/** This is a specialization of the std::numeric_limits class for pointxys. */
template<class T>
class numeric_limits< pointxy<T> >
{
public:
	static pointxy<T> max()
	{
		return pointxy<T>( numeric_limits<T>::max() );
	}
};

#else

template<>
inline uchar_pointxy std::numeric_limits<uchar_pointxy>::max()
{	return uchar_pointxy(numeric_limits<unsigned char>::max()); }

template<>
inline ushort_pointxy std::numeric_limits<ushort_pointxy>::max()
{	return ushort_pointxy(numeric_limits<unsigned short>::max()); }

#endif /* __VC__ */

} /* namespace std */


#endif /* _POINTXY_H_ */
