#include "eyebot.h"
#include "thread_adapter.h"
#include "world.h"
#include "ndrng.h"
#include "settings.h"

/* find the highest set bit, i.e. calc log2(h) */
size_t h2i( robios::PSDHandle h )
{
	for( int i = sizeof(h)*8-1; i >=0; --i )
		if( h & (1 << i) )
			return i;
	throw "invalid PSDHandle";
}

bool PSDs::invalidHandle( robios::PSDHandle h ) const
{
	if( h == 0 )
		return true;
	for(size_t i=0; i<psds.size(); ++i)
		if( h & (1<<i) && !psds[i].init )
			return true;
	if( h >= (1<<psds.size()) )
		return true;
	return false;
}


//BUG: what happens if we initialize one PSD several times?
//ANSWER: no errors or warning will occur, program continues to functino correctly
robios::PSDHandle PSDs::init( robios::DeviceSemantics s )
{
	for( size_t i=0; i < psds.size(); ++i )
		if( psds[i].devSem == s )
			if( psds[i].init)
				return 0;
			else
			{
				psds[i].init = true;
				return (1 << i);
			}

	dbg_msg( "unknown PSD DeviceSemantics: " << s << endl );
	return 0;
}

int PSDs::start( const Time& time, robios::PSDHandle h, bool cycle )
{
	if( invalidHandle(h) )
		return -1;

	if( busy( time ) )
	{
		dbg_msg( "busy psd (another measureing blocks driver)\n" );
		return 1;
	}	

	timeStamp.steps( time.steps() + 1 );

	for( size_t i=0; i < psds.size(); ++i )
		if( (1 << i) & h )
			psds[i].started = true;

	cycleMode = cycle;

	return 0;
}

void PSDs::stop()
{
	for( vector<PSD>::iterator i = psds.begin(); i != psds.end(); ++i )
		i->started = false;
}

bool PSDs::busy( const Time& time ) const
{
	if( !cycleMode )
		return time < timeStamp;

	for( vector<PSD>::const_iterator i = psds.begin(); i != psds.end(); ++i )
		if( i->started )
			return true;
	return false;
}

double calcPSD( const PSD& psd, id_t id )
{
	const World& world = theWorld;
	const GlobalRobi& robi = theWorld.robis[id];

	double_pointxy v = pointxy_from_polar(
		1.0, robi.pos.phi + psd.relPos.phi );
	double_pointxy p = robi.pos + rotate( psd.relPos, robi.pos.phi );
	Segment psd_seg( p, p + v * psd.maxReach );
	double min_dist = psd.maxReach;

	for( vector<Segment>::const_iterator i = world.segments.begin();
		 i != world.segments.end(); ++i )
	{
		double d = intersectionDistance( psd_seg, *i ) * psd.maxReach;
		if( d < min_dist )
			min_dist = d;
	}

	for( vector<GlobalRobi>::const_iterator i = world.robis.begin();
		 i != world.robis.end(); ++i )
	{
		if( &(*i) != &robi && psd_seg.intersecting( i->pos, i->radius ) )
		{
			Segment i_seg( i->pos - psd_seg.n * i->radius,
						   i->pos + psd_seg.n * i->radius );

			double d = intersectionDistance( psd_seg, i_seg ) * psd.maxReach;
			if( d < min_dist )
				min_dist = d;
		}
	}

	return min_dist;
}

int PSDs::get( const Time& time, id_t id, robios::PSDHandle h )
{
	if( h == 0 )
		/* convert from seconds to hundreth of a second */
		return int(time.fractional() * 100.); 

	if( invalidHandle( h ) || h & (h-1) )
		return -1;

	PSD& psd = psds[h2i(h)];

	if( ! psd.started )
	{
		dbg_msg( "PSD not started: " << h << endl );
		return -1;
	}

	if( time.steps() > timeStamp.steps() )
	{
		/* recalculate all psd values for this time step */
		for( vector<PSD>::iterator i = psds.begin(); i != psds.end(); ++i )
		{
			if( ! i->started )
				continue;

			double v = calcPSD( *i, id );
			
			v += generateNormalDistributedNumber( psdStandardDeviation );

			if( v >= psd.maxReach )
				i->cache = PSD_OUT_OF_RANGE;
			else
				i->cache = int(v * 1000.);
		}
		if( cycleMode )
			timeStamp.steps( time.steps() );
		else
			stop();
	}

	return psd.cache;
}

bool PSDs::check( const Time& time ) const
{
	return cycleMode ? timeStamp <= time && busy(time) : timeStamp <= time;
}


#define ENTER_ROBIOS_PSD \
	ENTER_ROBIOS; \

namespace robios {

PSDHandle PSDInit(DeviceSemantics s)
{
	ENTER_ROBIOS_PSD;
	return robi.psds.init( s );
}

int PSDRelease()
{
	ENTER_ROBIOS_PSD;
	return 0;
}

int PSDStart(PSDHandle h, BOOL cycle)
{
	ENTER_ROBIOS_PSD;
	return robi.psds.start( robi.virtualTime, h, cycle );
}

int PSDStop()
{
	ENTER_ROBIOS_PSD;
	robi.psds.stop();
	return 0;
}

BOOL PSDCheck()
{
	ENTER_ROBIOS_PSD;
	return robi.psds.check( robi.virtualTime );
}

int PSDGet(PSDHandle h)
{
	ENTER_ROBIOS_PSD;
	return robi.psds.get( robi.virtualTime, robi.id, h );
}

int PSDGetRaw(PSDHandle h)
{
	ENTER_ROBIOS_PSD;
	return robi.psds.get( robi.virtualTime, robi.id, h );
}


} /* namespace robios */
