#include "world.h"
#include "parser.h"
#include "thread_adapter.h" /* for the collision report message */

#include <deque>
#include <fstream>
#include <strstream> //BUG: could be replaced by (non-obsolete) sstream
#include <cmath>
#include "libutil/string_helpers.h"
#include "settings.h"

#ifdef __STL_USE_NAMESPACE_FOR_RELOPS
using namespace std::rel_ops;
#endif

typedef vector<GlobalRobi>::iterator ri_t;

/** change unit from mm to m and from deg to rad */
inline Position normalize_pos( const Position& pos )
{
	return Position( pos.x / 1000.0, pos.y / 1000.0,
					 mod2pi( pos.phi * M_PI / 180.0 ) );
}
/** change unit from mm to m */
inline double_pointxy normalize_pos( const double_pointxy& pos )
{
	return pos / 1000.0;
}

double intersectionDistance( const Segment& a, const Segment& b )
{
	/* if both points of a are on the same side of b, then they can
	 * not intersect. if they intersect on an end-point, we return
	 * 1 as well. 1 means "no intersection". */
	if( a.signedDistance( b.p ) * a.signedDistance( b.q ) >= 0. )
		return 1.0;

	double dp = b.signedDistance( a.p );
	double dq = b.signedDistance( a.q );

	/* see obove. */
	if( dp * dq >= 0. )
		return 1.0;

	dp = fabs( dp );
	dq = fabs( dq );

	/* d gets the ratio of dp and dq+dp, that is the distance of the
	 * intersection point from a.p as proportion of the distance
	 * between a.p and a.q. */
	return dp / (dp + dq);
}

/** this version calculates the intersection distance even if the intersection
 * lies outside the segment, doumentation see intersectionDistance */
double intersectionDistanceOfLines( const Segment& a, const Segment& b )
{
	double dp = fabs( b.signedDistance( a.p ) );
	double dq = fabs( b.signedDistance( a.q ) );

	return dp / (dp + dq);
}

void Ball::setState( const double_pointxy& pos, const double_pointxy& v,
					 const Time& t )
{
	startTime = t;
	startPosition = pos;
	double v_norm = v.norm();
	if( v_norm <= thresholdVelocity )
	{
		stopTime = startTime;
		startVelocity = double_pointxy( 0, 0 );
		deceleration = double_pointxy( 0, 0 );
	}
	else
	{
		stopTime = (thresholdVelocity - v_norm) / -friction
			+ startTime;
		startVelocity = v;
		deceleration = -friction * v / v_norm;
	}
}

double_pointxy Ball::calcPosition( const Time& currentTime ) const
{
	double t = min( currentTime, stopTime ) - startTime;
	return .5 * deceleration * t*t + startVelocity * t + startPosition;
}

double_pointxy Ball::calcVelocity( const Time& currentTime ) const
{
	if( currentTime >= stopTime )
		return double_pointxy(0.,0.);
	else
		return deceleration * (currentTime - startTime) + startVelocity;
}

/** this class represents a collision of a Ball with a Segment. the
 * method set sets up the collision parameters and the method collide
 * performs the collision by updating the current state of the ball. */
struct Collision
{
	Segment segment;
	double_pointxy segmentVelocity;
	/* the coefficients describing the way that the velocity gets modified:
	 * with an ideal collision the velocity component parallel to the
	 * segment is not changed, the component normal to the segment
	 * gets inverted. the base transformation projects the parallel
	 * component to the first and the normal component to the second
	 * component of the new vector. */
	double_pointxy coefficients;
	double distance;

	Collision() : distance(1.) {}
	inline operator bool() { return distance < 1.0; } const
	inline void reset() { distance = 1.0; }

	void set( const Segment& s, double d,
			  const double_pointxy& c,
			  const double_pointxy& sv = double_pointxy() )
	{ segment = s; distance = d; coefficients = c; segmentVelocity = sv; }

	double checkCollisionWithInnerSegment(
		const Ball& ball, const Segment& testSeg,
		const double_pointxy& newPos ) const
	{ 
		/* vector pointing from the ball to the segment */
		double_pointxy b2s = testSeg.n 
			* -(double)signum(testSeg.signedDistance(ball.pos));
		/* if move away from the wall, we can not collide */
		if( (newPos - ball.pos) * b2s < 0 )
			return 1.;
		/* find the point where the ball would collides with the inner
		 * segment, relative to the ball center (ball collision point). */
		double_pointxy bcp = b2s * ball.radius;
	
		/* line describes the line that the bcp moves at, the intersection
		 * of line and the segment testSeg is the point of the
		 * collision */
		Segment line( ball.pos + bcp, newPos + bcp );

		return intersectionDistance( line, testSeg );
	}

	double checkCollisionWithPoint(
		const Ball& ball, const double_pointxy& testPoint,
		const double_pointxy& newPos, Segment& virtualSeg )
	{
		Segment center(ball.pos, newPos);
		double bcpd = center.signedDistance( testPoint );
		if( fabs(bcpd) >= ball.radius )
			return 1.;

		double alpha = std::asin( bcpd / ball.radius );
		double_pointxy new_seg_vec =
			rotate( center.p2q(), alpha + M_PI/2. ).normalized();
		
		virtualSeg = Segment( testPoint + new_seg_vec,
							  testPoint - new_seg_vec );
		return checkCollisionWithInnerSegment( ball, virtualSeg, newPos );
	}

	void magic( const Ball& ball, const Segment& testSeg,
				const double_pointxy& newPos,
				const double_pointxy& c,
				const double_pointxy& sv = double_pointxy() )
	{
		if( ball.pos == newPos )
			throw Eyae( "internal error: magic on non moving ball" );

		/* first we check, if the ball collides the inner segment */
		double d = checkCollisionWithInnerSegment( ball, testSeg, newPos );
		if( d < 1. )
		{
			if( d < distance )
				set( testSeg, d, c, sv );
			return;
		}

		Segment virtualSeg_p;
		Segment virtualSeg_q;

		double dp = checkCollisionWithPoint( ball, testSeg.p, newPos,
											 virtualSeg_p );
		double dq = checkCollisionWithPoint( ball, testSeg.q, newPos,
											 virtualSeg_q );
		d = std::min( dp, dq );
		if( d < distance )
		{
			Segment& virtualSeg = dp < dq ? virtualSeg_p : virtualSeg_q;

			/* attention: I had to make this workaround here, if
			 * you want to make it properly, just rewrite the
			 * whole ball collision alrogithm */

			/* vector pointing from the ball to the segment */
			double_pointxy b2s = virtualSeg.n 
				* -(double)signum(virtualSeg.signedDistance(ball.pos));
			double x = b2s * sv;
			/* if the ball hits the robi on the back, we set the
			 * segment speed to zero. */
			set( virtualSeg, d, c, x < 0 ? sv : double_pointxy(0.,0.) );
		}
	}

	double_pointxy mirrorVelocity( const double_pointxy& old_v ) const
	{
//		dbg_msg( "old_v: " << old_v << endl );

		 /* first get the relative velocity of the ball to the segment,
		  * then perform a collision between a ball and a wall of
		  * infinite mass, i.e. a reflection by using the collision
		  * coefficients at last add the segment velocity again. */
		double_pointxy v = old_v - segmentVelocity;

		/* perform a base transformation to a base normal to the segment */
		/* parallel component */
		double_pointxy n1 = segment.p2q().normalized();
		/* normal component */
		double_pointxy n2 = segment.n;
		double_pointxy v_n( n1*v, n2*v );

		/* reflection */
		double_pointxy v_n_reflected( v_n.x * coefficients.x,
									  v_n.y * coefficients.y );

		/* back transformation */
		v = n1 * v_n_reflected.x + n2 * v_n_reflected.y;

		v += segmentVelocity;

//		dbg_msg( "new_v: " << v << endl );
		return v;
	}

	void collide( Ball& ball, const Time& newTime ) const
	{
		/* this is to make the process numerical stable, we have to
		 * prevent to get a second collision, with the segment, we just
		 * perform the collision. */
//		double distance = this->distance - 1e-10;

		/* use a linear approximization for the time of the collision */
		Time toc = distance * newTime + (1-distance) * ball.time;
		/* poc is the position of the ball center at the collision */
		double_pointxy poc = (1-distance) * ball.pos
			+ distance * ball.calcPosition(newTime);

		ball.setState( poc, mirrorVelocity( ball.calcVelocity(toc) ), toc );

		/* move the ball to the point and time of the collision */
		ball.pos = poc;
		ball.time = toc;
	}
};

struct Intersection
{
	virtual bool test( const GlobalRobi& robi ) = 0;
};

struct RobiIntersection : public Intersection
{
	const GlobalRobi& other;
	RobiIntersection( const GlobalRobi& other ) : other(other) {}
	bool test( const GlobalRobi& robi )
		{ return GlobalRobi::intersecting( robi, other ); }
};

struct SegmentIntersection : public Intersection
{
	const Segment& segment;
	SegmentIntersection( const Segment& segment ) : segment(segment) {}
	bool test( const GlobalRobi& robi )
		{ return segment.intersecting( robi.pos, robi.radius ); }
};

Intersection* intersecting( ri_t robi )
{
//BUG: don't find the first intersection but the closest
	for( vector<Segment>::iterator i = theWorld.segments.begin();
		 i != theWorld.segments.end(); ++i )
	{
		if( i->intersecting( robi->pos, robi->radius ) )
			return new SegmentIntersection( *i );
	}

	for( ri_t i = theWorld.robis.begin(); i != theWorld.robis.end(); ++i )
	{
		if( i != robi && GlobalRobi::intersecting( *robi, *i ) )
			return new RobiIntersection( *i );
	}

	return 0;
}

void World::update( const Time& newTime, const Time& lastTime )
{
	/* update is performed as follows:
	 *
	 * 1) move the balls, if a collision occurs, handle it,
	 * i.e. update the state of the ball. the balls don't interact
	 * with each other (not implemented yet). after this step, the
	 * balls are moved to their position at newTime.
	 *
	 * 2) move the robots one after an other. if a robot collides with
	 * a wall or an other robot then stop it. if it collides with a
	 * ball then update the balls state.
	 *
	 * the collisions between robots and balls in the first state
	 * happen between lastTime and newTime (linear interpolation) but
	 * the collisions in step 2 happens at newTime.
	 *
	 * in terms of collisions with balls, the robis are no circles
	 * anymore, but squares, that just fit in the usual circle.
	 *
	 * //BUG: the complete ball handling is not bug free in terms of
	 * the used physical model. there might be "complex" situations
	 * with several robots and a ball, where after a collision between
	 * a ball and a robi, the ball end up inside an other robi. or
	 * when a robi pushes a ball against the a wall segment or another
	 * robi.
	 */

	/* update the ball positions */
	for( vector<Ball>::iterator ball = balls.begin();
		 ball != balls.end(); ++ball )
	{
		double_pointxy new_pos;
		Collision c;

		do
		{
			c.reset();
			new_pos = ball->calcPosition( newTime );

			/* if the ball did not move, there is no collision */
			if( new_pos == ball->pos )
				break;

			/* find the segment that the ball collides with, that is,
			 * if any, the "first" of all that it would collide
			 * with. */
			for( vector<Segment>::iterator i = theWorld.segments.begin();
				 i != theWorld.segments.end(); ++i )
			{
				c.magic( *ball, *i, new_pos, Ball::wallReflectionCoefficients );
			}

			/* for the robots we don't check for the nearest one, we
			 * just take the first one. */
			for( ri_t i = theWorld.robis.begin(); i != theWorld.robis.end(); ++i )
			{
				/* vector pointing towards the current front direction */
				double_pointxy front =
					pointxy_from_polar( i->radius * M_SQRT1_2, i->pos.phi );
				/* vector pointing to the left (normal to the front vector) */
				double_pointxy left = rotate( front, M_PI / 2. );

				/* these segments build a bounding box around the
				 * robot and will be the boundaries for the ball
				 * manipulation */
				Segment f( i->pos + front + left, i->pos + front - left );
				Segment b( i->pos - front + left, i->pos - front - left );
				Segment l( i->pos + front + left, i->pos - front + left );
				Segment r( i->pos + front - left, i->pos - front - left );

				double_pointxy robi_v = pointxy_from_polar(
					i->vwController.v, i->pos.phi );

				c.magic( *ball, f, new_pos, i->kickerReflectionCoefficients, pointxy_from_polar( i->vwController.v + i->kickerSpeed, i->pos.phi ) );
				c.magic( *ball, b, new_pos, Ball::robiReflectionCoefficients, robi_v );
				c.magic( *ball, l, new_pos, Ball::robiReflectionCoefficients, robi_v );
				c.magic( *ball, r, new_pos, Ball::robiReflectionCoefficients, robi_v );
			}

			/* update the position depending on if there was a
			 * collision or not. */
			if( c )
				c.collide( *ball, newTime );
		}
		while( c );

		ball->pos = new_pos;
		ball->time = newTime;
	}

	/* update the robi positions */
	for( ri_t i = robis.begin(); i != robis.end(); ++i )
	{
		Position old_pos = i->pos;
		i->pos = i->vwController.calcPos( newTime );

		/* store the old v value in case it gets reset because the
		 * robi collides with a wall or an other robi. we need this
		 * value for the collisionhandling with a ball. */
		double old_velocity = i->vwController.v;

//BUG: TODO: checking if vehicle was to fast, that it tunneled the
//obstacle, i.e. the driven way was more than it's diameter.

		Intersection* intersection = intersecting( i );
		if( intersection )
		{
			/* this is a binary search like algorithm to find a
			 * position with no collision, but near to the obstacle */
			Time interTime = newTime;
			Time noninterTime = lastTime;
			Position interPos = i->pos;
			Position noninterPos = old_pos;

			while( (interPos - noninterPos).norm2() >
				   collisionDistanceAccuracy * collisionDistanceAccuracy )
			{
				Time t = (interTime + noninterTime) / 2.0;
				i->pos = i->vwController.calcPos( t );
				if( intersection->test( *i ) )
				{
					interTime = t;
					interPos = i->pos;
				}
				else
				{
					noninterTime = t;
					noninterPos = i->pos;
				}
			}
			
			dbg_msg( "c " );
			/* update the current position for new robi */
			i->pos = noninterPos;
			/* "stop" the robis movement */
			i->vwController.reset( noninterPos );
//BUG:
			theCoreAdapter.reportCollision( i - robis.begin(), noninterTime );

			delete intersection;
		}

		/* check for a collision with a ball */
		/* vector pointing towards the current front direction */
		double_pointxy front =
			pointxy_from_polar( i->radius * M_SQRT1_2, i->pos.phi );
		/* vector pointing to the left (normal to the front vector) */
		double_pointxy left = rotate( front, M_PI / 2. );

		/* generate a segment for the collision, this represents
		 * the front or the back of the robi, depending on it's
		 * moving direction (positiv or negative velocity) */
		Segment s( i->pos + (double)signum(old_velocity) * front + left, 
				   i->pos + (double)signum(old_velocity) * front - left );

		double_pointxy robi_movement = i->pos - old_pos;

		for( vector<Ball>::iterator ball = balls.begin();
			 ball != balls.end(); ++ball )
		{
			if( robi_movement.norm2() != 0.0 )
			{
				double_pointxy pseudo_new_pos = ball->pos;
				ball->pos += robi_movement;

				Collision c;
				if( signum(old_velocity) > 0 )
					/* front segment is the kicker */
					c.magic( *ball, s, pseudo_new_pos,
							 i->kickerReflectionCoefficients,
							 robi_movement / (newTime-lastTime)
							 + pointxy_from_polar(i->kickerSpeed, i->pos.phi) );
				else
					c.magic( *ball, s, pseudo_new_pos,
							 Ball::robiReflectionCoefficients,
							 robi_movement / (newTime-lastTime) );

				ball->pos = pseudo_new_pos;

				if( c )
				{
					/* move the ball "out of the robi" */
//BUG: has to check, if the ball collides again, which would mean, the robi
//pushed the ball against the wall or another robi -> robi has be stoped
					ball->setState(
						ball->pos + robi_movement.normalized()
						* (s.distance(ball->pos) + ball->radius),
						c.mirrorVelocity( ball->calcVelocity(newTime) ),
						newTime );
					
					ball->pos = ball->startPosition;
					ball->time = ball->startTime;
				}
			}
			
//BUG:
			/* we make shure the ball does not intersect with the robi
			 * after the movment of the robi, this is a workaround for
			 * the non-existance of a proper modeling of the w-component
			 * of the robis velocity during collisions. */
			double_pointxy r2b = ball->pos - i->pos;
			double r2b_norm = r2b.norm();
			if( r2b_norm < i->radius + ball->radius )
			{
				/* if the ball is not outside the robi circle, we
				 * check if it is inside the collision sqare and move
				 * it outside, if necessary */
				double angle_to_normal = drem( r2b.phi() - i->pos.phi, M_PI/2 );
				double min_r = (i->radius * M_SQRT1_2 + ball->radius)
					/ std::cos(angle_to_normal);

				if( r2b_norm < min_r + 1e-5 )
				{
					r2b *= (min_r - r2b_norm) / r2b_norm;
					ball->setState(
						ball->pos + r2b, ball->calcVelocity(newTime), newTime );
					
					ball->pos = ball->startPosition;
					ball->time = ball->startTime;

					dbg_msg( "spirited the ball out of the robi by "
							 << r2b_norm - min_r << " m." << endl );
				}
			}
		}
	}

	/* the diplay has to be updated actively here because of the robi
	 * cameras, so we can not use timer-based update (like for the
	 * QConsole update) */
	if( onUpdateHandler )
		onUpdateHandler->onUpdate();

#ifdef DEBUG_noe
	// to show, that the core is still running, even if the display
	// stopped, this is to find the origin of this bug
	static int c = 0;
	if( ++c == 20 )
	{
		c = 0;
		dbg_msg( '.' );
	}
#endif /* DEBUG */

}

Position VWController::calcPos( const Time& time ) const
{
	/* get the complete execution time of the current command
	 * (isNull if no command is running -> return startPos) */
	voodo d = duration();

	/* the used time for evaluation is either the duration of the
	 * of the drive command or the elapsed time since start of the
	 * command, wichever is less. */
	double t = d.isInfinite() ? time - startTime :
		min( (double)d, time - startTime );
	
	Position p = startPos;

	if( axisOffset != 0.0 )
		p += pointxy_from_polar( axisOffset, p.phi );

	if( v == 0.0 )
	{
		p.phi = mod2pi( p.phi + t * w );
	}
	else if( w == 0.0 )
	{
		p = p + pointxy_from_polar( 1.0, startPos.phi ) * v * t;
	}
	else
	{
		/* The following code is written with the assumption that w > 0
		 * means left turn. As one can see there is no definition by case
		 * for the sign of w. I just walked throw the tow cases and
		 * convinced myself, that this should work for both
		 * cases. Consider, that r can be negativ. */

		double r = fabs(v) / w;
		// normal vector pointing to the center of the rotation
		double_pointxy n = pointxy_from_polar( 1.0, startPos.phi + M_PI/2.0 );
		// center of rotation
		double_pointxy o = p + n * r;
		// resulting position relative to the origin
		double_pointxy q = pointxy_from_polar( r, startPos.phi 
											   - M_PI/2.0 + w * signum(v) * t );
		// adust origin
		p = o + q;
		// adjust phi
		p.phi = mod2pi( p.phi + w * signum(v) * t );
	}

	if( axisOffset != 0.0 )
		p -= pointxy_from_polar( axisOffset, p.phi );
	
	return p;
}

void VWController::update( const VWData& vw )
{
	/* calculate the current position based on the old state as new
	 * startPos */
	startPos = calcPos( vw.startTime );

	/* set the new state, including startTime, v, w and distance */
	VWData::operator=( vw );
}

voodo VWController::remainingDistance( const Time& time ) const
{
	if( done( time ) )
		return 0;
	if( distance.isInfinite() )
		return voodo();
	return distance * (time - startTime) / duration();
}

voodo VWController::duration() const
{
	if( stopped() )
		return 0.0;
	if( distance.isInfinite() )
		return voodo();
	return fabs( distance / (v != 0.0 ? v : w) );
}

void World::readWorld( const char* fileName )
{
	robis.clear();
	balls.clear();
	segments.clear();

	std::deque<Position> stack;

	ifstream f( fileName );
	if( ! f )
		throw EIosFailure( EMsgD() << "can not open world file: " << fileName );

	while( ! f.eof() )
	{
		string s;
		getline( f, s );
		s = trim( s );
		if( s.empty() || s[0] == ';' )
			continue;

		if( starts_with(s, "width") )
			width = str_to<double>( s ) / 1000.0;
		else if( starts_with(s, "height") )
			height = str_to<double>( s ) / 1000.0;
		else if( starts_with(s, "push") || starts_with(s, "change") )
			stack.push_back( normalize_pos(str_to<Position>(s)) );
		else if( starts_with(s, "pop") )
			stack.pop_back();
		else if( starts_with(s, "position") )
			robis.push_back( GlobalRobi(normalize_pos(str_to<Position>(s))) );
		else if( starts_with(s, "ball") )
			balls.push_back( Ball(normalize_pos(str_to<double_pointxy>(s))) );
		else if( isdigit(s[0]) || s[0] == '-' )
		{
			Segment seg = str_to<Segment>(s);
			/* adjust unit from mm to m */
			seg.p /= 1000.0;
			seg.q /= 1000.0;

			/* perform all transformations stored in the current
			 * stack */
			for( deque<Position>::reverse_iterator i = stack.rbegin();
				 i != stack.rend(); ++i )
			{
				seg.p = *i + rotate( seg.p, i->phi );
				seg.q = *i + rotate( seg.q, i->phi );
			}
				
			/* initialize the hesse normal form data. */
			seg.hesse();
			segments.push_back( seg );
		}
	}

	findBoundingBox();
}

void World::findBoundingBox()
{
	vector<Segment>::const_iterator i = segments.begin();
	boundingBox = *i;
	for( ; i != segments.end(); ++i )
	{
		boundingBox.p.x =
			std::min( boundingBox.p.x, std::min( i->p.x, i->q.x ) );
		boundingBox.p.y =
			std::min( boundingBox.p.y, std::min( i->p.y, i->q.y ) );
		boundingBox.q.x =
			std::max( boundingBox.q.x, std::max( i->p.x, i->q.x ) );
		boundingBox.q.y =
			std::max( boundingBox.q.y, std::max( i->p.y, i->q.y ) );
	}

#ifdef DEBUG
	cout << "width: " << width << ", height: " << height << endl
		 << "robis:" << endl;
	copy( robis.begin(), robis.end(),
		  ostream_iterator<GlobalRobi>( cout, "\n") );
	cout << "balls:" << endl;
	copy( balls.begin(), balls.end(),
		  ostream_iterator<Ball>( cout, "\n" ) );
	cout << "segments:" << endl;
	copy( segments.begin(), segments.end(),
		  ostream_iterator<Segment>( cout, "\n" ) );
#endif /* DEBUG */
}

void LocalRobi::readDescription( const char* fileName )
{
	ifstream f( fileName );
	if( ! f )
		throw EIosFailure(
			EMsgD() << "can not open robi description file: " << fileName );

	while( ! f.eof() )
	{
		string s;
		getline( f, s );
		s = trim( s );
		if( s.empty() || s[0] == '#' )
			continue;

		if( starts_with(s, "program") )
			program = trim(s);
		else if( starts_with(s, "name") )
			name = trim(s);
		else if( starts_with(s, "model") )
			model = trim(s);
		else if( starts_with(s, "psd") )
		{
			string name;
			istringstream iss(s);
			PSD psd;
			iss >> name >> psd.devSem >> psd.relPos;
			psd.relPos = normalize_pos( psd.relPos );
			psdPositions.push_back( psd.relPos );
			psds.psds.push_back( psd );
		}
		else if( starts_with(s, "camera") )
		{
			string name;
			istringstream iss(s);
			Camera cam;
			iss >> cam.relPos >> cam.relPosZ >> cam.pan >> cam.tilt
				>> cam.maxImageSize;
			cam.relPos = normalize_pos( cam.relPos );
			cam.relPosZ = cam.relPosZ / 1000.;
			cam.relPan = mod2pi( cam.relPan * M_PI / 180.0 );
			cam.relTilt = mod2pi( cam.relTilt * M_PI / 180.0 );
			cameras.push_back( cam );
		}
		else if( starts_with(s, "diameter") )
			radius = str_to<int>(s) / 1000. / 2;
		else if( starts_with(s, "speed") )
			maxV = str_to<int>(s) / 1000.;
		else if( starts_with(s, "turn") )
			maxW = str_to<int>(s) / 180. * M_PI;
		else if( starts_with(s, "axis") )
			axisOffset = str_to<int>(s) / 1000.;
	}
}

void World::readMaze(const char* fileName)
{
	// This function has been implemented by Jason Foo.

	robis.clear();
	balls.clear();
	segments.clear();

	// Import the STL namespace into just this function.
	using namespace std;

	// The unformatted maze file will be read into a vector of ASCII
	// strings.
	vector<string> asc;

	// The maze file contains a constant called "size" with the
	// following default value.
	double size = 0.180;

	// Open maze file for reading.  Throw an exception if an error
	// occurs.
	ifstream f(fileName);
	if(!f)
		throw EIosFailure( EMsgD() << "can not open maze file: " << fileName );

	// Read the maze file line-by-line.  Each line may either contain a
	// single numeric value, which defines the size constant for the
	// whole file, or a row of the maze description.  The variable "tmp"
	// contains a copy of the current line.  The variable "w" contains
	// the length of the longest line encountered so far (the width of
	// the maze).  The variable "h" containas the number of lines
	// encountered so far (the height of the maze).
	string tmp;
	int w = 0, h = 0;
	while(getline(f, tmp, '\n'))
	{
		// Try to parse an integer from the start of the line.
		istrstream ss(tmp.c_str());
		int new_size;
		ss >> new_size;
		if(ss.good())
		{
			// If the line begins with an integer, then replace the current
			// value of "size" with the value of the integer.
			size = (double)new_size / 1000;
		}
		else
		{
			// Otherwise, treat the line as being the next row of the maze.
			// For now, just read the line into hthe vector of strings
			// called "asc".
			asc.push_back(tmp);
			// Remember the length of the longest line so far, in a variable
			// called "w".
			int s = tmp.length();
			if(s > w) w = s;
			// Remember the number of lines encountered so far.
			h++;
		}
	}

	// Panic if the width is less than 3 or the height is less than 1.
	// These widths and heights are in characters, not maze squares.
	if(w < 3 || h < 1)
	{
		throw EIosFailure( EMsgD() << "invalid maze file: " << fileName );
	}

	// Axel: this is to add at least one ' ' to the right, so that the
	// drawing of horizontal lines finds a "stop" character before
	// j >= w
	w += 1;

	// Pad the right-hand end of each maze row string with spaces, so
	// that they are all the same width as the current value of "w".
	for(int i = 0; i < h; i++)
	{
		asc[i] = asc[i] + string(w - asc[i].length(), ' ');
	}

	// This is the point where my algorithm differs from the
	// implementation in EyeSim version 3.2.  My algorithm scans through
	// the odd-numbered columns in row-major order, generating the
	// horizontal line segments and robot starting positions.  It then
	// scans through the even-numbered columns in column-major order,
	// generating the vertical line segments.

	// Loop through the odd-numbered columns in row-major order,
	// starting with the bottom row.  We can assume that the i >= 1 and
	// j >= 3 because this has already been tested for.
	for(int i = h - 1; i >= 0; i--)
	{
		// These variables are used to construct a single line segment.
		bool drawing = false;
		int s;

		for(int j = 1; j < w; j += 2)
		{
			// Test the character at position (i, j) for horizontal line
			// segments.  This switch also includes the invalid character
			// test.
			switch(asc[i][j])
			{
			case '_':
			case 'U':
			case 'D':
			case 'L':
			case 'R':
			case 'S':
			case 'O':
				// If the character is an underscore, or one of the other
				// characters indicating the presence of a line segment, and
				// the pen is not drawing, then start drawing.
				if(!drawing)
				{
					drawing = true;
					s = j;
				}
				break;

			case ' ':
			case '.':
			case 'f':
			case 'F':
			case 'u':
			case 's':
			case 'd':
			case 'l':
			case 'r':
			case 'o':
				// If the character is a space, or one of the other characters
				// indicating the absence of a line segment, and the pen is
				// drawing, then stop drawing and add the finished line
				// segment to the vector of line segments.
				if(drawing)
				{
					drawing = false;
					// Calculate the start and end points in metres.  Remember
					// that j and s are increasing along the horizontal axis in
					// the positive direction, while i is decreasing along the
					// vertical axis in the positive direction.
					double_pointxy start((s - 1) / 2 * size, (h - 1 - i) * size);
					double_pointxy end((j - 1) / 2 * size, (h - 1 - i) * size);
					// Add a new segment to the world.
					segments.push_back(Segment(start, end));
				}
				break;

			default:
				throw EIosFailure(
					EMsgD() << "Undefined symbol '" << asc[i][j]
					<< "' in maze file: " << fileName );
			}

			// Test the character at position (i, j) for a robot starting
			// position.  The default direction for the robot is facing up
			// (i.e. in the same positive direction as the horizontal axis.
			switch(asc[i][j])
			{
			case 'u':
			case 'U':
			case 's':
			case 'S':
			{
				// Calculate the position of the robot, remembering that s
				// starts at 1 and is incremented in units of 2, but i starts
				// at 0 and is incremented in units of 1.
				Position position((j * size) / 2, (h - i - 0.5) * size, M_PI/2);
				// Add a new robot starting position to the world.  The
				// default robot radius is used.
				robis.push_back(GlobalRobi(position));
				break;
			}
			case 'd':
			case 'D':
			{
				Position position((j * size) / 2, (h - i - 0.5) * size, -M_PI/2);
				robis.push_back(GlobalRobi(position));
				break;
			}
			case 'l':
			case 'L':
			{
				Position position((j * size) / 2, (h - i - 0.5) * size, -M_PI);
				robis.push_back(GlobalRobi(position));
				break;
			}
			case 'r':
			case 'R':
			{
				Position position((j * size) / 2, (h - i - 0.5) * size, 0);
				robis.push_back(GlobalRobi(position));
				break;
			}
			case 'o':
			case 'O':
			{
				double_pointxy position((j * size) / 2, (h - i - 0.5) * size);
				balls.push_back( Ball( position ) );
				break;
			}
			}
		}
	}

	// Loop through the even-numbered columns in column-major order,
	// with the row loop starting at the bottom row.  We can assume that
	// the i >= 1 and j >= 3 because this has already been tested for.
	for(int j = 0; j < w; j += 2)
	{
		// These variables are used to construct a single line segment.
		bool drawing = false;
		int s;

		for(int i = h - 1; i >= 0; i--)
		{
			// Test the character at position (i, j) for vertical line
			// segments.  This switch also includes the invalid character
			// test.
			switch(asc[i][j])
			{
			case '|':
				// If the character is a vertical bar, and the pen is not
				// drawing, then start drawing.
				if(!drawing)
				{
					drawing = true;
					s = i;
				}
				break;
			case ' ':
			case '_':
			case '.':
			case 'f':
			case 'F':
				// If the character is a horizontal bar, or one of the other
				// characters used to represent the absence of a line segment,
				// and the pen is drawing, then stop drawing and add the
				// finished line segment to the vector of line segments.
				if(drawing)
				{
					drawing = false;
					// Calculate the start and end points in metres.  Remember
					// that j and s are increasing along the horizontal axis in
					// the positive direction, while i is decreasing along the
					// vertical axis in the positive direction.
					double_pointxy start(j / 2 * size, (h - 1 - s) * size);
					double_pointxy end(j / 2 * size, (h - 1 - i) * size);
					// Add a new segment to the world.
					segments.push_back(Segment(start, end));
				}
				break;
			default:
				throw EIosFailure(
					EMsgD() << "Undefined symbol '" << asc[i][j]
					<< "' in maze file: " << fileName );
			}
		}
	}

	findBoundingBox();

#ifdef DEBUG
	copy( asc.begin(), asc.end(), ostream_iterator<string>(cout, "\n") );
#endif /* DEBUG */
}
