/// @file base.hh
/// @author Joshua Petitt <petitj01@tartarus.uwa.edu.au>
/// @author Stefan Schmitt <sschmitt@ee.uwa.edu.au>
/// @version 1.0
/// @date 2003-06
/// 
/// Declares low-level base classes.

#ifndef BASE_HH
#define BASE_HH

#include "base/templates.hh"

namespace EyeMind {

/// Universal base class. Each instance has an unique tag for identification
/// purposes. This class cannot be copied.

class Tagged
{
private:
	static unsigned ticket; ///< Unique tag source.
	unsigned tag; ///< Unique instance tag.

	NO_COPYING(Tagged); ///< Prevent copying.

public:
	/// Default constructor; initializes data members.
	Tagged();

	/// @name Attribute manipulation functions.
	//@{
	unsigned Tag(){ return tag; } ///< Returns unique instance tag
	//@}

	/// Equality operator.
	bool operator==(const Tagged& n);
};



/// A class for objects that can update themselves.

class Updatable : public Tagged
{
private:
	/// Timebase for updating; 0 for explicit-only updating.
	float timebase;
public:
	/// Virtual destructor.
	virtual ~Updatable(){}
	
	/// Update data; returns \c true if updatable attributes may have
	/// changed.
	virtual bool Update(){ return false; }

	/// @name Attribute manipulation functions.
	//@{
	float Timebase(){ return timebase; }
	void Timebase(float t);
	//@}
};
	


/// A Port is an Updatable object that can be active or inactive, e.g. sensors
/// and actors.

class Port : public Updatable
{
private:
	bool active; ///< Active flag;
public:
	/// Default constructor; initializes data members.
	Port();

	/// @name Attribute manipulation functions.
	//@{
	bool Active(){ return active; }
	void Active(bool a){ active=a; }
	//@}
};


}; // namespace EyeMind

#endif // BASE_HH

