#ifndef  HDT_H
#define  HDT_H

#include "types.h"

/* HDT = hardware description table */

typedef int     (*TestFunc) (DeviceSemantics);
typedef char    String6[7];
typedef char    String10[11];

/* prototypes */

int HDT_Validate(void);
void *HDT_FindEntry(TypeID,DeviceSemantics);
DeviceSemantics  HDT_FindSemantics(TypeID , int);
int   HDT_TypeCount(TypeID);
char* HDT_GetString(TypeID, DeviceSemantics);

/* DTL entry structure */
/*---------------------------------------------------------------------------------*/
typedef struct
{
  TypeID      group;
  String6     name;
  TestFunc    testfunc;
} DTL_entry_type;

/* HDT entry structure */
/*---------------------------------------------------------------------------------*/
typedef struct
{
  TypeID              type_id;
  DeviceSemantics     semantics;
  String6             device_name;
  void*               data_area;
} HDT_entry_type;

/* hardware description structures */
/*---------------------------------------------------------------------------------*/
typedef BYTE image_type[16*64];

typedef struct
{
  int     driver_version;
  int     tpu_channel;
  int     tpu_timer;
  int     pwm_period;
  BYTE*   out_pin_address;
  short   out_pin_fbit;
  short   out_pin_bbit;
  BYTE*   conv_table;     /* NULL if no conversion needed */
} motor_type;

typedef struct
{
  int     driver_version;
  int     master_tpu_channel;
  int	  slave_tpu_channel;
  DeviceSemantics motor;	/* connected motor */
  unsigned int clicksPerMeter;  /* needed for VW-Interface and odometer */
  float   maxspeed;		/* (in m/s) only needed for VW-Interface */
} quad_type;


typedef struct
{
  int     driver_version;
  int     tpu_channel;
  int     tpu_timer;
  int     pwm_period;
  int     pwm_start;
  int     pwm_stop;
} servo_type;


typedef struct
{
  int     driver_version;
  int     tpu_channel;
  int     tpu_timer;
  short   transition;
} bump_type;

typedef struct
{
  int     driver_version;
  int     tpu_channel;
} ir_type;

typedef struct
{
  short   version;
  short   channel;
  void*   pc_port;
  short   pc_pin;
  void*   cal_port;
  short   cal_pin;
  void*   sdo_port;
  short   sdo_pin;
}compass_type;


typedef struct
{
  short   driver_version;
  short   tpu_channel;
  short   tpu_timer;
  short   tpu_priority;
  short   period_or_pulse;
  short   accumulate;
}ppwa_type;


#define TPU_LOW_PRIO  1
#define TPU_MID_PRIO  2
#define TPU_HIGH_PRIO 3

typedef struct
{
  short   version;
  short   channel;
  short   priority;
  /* new in version 1: */
  short   use_in_robios;
  int     type;
  int     length;
  int     tog_mask;
  int     inv_mask;
  int     mode;
  int     bufsize;
  int     delay;
  int     code_key1;
  int     code_key2;
  int     code_key3;
  int     code_key4;
} irtv_type;


typedef struct
{
  short   driver_version;
  short   tpu_channel;
  BYTE*   in_pin_address;
  short   in_pin_bit;
  short   in_logic;
  BYTE*   out_pin_address;
  short   out_pin_bit;
  short   out_logic;
  short*  dist_table;
} psd_type;

typedef struct
{
  int     version;
  short   low_limit;
  short   high_limit;
} battery_type;

typedef struct
{
  short   version;
  BYTE*   out_latch_address;
  short   nr_out;
  BYTE*   in_latch_address;
  short   nr_in;
} latch_type;

/* int interface... interface already defined in windows.h */
typedef struct
{
    int version;
    short robi_id;
    short remote_control;
    short ser_interface;
    short serspeed;
    short imagemode;
} remote_type;
/* int interface... interface already defined in windows.h */
typedef struct
{
  int     version;
  int     id;         
  int     serspeed;
  int     handshake;
  int     ser_interface;
  int     auto_download;
  int     remote_control;   /* moved to remote_type but is still here for compatibility to old HDTs */
  int     cammode;
  int     battery_display;
  int     CPUclock;
  float   user_version;
  String10 name;
  unsigned char robi_id;    /* moved to remote_type but is still here for compatibility to old HDTs */
} info_type;

typedef struct
{
  short     version;
  short     rom_ws;
  short     ram_ws;
  short     lcd_ws;
  short     io_ws;
  short     serpar_ws;
} waitstate_type;


#define DIFFERENTIAL_DRIVE 0
#define ACKERMAN_DRIVE 1
#define SYNCHRO_DRIVE 2
#define TRICYCLE_DRIVE 3
#define OMNI_DRIVE 4

/* 2 wheel differential drive */
typedef struct
{
  DeviceSemantics quad_left;
  DeviceSemantics quad_right;
  float           wheel_dist; /* meters */
} diff_data;

/* acker_data is not used by now! It's just an example for an implementation
*/
typedef struct
{
  DeviceSemantics quad_steer;
  DeviceSemantics quad_drive;
  float           steerwheel_dist; /* meters */
  float           drivewheel_dist; /* meters */
  float                   axes_dist;       /* meters */
} acker_data;

/* 3 wheel synchro drive */
typedef struct
{
  DeviceSemantics quad_drive;
  DeviceSemantics quad_steer;
  DeviceSemantics quad_turn;
} synchro_data;

/* 4 wheel omni-directional Mechanum drive */
typedef struct
{
  DeviceSemantics quad_frontleft;
  DeviceSemantics quad_frontright;
  DeviceSemantics quad_backleft;
  DeviceSemantics quad_backright;
  float           wheel_dist; /* meters */
  float           axes_dist;  /* meters */
} omni_data;

/* to be compatible with old hdts */
#define vw_type vw_type_diff

typedef struct
{
  int     version;
  int     drive_type;
  diff_data drive_spec;
} vw_type_diff; 

typedef struct
{
  int     version;
  int     drive_type;
  acker_data drive_spec;
} vw_type_acker;

typedef struct
{
  int     version;
  int     drive_type;
  synchro_data drive_spec;
} vw_type_synchro;

typedef struct
{
  int       version;
  int       drive_type;
  omni_data drive_spec;
} vw_type_omni;

/****************************************************************************/
/*  Constantsdefinition                                                       */
/****************************************************************************/

/* for TPU */

#define TIMER1 1
#define TIMER2 2

/* for INFO */

#define PLATFORM 0
#define VEHICLE 1
#define WALKER 2

#define BATTERY_OFF 0
#define BATTERY_ON 1

#define NO_AUTOLOAD 0
#define AUTOLOAD 1
#define AUTOLOADSTART 3

/* Remote Parameter */
#define REMOTE_OFF 0
#define REMOTE_ON 1
#define REMOTE_PC 2
#define REMOTE_EYE 3

#define IMAGE_OFF 0
#define IMAGE_REDUCED 1
#define IMAGE_FULL 2
   
/* for PPWA driver*/
#define PPWA_PERIOD  0
#define PPWA_PULSE   1

/* for Bumper */
#define RISING  0x7
#define FALLING 0xB
#define EITHER  0xF

/* misc */

#define AH  0
#define AL  1

#define ON  1
#define OFF 0

/**************************************************************************/
/*  TypeID-constants, for type of device                                      */
/**************************************************************************/

#define UNKNOWN_TYPE   0
#define END_OF_HDT     -1
#define CHECKSUM       -2

#define MOTOR          -100
#define PSD            -200
#define SERVO          -300
#define QUAD           -400
#define BUMP           -500
#define IR             -600
#define VW             -700
#define BATTERY        -800
#define COMPASS        -900
#define IRTV           -1000
#define STARTIMAGE     -1100
#define STARTMELODY    -1200
#define REMOTE         -1300
#define PPWA           -1400
#define LATCH          -1500
#define INFO           -10000
#define WAIT           -11000
#define DATA           -30000

#define UNKNOWN_SEMANTICS    0
                                   
#endif

