/****************************/
/* TransHex (Linux)         */
/****************************/
/* Version 1.0              */
/* 02.07.2001  Michael Kapp */
/****************************/

#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <errno.h>
#include <stdio.h>
#include <termios.h>

#define RTS TIOCM_RTS
#define DTR TIOCM_DTR


/* Verzoegerung (< 1000 ms!) */

void Delay(int ms)
{
	usleep(ms*1000);
}


/* Serielle Schnittstelle oeffnen */

int OpenSerial(const char *device)
{
	int fd;

	if ((fd = open(device, O_RDWR | O_NOCTTY)) < 0)
	{
		perror(device);
		exit(1);
	}
	return fd;
}


/* Serielle Schnittstelle schliessen */

void CloseSerial(int fd)
{
	close(fd);
}


/* Serielle Schnittstelle konfigurieren */

int SetSerial(int fd, long baud, int handshake)
{
	int baudcode;
	struct termios tio;

	/* aktuelle Einstellungen lesen */
	tcgetattr(fd, &tio);

	switch (baud)
	{
		case   2400: baudcode = B2400; break;
		case   9600: baudcode = B9600; break;
		case  19200: baudcode = B19200; break;
		case  38400: baudcode = B38400; break;
		case  57600: baudcode = B57600; break;
		case 115200: baudcode = B115200; break;
		default:
			fprintf(stderr, "Illegal baud rate\n");
			exit(1);
	}
	/* Baudrate, 8N1, Steuerleitungen ignorieren, Empfang ermoeglichen, Flusskontrolle */
	tio.c_cflag = baudcode | CS8 | CLOCAL | CREAD;
	if (handshake) tio.c_cflag |= CRTSCTS;

	/* Frame/Parity-Fehler ignorieren */
	tio.c_iflag = IGNPAR;

	/* Raw output */
	tio.c_oflag = 0;

	/* Kein Echo */
	tio.c_lflag = 0;

	/* Control chars */
	tio.c_cc[VINTR]    = 0;     /* Ctrl-c */
	tio.c_cc[VQUIT]    = 0;     /* Ctrl-\ */
	tio.c_cc[VERASE]   = 0;     /* del */
	tio.c_cc[VKILL]    = 0;     /* @ */
	tio.c_cc[VEOF]     = 0;     /* Ctrl-d */
	tio.c_cc[VSWTC]    = 0;     /* '\0' */
	tio.c_cc[VSTART]   = 0;     /* Ctrl-q */
	tio.c_cc[VSTOP]    = 0;     /* Ctrl-s */
	tio.c_cc[VSUSP]    = 0;     /* Ctrl-z */
	tio.c_cc[VEOL]     = 0;     /* '\0' */
	tio.c_cc[VREPRINT] = 0;     /* Ctrl-r */
	tio.c_cc[VDISCARD] = 0;     /* Ctrl-u */
	tio.c_cc[VWERASE]  = 0;     /* Ctrl-w */
	tio.c_cc[VLNEXT]   = 0;     /* Ctrl-v */
	tio.c_cc[VEOL2]    = 0;     /* '\0' */
	/* blocking read until 1 char arrives */
	tio.c_cc[VMIN] = 1;
	tio.c_cc[VTIME] = 0;

	/* Einstellungen setzen */

	if (tcsetattr(fd, TCSADRAIN, &tio) < 0)
	{
		perror("Serial initialization failed");
		exit(1);
	}

	return fd;
}


/* RTS/DTR setzen */

void SetLines(int fd, int rtsdtr)
{
	if (ioctl(fd, TIOCMBIS, &rtsdtr) < 0)
	{
		perror("ioctl failed");
		exit(1);
	}
}


/* RTS/DTR loeschen */

void ClrLines(int fd, int rtsdtr)
{
	if (ioctl(fd, TIOCMBIC, &rtsdtr) < 0)
	{
		perror("ioctl failed");
		exit(1);
	}
}


/* IRmate zuruecksetzen */

void ResetIRmate(int fd)
{
	SetLines(fd, RTS | DTR);
	Delay(100);
	ClrLines(fd, DTR);
	Delay(2);
	SetLines(fd, DTR);
	Delay(2);
}


/* IRmate-Baudrate setzen */

void SetIRmateSpeed(int fd, long baud)
{
	unsigned char ctrlbyte;

	switch (baud)
	{
		case   2400: ctrlbyte = 8; break;
		case   9600: ctrlbyte = 4; break;
		case  19200: ctrlbyte = 3; break;
		case  38400: ctrlbyte = 2; break;
		case  57600: ctrlbyte = 1; break;
		case 115200: ctrlbyte = 0; break;
		default:
			fprintf(stderr, "Illegal baud rate\n");
			exit(1);
	}
	/* ctrlbyte |= 16  fuer kurze Pulse */

	ClrLines(fd, RTS);
	Delay(2);
	if (write(fd, &ctrlbyte, 1) != 1)
	{
		perror("Serial write failed");
		exit(1);
	}
	Delay(100);
	SetLines(fd, RTS);
}


/* Hauptprogramm */

int main(int argc, char *argv[])
{
	char *DeviceName = "/dev/ttyS0";
	long BaudRate = 115200;
	int NoHandshake = 0;
	int IRmate = 0;
	char *FileName = NULL;

	int fd;
	FILE *hexfile;
	long size;
	long count;
	int percent, old_percent;
	int i;
	unsigned char c;

	/* Parameter parsen */
	i = 1;
	while (i<argc)
	{
		if (strcmp(argv[i], "-c")==0)
		{
			i++;
			if (i<argc)
				DeviceName = argv[i];
			else
			{
				fprintf(stderr, "Device name missing\n");
				exit(1);
			}
		}
		else if (strcmp(argv[i], "-b")==0)
		{
			i++;
			if (i<argc)
				BaudRate = atol(argv[i]);
			else
			{
				fprintf(stderr, "Baud rate missing\n");
				exit(1);
			}
		}
		else if (strcmp(argv[i], "-nohandshake")==0)
		{
			NoHandshake = 1;
		}
		else if (strcmp(argv[i], "-irmate")==0)
		{
			IRmate = 1;
			NoHandshake = 1;
		}
		else
		{
			if (argv[i][0]=='-')
			{
				fprintf(stderr, "Unknown option: %s\n", argv[i]);
				exit(1);
			}
			FileName = argv[i];
		}

		i++;
	}

	if (FileName==NULL)
	{
		fprintf(stderr, "TransHex 1.0 for Linux\n"
			"(c) 2001 Michael Kapp, Michael Kasper\n"
			"Usage: %s [OPTIONS] <HEXFILE>\n\n"
			"Available OPTIONS:\n"
			"-c <DEVICE>    Serial device to use\n"
			"-b <BAUD>      Transmission speed. Must be one of the following values:\n"
			"               2400, 9600, 19200, 38400, 57600, 115200\n"
			"-nohandshake   Don't use RTS/CTS handshaking\n"
			"-irmate        Enable transmission via IRmate (includes -nohandshake)\n\n"
			"Default values: -c /dev/ttyS0 -b 115200\n",
			argv[0]);
		exit(1);
	}

	/* Schnittstelle initialisieren */
	fd = OpenSerial(DeviceName);
	if (IRmate)
	{
		printf("Initializing IRmate...\n");
		SetSerial(fd, 9600, 0);
		ResetIRmate(fd);
		SetIRmateSpeed(fd, BaudRate);
	}
	SetSerial(fd, BaudRate, !NoHandshake);

	/* File uebertragen */
	hexfile = fopen(FileName, "rb");
	if (hexfile==NULL)
	{
		perror(FileName);
		exit(1);
	}
	errno = 0;
	fseek(hexfile, 0, SEEK_END);
	size = ftell(hexfile);
	fseek(hexfile, 0, SEEK_SET);
	if (errno!=0)
	{
		perror("Could not determine file size");
		exit(1);
	}
	old_percent = -1;
	count = 0;

	setbuf(stdout, NULL);	/* ungepufferte Ausgabe fuer Fortschrittsanzeige */

	while ((i=fgetc(hexfile))!=EOF)
	{
		c = i;
		if (write(fd, &c, 1) != 1)
		{
			perror("Serial write failed");
			exit(1);
		}
		count++;
		percent = count*100/size;
		if (percent!=old_percent)
		{
			printf("Starting download: %ld%%\r", percent);
			old_percent = percent;
		}
	}
	printf("\n");

	fclose(hexfile);
	CloseSerial(fd);
	return 0;
}
