// LOADER/MONITOR
//
// Copyright (c) B Knudsen Data 1999-2002, Trondheim - Norway.
//
// This code can be used freely for all kinds of testing and debugging
// PICmicro programs. Including this code in a commercial product
// requires a license of the CC5X compiler.
//
// Developed for use on the PIC16F870/871/873/874/876/877,
// PIC16F873A/874A/876A/877A devices.
//
// Version 2.0


// NOTE: this file must be included in a program (demo.c)

// DEMO.c :  demo application that includes the loader/monitor code
// DEMO1.c : demo application to be loaded using the loader/monitor
// DEMO2.c : demo application with hardcoded breakpoints and testpoints



#pragma origin LOADER_START

#pragma rambank BANK_RTM



char decodeHex(char digit)
{
    if (digit >= '0'  &&  digit <= '9')
        return digit - '0';
    if (digit >= 'A'  &&  digit <= 'F')
        return digit - 'A' + 10;
    return digit | 0x80;
}


#if defined HELPINFO && defined RTMONITOR
char nextHelpChar(char index)
{
    // HELP INFORMATION
    skip(index);
    #pragma return[] = "M/C<h>[+h], M/C<h>=<h>\r\n"
   #ifdef EXECUTE_SINGLE
    #pragma return[] = "G,R,F<h>\r\n"
   #else
    #pragma return[] = "G,R\r\n"
   #endif
    #pragma return[] = "\0"
}
#endif


#ifdef BLOCK_WRITE

void writeBuffer( void)
/*
  Writes data to FLASH data memory in blocks of <N> words
  each time in sequence.
  Note that EEPROM data memory is also updated through
  this routine to reduce total code size.
*/
{
    retest = 0;
    progBufferInit = 0;

   CHECK_BUFFER:
    // skip programming if no change is made
    uns8 i = NoOfProgBuffer / 2;
    FSR = &progBuffer[0] & 0xFF;
   #if __IRP_RAM__ != 0
    IRP = &progBuffer[0] / 256;
   #endif
    EEADR = baseAddrProg.low8;
    W = baseAddrProg.high8;
    EEADRH = W;
    EEPGD = 1;  // point to FLASH program memory
    if (W == 0x21)
        EEPGD = 0; // EEPROM data memory
    do  {
        // ----- required sequence starts -----
        RD = 1;
        nop();
        nop();
        // ----- required sequence ends -----
        if (INDF != EEDATA)
            goto UPDATE_FLASH;
        FSR ++;
        if (INDF != EEDATH  &&  EEPGD == 1)
            goto UPDATE_FLASH;
        FSR ++;
        EEADR ++;  // increment LSB of address
    } while (--i > 0);
    return;  // no writing needed

   UPDATE_FLASH:

    if (retest)
        goto PROGRAMMING_FAILED;

    i = NoOfProgBuffer / 2;
    FSR = &progBuffer[0] & 0xFF;
    EEADR = baseAddrProg.low8;
    // OK: EEADRH = baseAddrProg.high8;
    do  {
        // update latches
        EEDATA = INDF;
        FSR ++;
        EEDATH = INDF;
        FSR ++;
        WREN = 1;
        // GIE MUST be 0 during programming
        // ----- required sequence starts -----
        EECON2 = 0x55;
        EECON2 = 0xAA;
        WR = 1;     // start write / erase
        nop();
        nop();
        // ----- required sequence ends -----
        while (WR)  // required for data EEPROM write
            ;
        // WR is cleared automatically when writing finishes
        EEIF = 0;
        EEADR ++;  // increment LSB of address
    } while (--i > 0);

    WREN = 0;
    retest = 1;
    goto CHECK_BUFFER;

   PROGRAMMING_FAILED:
    programmingFailed();
}
#endif



char programMemory(char nb)
/*
  nb: 1..16 : program location(s)
      >= 0x80: read one location (EEDATA,EEDATH)

  dataAddressH, dataAddressL : points to memory location
    FLASH (program memory): 0 .. LOADER_START
    EEPROM: 0x2100 - 0x217F/0x21FF
  dataWord1H, dataWord1L: contents of first location
*/
{
    char i;
    for (i = 0; i < nb; )  {

        if (dataAddressH != 0x21)  {
            // (K-1) - x < 0 => fail
            W = ((LOADER_START-1) & 0xFF) - dataAddressL;
            W = dataAddressH;
            btss(Carry);
            W = incsz(dataAddressH);
            W = ((LOADER_START-1) / 256) - W;
            if (!Carry)
                return DownloaderOverwrite; // skip and set error flag
        }

      #ifdef BLOCK_WRITE
       REINIT:
      #else
        retest = 0;
      #endif
        EEADR = dataAddressL;
        W = dataAddressH;
        EEADRH = W;

        EEPGD = 1;  // FLASH program memory
        if (W == 0x21)
            EEPGD = 0; // EEPROM data memory

       RETEST_LOCATION:

        // check data word, skip if equal
        // ----- required sequence starts -----
        RD = 1;
        nop();
        nop();
        // ----- required sequence ends -----

        if (nb & 0x80)
            return 0;  // return if location should be read only

       #if __IRP_RAM__ != 0
        IRP = comTab / 256;
       #endif

       #ifdef BLOCK_WRITE
        if (!progBufferInit)  {
            // init address to FLASH/EEPROM block to program
            baseAddrProg.low8 = dataAddressL & ~(ProgBufferMask/2);
            EEADR = baseAddrProg.low8;
            baseAddrProg.high8 = dataAddressH;
            // OK: EEADRH = baseAddrProg.high8;

            // first copy existing memory contents to buffer
            FSR = &progBuffer[0] & 0xFF;
            uns8 n = NoOfProgBuffer / 2;
            do  {
                // ----- required sequence starts -----
                RD = 1;
                nop();
                nop();
                // ----- required sequence ends -----
                INDF = EEDATA;
                FSR ++;
                INDF = EEDATH;
                FSR ++;
                EEADR ++;  // increment LSB of address
            } while (--n);
            progBufferInit = 1;
        }
        if ((dataAddressL & ~(ProgBufferMask/2)) != baseAddrProg.low8  ||
            dataAddressH != baseAddrProg.high8)  {
            // data outside buffer, write buffer to FLASH
            writeBuffer(); // update FLASH
            goto REINIT;
        }
        // copy word to buffer
        FSR = &comTab[4+i];
        uns8 dataL, dataH;
        dataL = INDF;
        FSR++;
        dataH = INDF;
        W = dataAddressL + dataAddressL;  // * 2
        FSR = &progBuffer[ W & ProgBufferMask];
        INDF = dataL;
        FSR ++;
        INDF = dataH;

       #else

        FSR = &comTab[4+i];
        W = INDF;
        FSR ++;
        // INDF (msb) and W now holds the data to be written.

        // Now check what data was read:
        if (W != EEDATA  ||  (EEPGD == 1  &&  INDF != EEDATH))  {
            EEDATH = INDF;
            FSR --;
            EEDATA = INDF;

            // do not write more than once !
            if (retest)  {
               programmingFailed();
               goto SKIP_LOCATION;
            }

            // write data word
            WREN = 1;
            // GIE MUST be 0 during programming
            // ----- required sequence starts -----
            EECON2 = 0x55;
            EECON2 = 0xAA;
            WR = 1;
            nop();
            nop();
            // ----- required sequence ends -----
            WREN = 0;
            while (WR)  // required for data EEPROM write
                ;
            // WR is cleared automatically when writing finishes
            EEIF = 0;

            retest = 1;
            goto RETEST_LOCATION;
        }

       #endif

       SKIP_LOCATION:

        dataAddressL ++;
        if (dataAddressL == 0)
            dataAddressH ++;
       #ifdef DOWNLOADER
        noOfCodeWords ++;
       #endif
        i += 2;
    }
    return 0;
}



#ifdef DOWNLOADER

char dataRecord(void)
/*
 :BBaaaaTT112233...CC  : ASCII INTEL record
 comTab[] = 'BaaT123...C' : compressed record
*/
{
    char nb;

    nb = comI - 5;
    if (nb & 0x80 || nb != comTab[0] || nb & 1)
        return RecordFramingError;   // no action

    { // verify checksum
        char sum = 0;
        char i = comI;
        do  {
            sum += comTab[i-1];
        } while (--i > 0);
        if (sum != 0)
            return ChecksumError;   // no action
    }

    if (nb == 0  &&  comTab[3] == 0x01)  {
        eof = 1;
        return 0;
    }
   #ifdef ENABLE_INHX32
    if (comTab[3].2)
        return 0; // high address must be 0x0000 (not checked)
   #endif
    if (comTab[3] || (comTab[2] & 0x01))
        return RecordTypeError;

    // program data memory
    Carry = 0;
    dataAddressH = rr(dataAddressH);
    dataAddressL = rr(dataAddressL);
    return programMemory(nb);
}
#endif


void putChar(char ch)
{
    while (!TXIF)  // SERIAL-IO
        ;  // wait until previous character transmitted
    TXREG = ch;    // clear TXIF
}



char toHex(char c)
{
    c &= 0xF;
    c += '0';
    if (c > '9')
        c += 'A' - '9' - 1;
    return c;
}

void putHex(char digit)
{
    putChar(toHex(swap(digit)));
    putChar(toHex(digit));
}


#if defined RTMONITOR && defined EXECUTE_SINGLE
void userFunction(void)
{
    PCLATH = memAddress / 256;
    W = (char) memAddress;
    STATUS = 0;  // preset bank selection bits
    PCL = W;
}
#endif


void monitor(char W)
/*
  W = 1: startup, goto start-vector on exit
  W = 0: normal call, use return on exit if RTMONITOR is defined,
         otherwise goto start-vector
*/
{
    rtFlags = W & 1;
   #ifdef RESTORE_GIE
    if (GIE)
        savedGIE = 1;
   #endif
    GIE = 0;

    // initialize serial communication (SERIAL-IO)
    SPBRG = SPBRG_RATE;
    TXSTA = TXEN_transmit_enable | BRGH_async_high_speed;
    RCSTA = SPEN_Serial_Port_Enable | CREN_receive_enable;

    TRIS_TX = 0;  // TX-pin is output
    TRIS_RX = 1;  // RX-pin is input

   #ifdef DOWNLOADER
    // OK: eof = 0;
    echo = 1;
    iXtra = 0;
    maxXtra = 0;
    downloadErrors = 0;
    noOfCodeWords = 0;
   #endif
   #ifdef BLOCK_WRITE
    progBufferInit = 0;
   #endif

    /* execute commands */
    while (1)  {
       #ifdef DOWNLOADER
        char nXtra;
        intelHex = 0;
       #endif
        {
           #ifdef DOWNLOADER
            char lbc, msd;
           #endif
            comI = 0;
            errorFlag = 0;
            if (echo)
                putChar('>');
            while (1)  {
                char c;
               NEXT:

               #if defined DOWNLOADER
                if (nXtra < iXtra)
                    c = xBuffer[nXtra++];
                else
               #endif
                {
                    // READ NEXT CHARACTER FROM INPUT STREAM

                    do  {
                       #ifdef WATCHDOG_ENABLED
                        clrwdt();
                       #else
                        ;
                       #endif
                    } while (!RCIF);   // SERIAL-IO
                    c = RCREG; // clear RCIF
                    if (FERR || OERR)  {
                        framingOrOverflow();
                        errorFlag = 1;
                    }
                }

                if (c >= 'a')
                    c &= ~0x20;  // lower case => upper case
               #ifdef DOWNLOADER
                if (echo)
               #endif
                {
                    putChar(c);
                    if (c == CR)
                        putChar(LF);
                }
                if (c == ' ')
                    goto NEXT;  // skip spaces
                if (c == CR  ||  c == LF)
                    break;      // process input

                if (c == DEL)  {
                    // DEL will not work for INTEL DATA RECORDS!
                    comI --;
                    if (comI & 0x80)  // negative ?
                       comI = 0;
                    putChar(' ');
                    putChar(DEL);
                }
                else  {
                   #ifdef DOWNLOADER
                    if ( c == ':')  {
                        intelHex = 1;
                        echo = 0;
                        lbc = 0;
                        goto NEXT;
                    }
                    if (intelHex)  {
                        c = decodeHex(c);
                        if (c & 0x80)  {
                            errorFlag = 1;
                            recordFramingError();
                        }
                        if (!(++lbc & 1))  {
                            // compress 2 HEX digits (compress buffer)
                            comI -= 1;
                            c |= swap(msd);
                        }
                        msd = c;
                    }
                   #endif
                    if (comI >= NoOfKTab)  {
                        errorFlag = 1;
                        lineOverflow();
                    }
                    else
                        comTab[comI++] = c;
                }
            }
        }

        // Execute single line command or data record
        echo = 1;
        if (errorFlag)
            ;
        else if (comI == 0)  {
           #ifdef RTMONITOR
            if ( !startUp)
                break;     // return to application on 'Enter'
           #endif
        }
        else  {
           #ifdef DOWNLOADER
            if (intelHex)  {
                putChar(XOFF);
                iXtra = 0;
                nXtra = 0;
                {
                   WAIT:
                    char k = MinDelayAfterXOFF * 10;
                    do  {
                        char i = TenthMilliSec + 1;
                        do  {
                            if (RCIF)  {  // SERIAL-IO
                                if (iXtra == ExtraBufferSize)  {
                                    dataLost();
                                    iXtra--;
                                }
                                xBuffer[iXtra++] = RCREG; // clear RCIF
                                goto WAIT;
                            }
                        } while (--i > 0);
                    } while (--k > 0);
                }
                if (iXtra > maxXtra)
                    maxXtra = iXtra;
                W = dataRecord();
                downloadErrors |= W;
                if (W == 0)
                    echo = 0;
                putChar(XON);
               #ifdef WRITE_DOT
                putChar('.');
               #endif
            }
            else
           #endif
            // execute command
            {
               #ifdef RTMONITOR
                char t;
                uns16 memValue;
               #endif

                #define counter memValue.low8

                switch(comTab[0])  {

                  case 'G':  // execute user program
                    goto EXIT; /* to main(); */

                #ifdef RTMONITOR

                  case 'R':  // reset
                    startUp = 1;
                    break;

                 #ifdef HELPINFO
                  case 'H':  // help
                    {
                        char i = 0;
                        char t;
                        while (1) {
                            t = nextHelpChar(i);
                            if (t == '\0')
                                break;
                            putChar(t);
                            i++;
                        }
                    }
                    break;
                 #endif

                  default:  /* M,C,F */
                    // decode 16 bit address and 16 value
                    // format: <address><t><value>
                    {
                        char c;
                        char decodeI = 0;
                        bit address = 1;
                        while (1)  {
                            memValue = 0;
                            while (1)  {
                                c = 0x80;
                                if (++decodeI < comI)
                                    c = decodeHex(comTab[decodeI]);
                                if (c & 0x80)
                                    break;
                                memValue <<= 4;
                                memValue |= c;
                            }
                            if (!address)
                                break;
                            memAddress = memValue;
                            address = 0;
                            t = c;
                        }
                    }

                    {
                        char mi = 0;
                        char mask = 0xF;
                        bit accessRAM = 1;

                        switch(comTab[0])  {

                          case 'C':  // program code and EEPROM
                            mask &= ~8;  // 8 words on each line
                            accessRAM = 0;

                          case 'M':  // memory (RAM)

                            while (1)  {
                                // prepare address to (next) location
                                if (accessRAM)  {
                                   #if __IRP_RAM__ != 0
                                    IRP = memAddress.8;
                                   #endif
                                    FSR = (char) memAddress;
                                }
                                else  {
                                    dataAddressH = memAddress / 256;
                                    dataAddressL = (char) memAddress;
                                    dataWord1H = memValue / 256;
                                    dataWord1L = memValue % 256;
                                }

                                if (t == ('='|0x80))  {
                                    // Write location
                                    if (accessRAM)
                                        INDF = (char) memValue;
                                    else
                                        programMemory(1);
                                    break;  // single word write only
                                }
                                // Write address to new line
                                if (!(mi & mask))  {
                                    putHex(memAddress / 256);
                                    putHex(memAddress % 256);
                                    putChar(':');
                                    putChar(' ');
                                }
                                if (!(mi & 3))
                                    putChar(' '); // for readability
                                if (accessRAM)
                                    putHex(INDF);
                                else  {
                                   #ifdef BLOCK_WRITE
                                    if (progBufferInit)
                                        writeBuffer();  // update FLASH
                                   #endif
                                    programMemory(0x80); // read
                                    if (!(dataAddressH & 0x20))
                                        putHex(EEDATH);  // 16 bit data
                                    putHex(EEDATA);
                                    // 256 locations are too many when
                                    // displaying program memory
                                    if (counter == 0)
                                       counter |= 0x80;  // 128 words
                                }
                                putChar(' ');
                                memAddress ++;
                                // new line after 16 (or 8) locations
                                if (!(++mi & mask))  {
                                    putChar(CR);
                                    putChar(LF);
                                }
                                counter --; // up to 256 locations
                                if ((char)counter == 0)
                                    break;
                            }
                            break;

                         #ifdef EXECUTE_SINGLE
                          case 'F':  // execute function
                           #ifdef BLOCK_WRITE
                            if (progBufferInit)
                                writeBuffer();  // update FLASH
                           #endif
                            // use call to get a return address on the stack
                            userFunction();
                            PCLATH = LOADER_START / 256; // required
                            break;
                         #endif

                          default:
                            unknownCommand();
                            break;
                        }
                    }
                #else

                  default:
                    unknownCommand();
                    break;
                #endif
                }
            }
        }

       #ifdef DOWNLOADER
        if (eof)  {
            echo = 1;
            putChar('#');
            putHex(noOfCodeWords / 256);
            putHex(noOfCodeWords % 256);
            putChar(':');
            putHex(maxXtra);
            putChar(' ');
            putChar('/');
            putHex(downloadErrors);
            putChar(CR);
            putChar(LF);
          #if defined AUTOSTART || !defined RTMONITOR
            // OPTIONAL: test programming errors before automatic start
           #ifdef RTMONITOR
            startUp = 1;  // ensure start of the application
           #endif
            goto EXIT;
          #endif
        }
       #endif
    }

  EXIT:

   #ifdef BLOCK_WRITE
    if (progBufferInit)
        writeBuffer();  // update FLASH
   #endif

   #ifdef RESTORE_GIE
    if (savedGIE)
        GIE = 1;
   #endif
   #ifdef RTMONITOR
    if (!startUp)
        return;  // return to application code (after monitor(0);)
   #endif
    PCLATH = 0;
   #ifdef  START_MONITOR_ON_RESET
    PCL = 3;  // goto application resetVector
   #else
    PCL = 0;  // goto application resetVector
   #endif
}

#if __CodePages__ == 4
 #pragma origin 0x1FFE
#elif __CodePages__ == 2
 #pragma origin 0xFFE
#else
 #pragma origin 0x7FE
#endif

#pragma update_RP 0
// this is the startup link
void linkMonitor(void)
{
    monitor(1);    // goto link
}
#pragma update_RP 1

