
/*
 *   IMPROV
 *
 *   by Michael Rudolph <rudolpml@hermes.informatik.uni-stuttgart.de>
 *   based on XFQCAM by Paul Chinn <loomer@svpal.org>
 *
 *   QuickCam code by Thomas Davis
 *   Additional QuickCam code Scott Laird <scott@laird.com>
 *   Adapted by Thomas Braunl, Uni Stuttgart, 1996
 *
 *   $Id: xdemo.c,v 1.7 1996/07/04 12:08:13 rudolpml Exp $
 */


#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>
#include <forms.h>
#include "xfqcam.h"
#include "controlpanel.h"
#include "qcam.h"
#include "improc.h"


static BYTE  im0[IMAGE_SIZE];
static BYTE  im1[IMAGE_SIZE];
static BYTE  * imageBuffer;
static BYTE  images[2][NO_OF_IMAGES+1][IMAGE_SIZE];


static int iplNOPFlag[NO_OF_IMAGES];
IPLOP iplOps[NO_OF_IMAGES][MAX_IPL_OPS];
int iplCount[NO_OF_IMAGES];
int iplEditCount[NO_OF_IMAGES];


static char struc[]= { 1, 1, 1, 1, 1, 1, 1, 1, 1 };
static int    diff  = 0;
static int    done  = 0;
static int num_vals = 16;
static int bufferFlag;


extern IPL_CONFIG _IPLConfig;
extern FD_QuickCam * mainwin;
extern BYTE extraImageBuffer[];
extern volatile int stopFlag;


/*
 *    forward declaration of local helper functions
 */

static void InitIPLOps(void);
static void ProcessImage(int imageNumber,
                         BYTE * imageIn,
                         BYTE * imageOut,
                         E_IPLOP operation,
                         double * args);



int main(int argc, char ** argv)
{
    int i, j;
    int rc;
    BYTE * opBuffer;

    InitIPLOps();

    _IPLConfig.colorBlack  = 0;
    _IPLConfig.colorWhite  = 15;
    _IPLConfig.imageWidth  = 160;
    _IPLConfig.imageHeight = 120;

    IPL_init(&_IPLConfig);

    rc = QC_init();

    if (rc < 0)
    {
        if (rc == -1)
            fprintf(stderr, "quickcam already in use ?! (remove /tmp/.quickcam and try again)\n");
        exit(1);
    }

    if (argc > 1)
        QCAMLoadFile(argv[1]);

    if (QCAMinit() != 0)
    {
        fprintf(stderr, "qcam init failed\n");
        QC_exit();
        exit(1);
    }

    while(!done)
    {
        QCAMget(&images[bufferFlag][0][0], &diff, &done);

        if (stopFlag)
            continue;

        fl_clear_browser(mainwin -> FD_LOG_BROWSER);

        imageBuffer = &images[bufferFlag][0][0];
        opBuffer    = &im0[0];

        for (i = 0; i < NO_OF_IMAGES; i++)
	{
            int nopFlag = 1;

            for (j = 0; j < iplCount[i]; j++)
	    {
                if (iplOps[i][j].operation != E_NOP)
		{
                    ProcessImage(i,
                                 imageBuffer,
                                 opBuffer,
                                 iplOps[i][j].operation,
                                 &iplOps[i][j].param[0]);

                    if (opBuffer == &im0[0])
		    {
                        opBuffer = &im1[0];
                        imageBuffer = &im0[0];
		    }
                    else
		    {
                        opBuffer = &im0[0];
                        imageBuffer = &im1[0];
		    }

                    nopFlag = 0;
                    iplNOPFlag[i] = 0;
		}
	    }

            if (nopFlag != 0)
  	    {
                if (iplNOPFlag[i] == 0)
		{
                    memset(&images[bufferFlag][i+1][0], 0, IMAGE_SIZE);
                    QCAMput(&images[bufferFlag][i+1][0], i + 1);
                    iplNOPFlag[i] = 1;
		}
            }
            else
	    {
                memcpy(&images[bufferFlag][i+1][0], imageBuffer, IMAGE_SIZE);
                QCAMput(imageBuffer, i + 1);
	    }
	}

        bufferFlag = 1 - bufferFlag;
    }


    QCAMexit();
    QC_exit();

    exit(0);
}

static void InitIPLOps(void)
{
    int i;

    for (i = 0; i < NO_OF_IMAGES; i++)
    {
        iplCount[i] = 1;
        iplOps[i][0].operation = E_NOP;
    }
}

static void ProcessImage(int imageNumber,
                         BYTE * imageIn,
                         BYTE * imageOut,
                         E_IPLOP operation,
                         double * args)
{
    XYDistance distance[128];


    switch (operation)
    {
        case E_NOP:
            break;
        case E_Laplace:
            IPL_laplace(imageIn, imageOut);
            break;
        case E_Sobel:
            IPL_sobel(imageIn, imageOut);
            break;
        case E_Median:
            IPL_median(imageIn, imageOut);
            break;
        case E_Mean:
            IPL_mean(imageIn, imageOut);
            break;
        case E_Threshold:
            IPL_threshold(imageIn, imageOut, (BYTE)(args[0] * 16));
            break;
        case E_GrayStretch:
            IPL_gray_stretch(imageIn, imageOut);
            break;
        case E_GrayReduce:
            IPL_gray_reduce(imageIn, imageOut, num_vals);
            break;
        case E_Erosion:
            IPL_erosion(imageIn, imageOut, struc);
            break;
        case E_Dilation:
            IPL_dilation(imageIn, imageOut, struc);
            break;
        case E_Open:
            IPL_open(imageIn, imageOut, struc);
            break;
        case E_Close:
            IPL_close(imageIn, imageOut,struc);
            break;
        case E_Fill:
            IPL_fill(imageIn,
                     imageOut,
                     (int)(args[0] * _IPLConfig.imageWidth),
                     (int)(args[1] * _IPLConfig.imageHeight),
                     struc);
            break;

        case E_ShowXY:
            IPL_showxy(imageIn,
                       imageOut,
                       (int)(args[0] * _IPLConfig.imageWidth),
                       (int)(args[1] * _IPLConfig.imageHeight));
            break;
        case E_Connected:
            IPL_connected(imageIn,
                     imageOut,
                     (int)(args[0] * _IPLConfig.imageWidth),
                     (int)(args[1] * _IPLConfig.imageHeight),
                     struc);
            break;
        case E_Boundary:
            IPL_boundary(imageIn, imageOut, struc);
            break;
        case E_Skeleton:
            IPL_skeleton(imageIn, imageOut, struc);
            break;
        case E_Circles:
            IPL_FindCircles(imageIn, imageOut, &distance[0]);
            break;
        case E_Negation:
            IPL_negation(imageIn, imageOut);
            break;

        case E_Noise:
            IPL_noise(imageIn, imageOut, args[0]);
            break;

        case E_Difference:
            IPL_difference(&images[bufferFlag][imageNumber][0],
                           &images[1-bufferFlag][imageNumber][0],
                           imageOut);
            break;

        case E_Min:
	    IPL_min(imageIn, imageOut);
            break;

        case E_Max:
            IPL_max(imageIn, imageOut);
            break;

        case E_Corner:
            IPL_corner(imageIn, imageOut);
            break;

        case E_Count:
            IPL_count(imageIn, imageOut, (BYTE)(args[0] *
                                                 (_IPLConfig.colorWhite-
                                                  _IPLConfig.colorBlack)));
            break;

        case E_Overlay:
            IPL_overlay(&images[bufferFlag][0][0],
                        imageIn,
                        imageOut,
                        (BYTE)(args[0] * (_IPLConfig.colorWhite-
                                          _IPLConfig.colorBlack)));
            break;

        case E_Dithering:
            IPL_dither(imageIn, imageOut);
            break;

        case E_Region:
            IPL_region_growing(imageIn,
                               imageOut,
                               (BYTE)(args[0] * (_IPLConfig.colorWhite-
                                                 _IPLConfig.colorBlack)));
            break;

        default:
            IPL_identity(imageIn, imageOut);
            break;
    }
}


int printf(const char * format, ...)
{
    char temp[1024];
    va_list vargs;
    int count;

    va_start(vargs, format);
    count = vsprintf(temp, format, vargs);
    fl_add_browser_line(mainwin -> FD_LOG_BROWSER, temp);
    va_end(vargs);

    return count;
}
