#include "Improv.h"

Improv* Improv::_instance = 0;

Improv* Improv::Instance(int argc, char **argv) 
{
	//  Return the singleton
	if(_instance == 0)
		_instance = new Improv(argc, argv);

	return _instance;
}

Improv::Improv(int argc, char **argv)
{
	config = new improvConfig();
	config->load(argc, argv);
	if(!config->load()) {
		//config->clear();
		config->clearLayout();
		config->load(argc, argv);
		config->loadDefaults();
		config->setModified(true);
	}
	else if(!config->verifyWidgetConfig()) {
		fprintf(stderr,"Error: Invalid custom widget details in config file %s - resorting to default layout.\n", config->getDefaultFilename());
		//config->clear();
		config->clearLayout();
		config->load(argc, argv);
		config->loadDefaults();
		config->setModified(true);
	}
	else {
		config->loadDefaults();
	}

	this->pluginMenu = NULL;
	constructPluginMenu();

	animThread = new AnimThread();
	animThread->setFPS(20);
	animThread->setUnlimitedFPS(false);

	this->menuId = -1;
	this->paramWidget = 0;
	this->statusBar = 0;
	this->statusBarMsg = NULL;
	this->fpsMsg = NULL;
	this->resultMsg = NULL;
	this->camOptOpen = false;
}

Improv::~Improv() {
	playing = false;
	delete paramWidget;
	delete statusBar;
	delete animThread;
	delete pluginMenu;
	delete config;
	seqWidgetList.clear();
	srcWidgetList.clear();
}

bool Improv::isPlaying(void)
{
	return playing;
}

bool Improv::isReady(void) {
	if((srcWidgetList.empty()==false)&&(seqWidgetList.empty()==false)){
		return true;
	}
	else {
		return false;
	}
}

void Improv::setPlaying(bool opt)
{
	QValueList<SrcWidget*>::iterator it;
	playing = opt;
	if(opt == true) {
		for(it = srcWidgetList.begin(); it != srcWidgetList.end(); it++) {
			(*it)->setSrcEnabled(false);
		}
		if(animThread->running()) {
			animThread->exit();
		}
		if(resultMsg!=NULL) resultMsg = "";
		animThread->start();
	}
	else {
		resultMsg = "";
		for(it = srcWidgetList.begin(); it != srcWidgetList.end(); it++) {
			(*it)->setSrcEnabled(true);
		}
	}
}

int Improv::seq_file_select(const struct dirent *entry)
{
	if ( (strcmp(entry->d_name, ".") == 0)  ||
			 (strcmp(entry->d_name,  "..") == 0))
		return 0;
	else return 1;
}

int Improv::plugin_file_select(const struct dirent *entry)
{
	if ( (strcmp(entry->d_name, ".") == 0)  ||
		 (strcmp(entry->d_name, "..") == 0) ||
		 (strstr(entry->d_name, ".so") == NULL))
		return 0;
	else return 1;
}

void Improv::constructPluginMenu(void)
{
	int num_plugs, i;
	char *pluginPath, *thisPlugin;
	struct dirent **namelist;
	ipPlugin *plugin;
	QValueList<Plugin*>::iterator it;

	pluginPath = this->config->getPluginPath();

	assert(pluginPath);

	num_plugs = scandir(pluginPath, &namelist, plugin_file_select, alphasort);

	i = num_plugs;

	if (num_plugs <= 0) {
		puts("Error: Didn't find any plugins");
		if(getPluginMenu() != NULL) {
			pluginList.clear();
			delete pluginMenu;
		}
		pluginMenu = NULL;
	}
	else {
		pluginMenu = new PluginMenu();
		while(i--) {
			thisPlugin = (char *)malloc(sizeof(pluginPath) + sizeof(namelist[i]->d_name) + 2*sizeof(char));
			if(thisPlugin == NULL) {
				fprintf(stderr,"Error: constructPluginMenu - No memory available for plugin name\n");
				return;
			}
			thisPlugin = strcpy(thisPlugin, pluginPath);
			if(pluginPath[strlen(pluginPath)-1]!='/') thisPlugin = strcat(thisPlugin, "/");
			thisPlugin = strcat(thisPlugin, namelist[i]->d_name);
			plugin = new ipPlugin(thisPlugin);
			if(plugin->isReady() == false) {
				fprintf(stderr,"Error: Unable to initialize plugin '%s'\n", namelist[i]->d_name);
			}
			else {
				for(it=pluginList.begin();it!=pluginList.end();it++) {
					if((*it)->getUniqueId() == plugin->getUniqueId()) {
						fprintf(stderr,"Error: Two plugins ('%s' and\n       '%s') have the same id (%d)\n       %s was not loaded\n",
							(*it)->getFileName(), plugin->getFileName(), plugin->getUniqueId(), plugin->getFileName());
						break;
					}
				}
				if(it==pluginList.end()) {
					pluginList.append(plugin);
					pluginMenu->addPlugin(plugin);
					printf("Loaded %s\n", namelist[i]->d_name);
					free(namelist[i]);
					free(thisPlugin);
				}
			}
		}
		free(namelist);
		if(pluginMenu->getPluginCount()==0) {
			puts("Error: Didn't find any plugins");
			pluginList.clear();
			delete pluginMenu;
			pluginMenu = NULL;
		}
	}
}

void Improv::addSrcWidget(SrcWidget* sw)
{
	assert(sw);
	srcWidgetList.append(sw);
}
		
SrcWidget* Improv::getSrcWidget(int sourceId)
{
	QValueList<SrcWidget*>::iterator it;
	for(it = srcWidgetList.begin(); it != srcWidgetList.end(); it++) {
		if((*it)->getSourceId()==sourceId) return (*it);
	}
	fprintf(stderr,"Error: looking for src %d, and couldn't find it!\n",sourceId);
	return (SrcWidget *)NULL;
}

QValueList<SrcWidget*> Improv::getSrcWidgetList(void) {
	assert(!srcWidgetList.isEmpty());
	return(srcWidgetList);
}

SeqWidget* Improv::getSeqWidget(int id) {
	QValueList<SeqWidget*>::iterator it;
	for(it = seqWidgetList.begin(); it != seqWidgetList.end(); it++) {
		if((*it)->getId()==id) {
			return (*it);
		}
	}
	return (SeqWidget *)NULL;
}

SeqWidget* Improv::getSeqWidget(int sourceId, int position, int number=0) {
	int i=0;
	QValueList<SeqWidget*>::iterator it;
	if(sourceId==-1) {
		// get any source widget, at position.
		for(it = seqWidgetList.begin(); it != seqWidgetList.end(); it++) {
			if((*it)->getPosition()==position) {
				if(i==number) return (*it);
				else i++;
			}
		}
	}
	else {
		// get source widget with position & sourceId.
		for(it = seqWidgetList.begin(); it != seqWidgetList.end(); it++) {
			if(((*it)->getSourceId()==sourceId)&&((*it)->getPosition()==position)) {
				if(i==number) return (*it);
				else i++;
			}
		}
	}
	return (SeqWidget *)NULL;
}

int Improv::getWidgetCount(int sourceId, int position) {
	int count=0;
	QValueList<SeqWidget*>::iterator it;
	if(position==0) {
		if(sourceId==-1) return config->getNumSources();
		else return 1;
	}
	else {
		if(sourceId==-1) {
			// get number of sequence widgets, belonging to any source.
			for(it = seqWidgetList.begin(); it != seqWidgetList.end(); it++) {
				if((*it)->getPosition()==position) count++;
			}
		}
		else {
			// get number of sequence widgets, belonging to sourceId.
			for(it = seqWidgetList.begin(); it != seqWidgetList.end(); it++) {
				if(((*it)->getPosition()==position)&&((*it)->getSourceId()==sourceId)) count++;
			}
		}
		return count;
	}
}

QValueList<SeqWidget*> Improv::getSeqWidgetList(void) {
	//assert(seqWidgetList);
	return(seqWidgetList);
}

QPopupMenu* Improv::getPluginMenu(void)
{
	//assert(pluginMenu);
	if(pluginMenu == NULL) { return NULL; }
	else {
		if(pluginMenu->getPluginCount()==0) { return NULL; }
		else { return(pluginMenu->getMainMenu()); }
	}
}
		
void Improv::setMenuId(int id)
{
	this->menuId = id;
}

int Improv::getMenuId(void)
{
	return this->menuId;
}

void Improv::setFPS(int fps)
{
	animThread->setFPS(fps);
}

void Improv::setUnlimitedFPS(bool toggle) {
	animThread->setUnlimitedFPS(toggle);
}

void Improv::copyPictures(Picture *pic)
{
	if(pic) {
		for(QValueList<SeqWidget*>::iterator it = seqWidgetList.begin(); 
				it != seqWidgetList.end(); ++it) {
			(*it)->copyPicture(pic);
		}
	}
}

void Improv::addSeqWidget(SeqWidget* sw)
{
	assert(sw);
	seqWidgetList.append(sw);
}

void Improv::removeSeqWidget(SeqWidget* sw) {
	assert(sw);
	seqWidgetList.remove(sw);
}
		
void Improv::tick(int frames=1)
{
	QValueList<SeqWidget*>::iterator seqIt;
	QValueList<SrcWidget*>::iterator srcIt;

	// Tick all the src widget (if they are ready)
	for(srcIt = srcWidgetList.begin(); srcIt != srcWidgetList.end(); srcIt++) {
		if((*srcIt)->isReady() == true) {
			(*srcIt)->tick(frames);
		}
	}	

	// Tick all the seq widgets (in order of id)
	for(seqIt = seqWidgetList.begin(); seqIt != seqWidgetList.end(); seqIt++) {
		if((*seqIt)->isActive() == true) {
			(*seqIt)->tick();
		}
	}
}

int Improv::getPluginUniqueId(int id)
{
	return pluginMenu->getUniqueId(id);
}

QString* Improv::getPluginName(int id)
{
	PluginMenu::PluginOp* po;

	po = pluginMenu->getPluginOp(id);
	return po->name;
}

int Improv::getPluginInputCount(int id)
{
	PluginMenu::PluginOp* po;

	po = pluginMenu->getPluginOp(id);
	return po->inputCount;
}

int Improv::getPluginNumParams(int id)
{
	PluginMenu::PluginOp* po;

	po = pluginMenu->getPluginOp(id);
	return po->numParams;
}

QStringList* Improv::getPluginParamNames(int id)
{
	PluginMenu::PluginOp* po;
	po = pluginMenu->getPluginOp(id);

	if(po->numParams) {
		return po->paramNames;
	}
	else return 0;
}

void Improv::process(Picture** p_in, Picture* p_out, int menuId, int p_inCount=1, int bufferId=-1, float* params = NULL)
{
	QValueList<Picture **>::iterator buffer;	
	PluginMenu::PluginOp* po;
	ipPlugin* ip;
	void *result=NULL;
	int resultInt = NOERROR;
	QString e;
	bool resultMessage=false;
	
	assert(p_inCount>0);
	
	po = pluginMenu->getPluginOp(menuId);
	assert(po);

	// result handling
	if((po->resultType != NORESULT)&&(po->resultSize>0)) {
		result = malloc(po->resultSize);
		if(result == NULL) {
			fprintf(stderr, "Error: %s - No memory available for plugin function result\n", po->name->latin1());
			return;
		}
	}

	ip = (ipPlugin*) po->plugin;

	/* If neccessary, get the required amount of frames
	 * from the buffer, and then pass this to the function.
	 */
	if((po->inputCount>1)&&(bufferId>=0)) {
		// get the buffer using instanceId
		buffer = frameBuffer.at(bufferId);
		// free the required number of images off the end of the buffer
		for(int j=0,i=po->inputCount-1;j<p_inCount;j++,i--) {
			freeImage((*buffer)[i]);
			//freeImage((*buffer)[po->inputCount-1]);
		}
		// shift all the images along in the buffer
		for(int i=po->inputCount-1;i>0;i=i-p_inCount) {
			(*buffer)[i] = (*buffer)[i-p_inCount];
		}
		// place all the new images at the start of the buffer
		for(int i=0;i<p_inCount;i++) {
			(*buffer)[i] = copyImage(p_in[i]);
		}
		if(po->resultType == NORESULT) {
			ip->processId(po->opId, (*buffer), p_out, params, &resultInt);
		}
		else {
			ip->processId(po->opId, (*buffer), p_out, params, result);
		}
	}
	else {
		if(po->resultType == NORESULT) {
			ip->processId(po->opId, p_in, p_out, params, &resultInt);
		}
		else {
			ip->processId(po->opId, p_in, p_out, params, result);
		}
	}

	switch(po->resultType) {
		case NORESULT:
			if(resultInt != NOERROR) {
				resultMessage = true;
				switch(resultInt) {
					case WRONGFORMAT:
						setStatusResultMessage(e.sprintf("%s: Image is of wrong format",po->name->latin1()).latin1());
						break;
					case TOOLARGE:
						setStatusResultMessage(e.sprintf("%s: Image is too large",po->name->latin1()).latin1());
						break;
					default:
						setStatusResultMessage(e.sprintf("%s: An unknow error occured",po->name->latin1()).latin1());
						break;
				}
			}
			//else setStatusResultMessage("");
			break;
		case TEXT:
			resultMessage = true;
			setStatusResultMessage((char *)result);
			break;
		default:
			setStatusResultMessage("");
	}
	//if(resultMessage==false) setStatusResultMessage("");
}

void Improv::activateParamWidget(int numParams, QStringList* paramNames, float* params)
{
	assert(paramWidget);
	assert(paramNames);

	paramWidget->activateParams(numParams, paramNames, params);
}

void Improv::disableParamWidget(void)
{
	(this->paramWidget)->disable();
}

void Improv::setParamWidget(ParamWidget* pw)
{
	assert(pw);
	this->paramWidget = pw;
}

void Improv::setStatusBar(QStatusBar* statusBar)
{
	assert(statusBar);
	this->statusBar = statusBar;
}

bool Improv::statusBarReady(void)
{
	if(this->statusBar)
		return true;

	return false;
}

void Improv::setStatusMessage(const char* msg)
{
	assert(statusBar);
	statusBarMsg = strdup(msg);
	statusBar->message(msg);
}

void Improv::setStatusMessage(const char* msg, int ms)
{
	assert(statusBar);
	//statusBarMsg = msg;
	statusBar->message(msg,ms);
}

void Improv::setStatusFPSMessage(const char* msg)
{
	QString s;
	assert(statusBar);
	fpsMsg = strdup(msg);
	statusBar->message(s.sprintf("%s %s   %s",statusBarMsg, fpsMsg, resultMsg));
}

void Improv::setStatusResultMessage(const char* msg)
{
	QString s;
	assert(statusBar);
	resultMsg = strdup(msg);
	if(playing)
		statusBar->message(s.sprintf("%s %s   %s",statusBarMsg, fpsMsg, resultMsg));
	else
		statusBar->message(s.sprintf("%s    %s",statusBarMsg, resultMsg));
}

const char *Improv::getStatusMessage(void)
{
	assert(statusBarMsg);
	return statusBarMsg;
}

int Improv::getNumBuffers(void) {
	return frameBuffer.count();
}

void Improv::addFrameBuffer(int id) {
	Picture **buffer;
	PluginMenu::PluginOp* po;
	po = pluginMenu->getPluginOp(id);
	assert(po->inputCount>1);
	buffer = (Picture **)malloc(sizeof(Picture *)*po->inputCount);
	for(int i=0;i<po->inputCount;i++) {
		buffer[i] = newImage();
	}
	frameBuffer.append(buffer);
}

void Improv::removeFrameBuffer(int bufferId, int menuId) {
	if(bufferId>=0) {
		QValueList<Picture **>::iterator buffer;
		PluginMenu::PluginOp* po;
		po = pluginMenu->getPluginOp(menuId);
		buffer = frameBuffer.at(bufferId);
		for(int i=0;i<po->inputCount;i++) {
			freeImage((*buffer)[i]);
		}
	}
}

void Improv::setCamOptAction(QAction *action) {
	camOptionAction = action;
}

void Improv::setCamOpt(ImprovQTCamOpt *options) {
	if(camOptOpen == true) camOptions->close();
	camOptions = options;
	camOptOpen = true;
}

ImprovQTCamOpt *Improv::getCamOpt(void) {
	if(camOptions) return camOptions;
	else return (ImprovQTCamOpt*)NULL;
}

void Improv::setCamOptOpen(bool open) {
	camOptOpen = open;
	camOptionAction->setOn(open);
}

bool Improv::camOptIsOpen(void) {
	return camOptOpen;
}

void Improv::updateCamOpt(imageSeqType type, Camera *camera) {
	if(camOptions->getCameraType()==type) {
		camera->setBrightness(camOptions->getBrightness());
		camera->setContrast(camOptions->getContrast());
		camera->setSaturation(camOptions->getSaturation());
	}
	camOptions->setCameraType(type,camera);
}
