/* Camera.h
 * Last modified:
 * Authors: Daniel Venkitachalam <venki-d@ee.uwa.edu.au>
 *          Leon Koch <leon@redfishsoftware.com.au>
 */
/** @class Camera
 * Abstract camera class.
 */

#ifndef CAMERA_H
#define CAMERA_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "FW/labImage.h"

#define DEFAULT_BLACK 0
#define DEFAULT_WHITE 255
#define DEFAULT_WIDTH 82
#define DEFAULT_HEIGHT 62
#define DEFAULT_BPP 24

typedef unsigned char BYTE;

class Camera {
 public:
	/** @struct CamInfo
	 * Structure used to hold information on the camera's attributes.
	 */
	typedef struct {
		/** width of image, including border */
		int width;
		/** height of image, including border */
		int height;
		/** colour depth (bits per pixel) */
		int bpp;
	} CamInfo;

	/** Return true on success and false if camera is unsupported 
	 * on this machine virtual bool open(const char* device) = 0;
	 */
	virtual bool open(void) = 0;

	/** Releases camera. Must be called if open succeded. */
	virtual void close(void) = 0;

	/** Fetch next frame from camera */
	virtual void read(Picture *p_frame) = 0;

	/** Does this camera return color or grey scale images */
	virtual bool iscolor(void) { return false; };

	/** Get information on image width, height and bits per pixel */
	virtual void get_info(CamInfo *info) = 0;

	/** Set the brightness control of the camera.
	 * Note that for camera with autobrightness chips, this usually has no effect.
	 */
	virtual void setBrightness(int value) = 0;
	/** Set the contrast control of the camera. */
	virtual void setContrast(int value) = 0;
	/** Set the saturation control of the camera. */
	virtual void setSaturation(int value) = 0;
	/** Toggle the autobrightness feature of the camera driver.
	 * Note that if the camera has autobrightness on the chip,
	 * then this can not (usually) be disabled by this function.
	 */
	virtual void setAutoBrightness(bool on) = 0;
	/** Return the brightness control of the camera. */
	virtual int getBrightness(void) = 0;
	/** Return the contrast control of the camera. */
	virtual int getContrast(void) = 0;
	/** Return the saturation control of the camera. */
	virtual int getSaturation(void) = 0;
	/** Return whether the autobrightness feature of the camera driver is enabled or not. */
	virtual bool getAutoBrightness(void) = 0;
	
};

#endif /* Camera_h */
