/*
 *    LK Image Processing Library
 *   Leon Koch - Ciips - UWA, 2002
 *    leon@redfishsoftware.com.au
 *
 */

//#ifdef HAVE_CONFIG_H
//#include <config.h>
//#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <assert.h>

#include "../FW/labImage.h"
#include "../FW/labImgDraw.h"
#include "../improv_plugin.h"
#include "../utils.h"
#include "LK_plugin.h"

#define NUMBER_OF_OPERATIONS 37

IP_Descriptor ipl_plugin =
{
	2,
	"LK Image Processing Library",
	"Leon Koch - Ciips - UWA",
	"2002",
	"leon@redfishsoftware.com.au",
	NUMBER_OF_OPERATIONS,
	NULL              /*  IP_ops, filled out by the _init function  */
};

IP_Handle IP_Init()
{
	IP_Op *op;
	int i = 0;

	/*  Fill out the IP ops descriptions  */
	ipl_plugin.Ops = calloc(NUMBER_OF_OPERATIONS, sizeof(IP_Op));

	/** transformation **/

	/* Flip Horizontal */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Transformation");
	op->Operation  = strdup("Flip Horizontal");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = LK_flip_horizontal;

	/* Flip Vertical */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Transformation");
	op->Operation  = strdup("Flip Vertical");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = LK_flip_vertical;

	/* Resample Image Size */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Transformation");
	op->Operation  = strdup("Half Resolution");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = LK_half_resolution;

	/* Sample a bit plane */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Transformation");
	op->Operation  = strdup("Bit Plane");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 1;
	op->ParamNames = calloc(1, sizeof(char *));
	op->ParamNames[0] = strdup("Bit Level");
	op->resultType = NORESULT;
	op->process    = LK_bit_plane;

	/* remix colours */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Transformation");
	op->Operation  = strdup("Remix colours");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 4;
	op->ParamNames = calloc(4, sizeof(char *));
	op->ParamNames[0] = strdup("R");
	op->ParamNames[1] = strdup("G");
	op->ParamNames[2] = strdup("B");
	op->ParamNames[3] = strdup("A");
	op->resultType = NORESULT;
	op->process    = LK_remix_colours;

	/* brightness */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Transformation");
	op->Operation  = strdup("Brightness");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 1;
	op->ParamNames = calloc(1, sizeof(char *));
	op->ParamNames[0] = strdup("Brightness");
	op->resultType = NORESULT;
	op->process    = LK_brightness;

	/* contrast */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Transformation");
	op->Operation  = strdup("Contrast");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 1;
	op->ParamNames = calloc(1, sizeof(char *));
	op->ParamNames[0] = strdup("Contrast");
	op->resultType = NORESULT;
	op->process    = LK_contrast;

	/* red */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Transformation");
	op->Operation  = strdup("Red Channel");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = LK_red;

	/* green */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Transformation");
	op->Operation  = strdup("Green Channel");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = LK_green;

	/* blue */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Transformation");
	op->Operation  = strdup("Blue Channel");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = LK_blue;

	/* intensity */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Transformation");
	op->Operation  = strdup("Intensity");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = LK_intensity;

	/* saturation */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Transformation");
	op->Operation  = strdup("Saturation");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = LK_saturation;

	/* hue */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Transformation");
	op->Operation  = strdup("Hue");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = LK_hue;

	/** combination **/

	/*  XOR  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Combination");
	op->Operation  = strdup("XOR");
	op->Index      = i;
	op->InputCount = 2;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = LK_xor;

	/* average two images */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Combination");
	op->Operation  = strdup("Average");
	op->Index      = i;
	op->InputCount = 2;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = LK_average;

	/* subtract two images */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Combination");
	op->Operation  = strdup("Subtraction");
	op->Index      = i;
	op->InputCount = 2;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = LK_subtraction;

	/* add two images */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Combination");
	op->Operation  = strdup("Addition");
	op->Index      = i;
	op->InputCount = 2;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = LK_addition;

	/* multiply two images */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Combination");
	op->Operation  = strdup("Multiplication");
	op->Index      = i;
	op->InputCount = 2;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = LK_multiplication;

	/* divide two images */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Combination");
	op->Operation  = strdup("Division");
	op->Index      = i;
	op->InputCount = 2;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = LK_division;

	/* lighten one image using another image */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Combination");
	op->Operation  = strdup("Lighten");
	op->Index      = i;
	op->InputCount = 2;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = LK_lighten;

	/* darken one image using another image */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Combination");
	op->Operation  = strdup("Darken");
	op->Index      = i;
	op->InputCount = 2;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = LK_darken;

	/** spatial filtering **/

	/* high boost 3x3 - the general sharpen filter. */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Edges");
	op->Operation  = strdup("High Boost\t(3x3)");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 1;
	op->ParamNames = calloc(1, sizeof(char *));
	op->ParamNames[0] = strdup("High Boost");
	op->resultType = NORESULT;
	op->process    = LK_highboost3;

	/* mean (low pass) - variable */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Edges");
	op->Operation  = strdup("Mean\t(variable)");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 1;
	op->ParamNames = calloc(1, sizeof(char *));
	op->ParamNames[0] = strdup("Window Size");
	op->resultType = TEXT;
	op->resultSize = sizeof(char)*24;
	op->process    = LK_meanV;

	/** derivative filtering **/

	/* sobel 3x3 */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Edges");
	op->Operation  = strdup("My Sobel\t(3x3)");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = LK_sobel3;

	/* perwitt 3x3 */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Edges");
	op->Operation  = strdup("Perwitt\t(3x3)");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = LK_perwitt3;

	/** advanced **/

	/* Match grey range(t) */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Advanced");
	op->Operation  = strdup("Match grey range\t(t)");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 1;
	op->ParamNames = calloc(1, sizeof(char *));
	op->ParamNames[0] = strdup("T");
	op->resultType = NORESULT;
	op->process    = LK_match_grey_range;

	/* Intensity slice (8 colours) */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Advanced");
	op->Operation  = strdup("Intensity Slice");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = LK_intensityslice;

	/* horizontal blur */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Advanced");
	op->Operation  = strdup("Horizontal Blur");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 1;
	op->ParamNames = calloc(1, sizeof(char *));
	op->ParamNames[0] = strdup("T");
	op->resultType = NORESULT;
	op->process    = LK_hblur;

	/** frequency domain filtering **/

	/* fast fourier transform */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Frequency Filtering");
	op->Operation  = strdup("FFT");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = LK_fft;

	/** binary images **/

	/* label objects with grey scale values */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Binary Images");
	op->Operation  = strdup("Label Objects");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = TEXT;
	op->resultSize = sizeof(char)*33;
	op->process    = LK_labelobjects;

	/* label objects with grey scale values (without result text) */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Binary Images");
	op->Operation  = strdup("Label Objects (without text)");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = LK_labelobjects_wr;

	/* Moment 1: locate the centroid of the binary object */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Binary Images");
	op->Operation  = strdup("Object Centroid");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = LK_object_centroid;

	/* Moment 2: Angle of miniumu inertia. Also uses
	 * LK_locateobject_centroid to find the center of the object. */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Binary Images");
	op->Operation  = strdup("Minimum Inertia");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = LK_minimum_inertia;

	/* volume of an object labeled with a given grey value */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Binary Images");
	op->Operation  = strdup("Object Volume");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = TEXT;
	op->resultSize = sizeof(char)*29;
	op->process    = LK_objectvolume;

	/** Stereo Matching **/

	/* local (area) based stereo matching with unreliable match detector */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Stereo Matching");
	op->Operation  = strdup("Local + Rejection");
	op->Index      = i;
	op->InputCount = 2;
	op->ParamCount = 1;
	op->ParamNames = calloc(1, sizeof(char *));
	op->ParamNames[0] = strdup("d Max");
	op->resultType = NORESULT;
	op->process    = LK_stereoLocal;

	/* local (area) based stereo matching, using bidirectional
	 * matching to detect invalid matches */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Stereo Matching");
	op->Operation  = strdup("Local + Bidirectional Check");
	op->Index      = i;
	op->InputCount = 2;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = LK_stereoLocal_bidirectional;

	/* */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Stereo Matching");
	op->Operation  = strdup("Correlation");
	op->Index      = i;
	op->InputCount = 2;
	op->ParamCount = 1;
	op->ParamNames = calloc(1, sizeof(char *));
	op->ParamNames[0] = strdup("d Max");
	op->resultType = NORESULT;
	op->process    = LK_stereoCorrelation;

	/** Testing **/

	/* test */
//	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
//	op->Category   = strdup("Test");
//	op->Operation  = strdup("Test");
//	op->Index      = i;
//	op->InputCount = 1;
//	op->ParamCount = 0;
//	op->ParamNames = NULL;
//	op->resultType = NORESULT;
//	op->process    = LK_test;

	if( i != NUMBER_OF_OPERATIONS ) {
		printf("LK_plugin.c ERROR: Mismatched number of operations:\n" \
				" (i, NUMBER_OF_OPERATIONS) = (%d, %d)\n", i, NUMBER_OF_OPERATIONS);
		exit(1);
	}

	return NULL;
}


	const IP_Descriptor* 
IP_Descriptor_get(unsigned long Index)
{
	return &ipl_plugin;
}

pluginReturnType LK_flip_horizontal(IP_Handle instance, Picture **in, Picture *out, float *params, void *result)
{
	int i, j, tempIn, tempOut;
	int width, height;
	BYTE *imageOut,*imageIn;
	Picture *p_imageIn = *in;

	imageIn=(BYTE *)p_imageIn->data;
	imageOut=(BYTE *)out->data;

	width=p_imageIn->width;
	height=p_imageIn->height;

	switch(p_imageIn->format) {
		case pix_grey:
			// loop over each row in the image, starting at the first row
			for	(i = 0; i < height - 1; i++) {
				/* for optimisation */
				tempIn = i * width;
				tempOut = (height - 1) * width - tempIn;
				/* loop over the row */
				for (j = 0; j < width - 1; j++) {
					imageOut[tempOut + j] = (BYTE)imageIn[tempIn + j];
				}
			}
			break;
		case pix_rgb24:
			// loop over each row in the image, starting at the first row
			for (i = 0; i < height - 1; i++) {
				/* for optimisation */
				tempIn = i * 3 * width;
				tempOut = (height - 1) * 3 * width - tempIn;
				/* loop over the row */
				for (j = 0; j < 3 * width - 1; j++) {
					imageOut[tempOut + j] = (BYTE)imageIn[tempIn + j];
				}
			}
			break;
		case pix_rgb32:
		case pix_bgr32:
			// loop over each row in the image, starting at the first row
			for (i = 0; i < height - 1; i++) {
				/* for optimisation */
				tempIn = i * 4 * width;
				tempOut = (height - 1) * 4 * width - tempIn;
				/* loop over the row */
				for (j = 0; j < 4 * width - 1; j++) {
					imageOut[tempOut + j] = (BYTE)imageIn[tempIn + j];
				}
			}
			break;
		default:
			return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType LK_flip_vertical(IP_Handle instance, Picture **in, Picture *out, float *params, void *result)
{
	int i, j, tempIn;
	int width, height;
	BYTE *imageOut,*imageIn;
	Picture *p_imageIn = *in;

	imageIn=(BYTE *)p_imageIn->data;
	imageOut=(BYTE *)out->data;

	width=p_imageIn->width;
	height=p_imageIn->height;

	switch(p_imageIn->format) {
		case pix_grey:
			// loop over each row in the image, starting at the first row
			for (i = 0; i < height - 1; i++) {
				/* for optimisation */
				tempIn = i * width;
				/* loop over the row */
				for (j = 0; j < width - 1; j++) {
					imageOut[tempIn + width - j] = (BYTE)imageIn[tempIn + j];
				}
			}
			break;
		case pix_rgb24:
			// loop over each row in the image, starting at the first row
			for (i = 0; i < height - 1; i++) {
				/* for optimisation */
				tempIn = i * 3 * width;
				/* loop over the row */
				for (j = 0; j < width - 1; j++) {
					/* RGB values in order */
					imageOut[tempIn + 3 * (width - j)    ] = (BYTE)imageIn[tempIn + 3 * j    ];
					imageOut[tempIn + 3 * (width - j) + 1] = (BYTE)imageIn[tempIn + 3 * j + 1];
					imageOut[tempIn + 3 * (width - j) + 2] = (BYTE)imageIn[tempIn + 3 * j + 2];
				}
			}
			break;
		case pix_rgb32:
		case pix_bgr32:
			// loop over each row in the image, starting at the first row
			for (i = 0; i < height - 1; i++) {
				/* for optimisation */
				tempIn = i * 4 * width;
				/* loop over the row */
				for (j = 0; j < width - 1; j++) {
					/* RGB values in order */
					imageOut[tempIn + 4 * (width - j)    ] = (BYTE)imageIn[tempIn + 4 * j    ];
					imageOut[tempIn + 4 * (width - j) + 1] = (BYTE)imageIn[tempIn + 4 * j + 1];
					imageOut[tempIn + 4 * (width - j) + 2] = (BYTE)imageIn[tempIn + 4 * j + 2];
					imageOut[tempIn + 4 * (width - j) + 3] = (BYTE)imageIn[tempIn + 4 * j + 3];
				}
			}
			break;
		default:
			return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType LK_half_resolution(IP_Handle instance, Picture **in, Picture *out, float *params, void *result)
{
	int i, j, outWidth, outHeight;
	int width, height;
	BYTE *imageOut,*imageIn;
	Picture *p_imageIn = *in;

	imageIn=(BYTE *)p_imageIn->data;
	imageOut=(BYTE *)out->data;

	width=p_imageIn->width;
	height=p_imageIn->height;

	outWidth = (int)floor(width/2);
	outHeight = (int)floor(height/2);
	
	out->width = outWidth;
	out->height = outHeight;
	
	switch(p_imageIn->format) {
		case pix_grey:
			// loop over each row in the image, starting at the first row
			for (i = 0; i < outHeight; i++) {
				/* loop over the row */
				for (j = 0; j < outWidth; j++) {
					imageOut[i * outWidth + j] = (BYTE)imageIn[i * 2 * width + j * 2];
				}
			}
			break;
		case pix_rgb24:
			// loop over each row in the image, starting at the first row
			for (i = 0; i < outHeight; i++) {
				/* loop over the row */
				for (j = 0; j < outWidth; j++) {
					/* RGB values in order */
					imageOut[i * 3 * outWidth + 3 * j    ] = (BYTE)imageIn[i * 6 * width + j * 6    ];
					imageOut[i * 3 * outWidth + 3 * j + 1] = (BYTE)imageIn[i * 6 * width + j * 6 + 1];
					imageOut[i * 3 * outWidth + 3 * j + 2] = (BYTE)imageIn[i * 6 * width + j * 6 + 2];
				}
			}
			break;
		case pix_rgb32:
		case pix_bgr32:
			// loop over each row in the image, starting at the first row
			for (i = 0; i < outHeight; i++) {
				/* loop over the row */
				for (j = 0; j < outWidth; j++) {
					/* RGB values in order */
					imageOut[i * 4 * outWidth + 4 * j    ] = (BYTE)imageIn[i * 8 * width + j * 8    ];
					imageOut[i * 4 * outWidth + 4 * j + 1] = (BYTE)imageIn[i * 8 * width + j * 8 + 1];
					imageOut[i * 4 * outWidth + 4 * j + 2] = (BYTE)imageIn[i * 8 * width + j * 8 + 2];
					imageOut[i * 4 * outWidth + 4 * j + 3] = (BYTE)imageIn[i * 8 * width + j * 8 + 3];
				}
			}
			break;
		default:
			return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType LK_bit_plane(IP_Handle instance, Picture **in, Picture *out, float *params, void *result)
{
	int i, width, height;
	int mask = 0x01;
	BYTE *imageOut,*imageIn;
	Picture *p_imageIn = *in;

	imageIn=(BYTE *)p_imageIn->data;
	imageOut=(BYTE *)out->data;

	width=p_imageIn->width;
	height=p_imageIn->height;
	
	mask <<= (int)floor(params[0]*8);

	if(p_imageIn->format==pix_grey) {
		for(i=0;i<width*height;i++) {
			imageOut[i] = (imageIn[i] & mask) ? WHITE : BLACK;
		}
		return NOERROR;
	}
	else {
		return WRONGFORMAT;
	}
}

pluginReturnType LK_remix_colours(IP_Handle instance, Picture **in, Picture *out, float *params, void *result)
{
	int i, width, height;
	float rf, gf, bf, af;
	BYTE *imageOut,*imageIn;
	Picture *p_imageIn = *in;

	imageIn=(BYTE *)p_imageIn->data;
	imageOut=(BYTE *)out->data;

	width=p_imageIn->width;
	height=p_imageIn->height;
	
	rf = params[0];
	gf = params[1];
	bf = params[2];
	af = params[3];

	switch(p_imageIn->format) {
		case pix_rgb24:
			for(i=0;i<3*width*height;i+=3) {
				imageOut[i  ] = floor((float)imageIn[i  ]*rf);
				imageOut[i+1] = floor((float)imageIn[i+1]*gf);
				imageOut[i+2] = floor((float)imageIn[i+2]*bf);
			}
			break;
		case pix_rgb32:
			for(i=0;i<4*width*height;i+=4) {
				imageOut[i  ] = floor((float)imageIn[i  ]*rf);
				imageOut[i+1] = floor((float)imageIn[i+1]*gf);
				imageOut[i+2] = floor((float)imageIn[i+2]*bf);
				imageOut[i+3] = floor((float)imageIn[i+3]*af);
			}
			break;
		case pix_bgr32:
			for(i=0;i<4*width*height;i+=4) {
				imageOut[i  ] = floor((float)imageIn[i  ]*bf);
				imageOut[i+1] = floor((float)imageIn[i+1]*gf);
				imageOut[i+2] = floor((float)imageIn[i+2]*rf);
				imageOut[i+3] = floor((float)imageIn[i+3]*af);
			}
			break;
		default:
			return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType LK_brightness(IP_Handle instance, Picture **in, Picture *out, float *params, void *result)
{
	int i, width, height;
	float intensity;
	BYTE *imageOut,*imageIn;
	Picture *p_imageIn = in[0];

	imageIn=(BYTE *)p_imageIn->data;
	imageOut=(BYTE *)out->data;

	width=p_imageIn->width;
	height=p_imageIn->height;

	// scaling factor
	intensity=params[0]*SCALING_BRIGHTNESS;
	
	switch(p_imageIn->format) {
		case pix_grey:
		case pix_rgb24:
		case pix_rgb32:
		case pix_bgr32:
		  for (i=0; i<p_imageIn->datasize; i++) {
				imageOut[i]=(floor((float)imageIn[i]*intensity)>WHITE) ? WHITE : floor((float)imageIn[i]*intensity);
			}
			break;
		default:
			return WRONGFORMAT;
	}
	return NOERROR;
}

/* contrast (gamma) adjustment */
pluginReturnType LK_contrast(IP_Handle instance, Picture **in, Picture *out, float *params, void *result)
{
	int i, width, height;
	float g;
	BYTE *imageOut,*imageIn;
	Picture *p_imageIn = in[0];

	imageIn=(BYTE *)p_imageIn->data;
	imageOut=(BYTE *)out->data;

	width=p_imageIn->width;
	height=p_imageIn->height;

	// scaling factor
	g=params[0]*SCALING_GAMMA;
	
	switch(p_imageIn->format) {
		case pix_grey:
		case pix_rgb24:
		case pix_rgb32:
		case pix_bgr32:
		  for (i=0; i<p_imageIn->datasize; i++) {
				imageOut[i] = 255*pow(((float)imageIn[i]/255),(1/g));
			}
			break;
		default:
			return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType LK_average(IP_Handle instance, Picture **in, Picture *out, float *params, void *result)
{
	int i, j, type=1, width, height;
	BYTE *imageOut,*imageIn1,*imageIn2;
	Picture *p_imageIn1 = in[0];
	Picture *p_imageIn2 = in[1];

	if(p_imageIn2==NULL) return WRONGFORMAT;
	
	imageIn1=(BYTE *)p_imageIn1->data;
	imageIn2=(BYTE *)p_imageIn2->data;
	if(imageIn2==NULL) return WRONGFORMAT;
	imageOut=(BYTE *)out->data;

	width=p_imageIn1->width;
	height=p_imageIn1->height;
	
	// find the type of combination
	
	if((p_imageIn1->bytes_per_pixel==1)&&(p_imageIn2->bytes_per_pixel==1)) type=0;
	if((p_imageIn1->bytes_per_pixel==4)&&(p_imageIn2->bytes_per_pixel==1)) {
		type=1;
	  imageIn1=(BYTE *)p_imageIn2->data;
		imageIn2=(BYTE *)p_imageIn1->data;
	}
	if((p_imageIn1->bytes_per_pixel==4)&&(p_imageIn2->bytes_per_pixel==4)) type=2;

	// Average the images, using different cases depending upon their format.
	switch(type) {
		case 0:
		  for (i=0; i<width*height; i++) {
				imageOut[i]=floor((float)(imageIn1[i]+imageIn2[i])/2);
			}
			break;
		case 1:
		  for (i=0,j=0; i<4*width*height; i+=4) {
				imageOut[i  ]=floor((float)(imageIn1[j]+imageIn2[i  ])/2);
				imageOut[i+1]=floor((float)(imageIn1[j]+imageIn2[i+1])/2);
				imageOut[i+2]=floor((float)(imageIn1[j]+imageIn2[i+2])/2);
				imageOut[i+3]=floor((float)(imageIn1[j]+imageIn2[i+3])/2);
				j++;
			}
			break;
		case 2:
		  for (i=0; i<4*width*height; i++) {
				imageOut[i]=floor((float)(imageIn1[i]+imageIn2[i])/2);
			}
			break;
		default:
			return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType LK_subtraction(IP_Handle instance, Picture **in, Picture *out, float *params, void *result)
{
	int i, j, type=-1, width, height;
	BYTE *imageOut,*imageIn1,*imageIn2;
	Picture *p_imageIn1 = in[0];
	Picture *p_imageIn2 = in[1];

	if(p_imageIn2==NULL) return WRONGFORMAT;
	
	imageIn1=(BYTE *)p_imageIn1->data;
	imageIn2=(BYTE *)p_imageIn2->data;
	if(imageIn2==NULL) return WRONGFORMAT;
	imageOut=(BYTE *)out->data;

	width=p_imageIn1->width;
	height=p_imageIn1->height;
	
	// find the type of combination
	if((p_imageIn1->bytes_per_pixel==1)&&(p_imageIn2->bytes_per_pixel==1)) type=0;
	if((p_imageIn1->bytes_per_pixel==1)&&(p_imageIn2->bytes_per_pixel==4)) type=1;
	if((p_imageIn1->bytes_per_pixel==4)&&(p_imageIn2->bytes_per_pixel==1)) type=2;
	if((p_imageIn1->bytes_per_pixel==4)&&(p_imageIn2->bytes_per_pixel==4)) type=3;

	// Subtract the images, using different cases depending upon their format.
	switch(type) {
		case 0:
		  for (i=0; i<width*height; i++) {
				imageOut[i]=((imageIn1[i]-imageIn2[i])<0) ? BLACK : imageIn1[i]-imageIn2[i];
			}
			break;
		case 1:
		  for (i=0,j=0; i<4*width*height; i+=4) {
				imageOut[i  ]=((imageIn1[j]-imageIn2[i  ])<0) ? BLACK : imageIn1[j]-imageIn2[i  ];
				imageOut[i+1]=((imageIn1[j]-imageIn2[i+1])<0) ? BLACK : imageIn1[j]-imageIn2[i+1];
				imageOut[i+2]=((imageIn1[j]-imageIn2[i+2])<0) ? BLACK : imageIn1[j]-imageIn2[i+2];
				imageOut[i+3]=((imageIn1[j]-imageIn2[i+3])<0) ? BLACK : imageIn1[j]-imageIn2[i+3];
				j++;
			}
			break;
		case 2:
		  for (i=0,j=0; i<4*width*height; i+=4) {
				imageOut[i  ]=((imageIn1[i  ]-imageIn2[j])<0) ? BLACK : imageIn1[i  ]-imageIn2[j];
				imageOut[i+1]=((imageIn1[i+1]-imageIn2[j])<0) ? BLACK : imageIn1[i+1]-imageIn2[j];
				imageOut[i+2]=((imageIn1[i+2]-imageIn2[j])<0) ? BLACK : imageIn1[i+2]-imageIn2[j];
				imageOut[i+2]=((imageIn1[i+3]-imageIn2[j])<0) ? BLACK : imageIn1[i+3]-imageIn2[j];
				j++;
			}
			break;
		case 3:
		  for (i=0; i<4*width*height; i++) {
				imageOut[i]=((imageIn1[i]-imageIn2[i])<0) ? BLACK : imageIn1[i]-imageIn2[i];
			}
			break;
		default:
			return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType LK_addition(IP_Handle instance, Picture **in, Picture *out, float *params, void *result)
{
	int i, j, type=1, width, height;
	BYTE *imageOut,*imageIn1,*imageIn2;
	Picture *p_imageIn1 = in[0];
	Picture *p_imageIn2 = in[1];

	if(p_imageIn2==NULL) return WRONGFORMAT;
	
	imageIn1=(BYTE *)p_imageIn1->data;
	imageIn2=(BYTE *)p_imageIn2->data;
	if(imageIn2==NULL) return WRONGFORMAT;
	imageOut=(BYTE *)out->data;

	width=p_imageIn1->width;
	height=p_imageIn1->height;
	
	// find the type of combination
	if((p_imageIn1->bytes_per_pixel==1)&&(p_imageIn2->bytes_per_pixel==1)) type=0;
	if((p_imageIn1->bytes_per_pixel==1)&&(p_imageIn2->bytes_per_pixel==4)) type=1;
	if((p_imageIn1->bytes_per_pixel==4)&&(p_imageIn2->bytes_per_pixel==1)) type=2;
	if((p_imageIn1->bytes_per_pixel==4)&&(p_imageIn2->bytes_per_pixel==4)) type=3;

	// Add the images, using different cases depending upon their format.
	switch(type) {
		case 0:
		  for (i=0; i<width*height; i++) {
				imageOut[i]=((imageIn1[i]+imageIn2[i])>WHITE) ? WHITE : imageIn1[i]+imageIn2[i];
			}
			break;
		case 1:
		  for (i=0,j=0; i<4*width*height; i+=4) {
				imageOut[i  ]=((imageIn1[j]+imageIn2[i  ])>WHITE) ? WHITE : imageIn1[j]+imageIn2[i  ];
				imageOut[i+1]=((imageIn1[j]+imageIn2[i+1])>WHITE) ? WHITE : imageIn1[j]+imageIn2[i+1];
				imageOut[i+2]=((imageIn1[j]+imageIn2[i+2])>WHITE) ? WHITE : imageIn1[j]+imageIn2[i+2];
				imageOut[i+3]=((imageIn1[j]+imageIn2[i+3])>WHITE) ? WHITE : imageIn1[j]+imageIn2[i+3];
				j++;
			}
			break;
		case 2:
		  for (i=0,j=0; i<4*width*height; i+=4) {
				imageOut[i  ]=((imageIn1[i  ]+imageIn2[j])>WHITE) ? WHITE : imageIn1[i  ]+imageIn2[j];
				imageOut[i+1]=((imageIn1[i+1]+imageIn2[j])>WHITE) ? WHITE : imageIn1[i+1]+imageIn2[j];
				imageOut[i+2]=((imageIn1[i+2]+imageIn2[j])>WHITE) ? WHITE : imageIn1[i+2]+imageIn2[j];
				imageOut[i+3]=((imageIn1[i+3]+imageIn2[j])>WHITE) ? WHITE : imageIn1[i+3]+imageIn2[j];
				j++;
			}
			break;
		case 3:
		  for (i=0; i<4*width*height; i++) {
				imageOut[i]=((imageIn1[i]+imageIn2[i])>WHITE) ? WHITE : imageIn1[i]+imageIn2[i];
			}
			break;
		default:
			return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType LK_multiplication(IP_Handle instance, Picture **in, Picture *out, float *params, void *result)
{
	int i, j, type=1, width, height;
	BYTE *imageOut,*imageIn1,*imageIn2;
	Picture *p_imageIn1 = in[0];
	Picture *p_imageIn2 = in[1];

	if(p_imageIn2==NULL) return WRONGFORMAT;
	
	imageIn1=(BYTE *)p_imageIn1->data;
	imageIn2=(BYTE *)p_imageIn2->data;
	if(imageIn2==NULL) return WRONGFORMAT;
	imageOut=(BYTE *)out->data;

	width=p_imageIn1->width;
	height=p_imageIn1->height;
	
	// find the type of combination
	if((p_imageIn1->bytes_per_pixel==1)&&(p_imageIn2->bytes_per_pixel==1)) type=0;
	if((p_imageIn1->bytes_per_pixel==1)&&(p_imageIn2->bytes_per_pixel==4)) type=1;
	if((p_imageIn1->bytes_per_pixel==4)&&(p_imageIn2->bytes_per_pixel==1)) type=2;
	if((p_imageIn1->bytes_per_pixel==4)&&(p_imageIn2->bytes_per_pixel==4)) type=3;

	// Multiply the images, using different cases depending upon their format.
	switch(type) {
		case 0:
		  for (i=0; i<width*height; i++) {
				imageOut[i]=((imageIn1[i]*imageIn2[i])>WHITE) ? WHITE : imageIn1[i]*imageIn2[i];
			}
			break;
		case 1:
		  for (i=0,j=0; i<4*width*height; i+=4) {
				imageOut[i  ]=((imageIn1[j]*imageIn2[i  ])>WHITE) ? WHITE : imageIn1[j]*imageIn2[i  ];
				imageOut[i+1]=((imageIn1[j]*imageIn2[i+1])>WHITE) ? WHITE : imageIn1[j]*imageIn2[i+1];
				imageOut[i+2]=((imageIn1[j]*imageIn2[i+2])>WHITE) ? WHITE : imageIn1[j]*imageIn2[i+2];
				imageOut[i+3]=((imageIn1[j]*imageIn2[i+3])>WHITE) ? WHITE : imageIn1[j]*imageIn2[i+3];
				j++;
			}
			break;
		case 2:
		  for (i=0,j=0; i<4*width*height; i+=4) {
				imageOut[i  ]=((imageIn1[i  ]*imageIn2[j])>WHITE) ? WHITE : imageIn1[i  ]*imageIn2[j];
				imageOut[i+1]=((imageIn1[i+1]*imageIn2[j])>WHITE) ? WHITE : imageIn1[i+1]*imageIn2[j];
				imageOut[i+2]=((imageIn1[i+2]*imageIn2[j])>WHITE) ? WHITE : imageIn1[i+2]*imageIn2[j];
				imageOut[i+3]=((imageIn1[i+3]*imageIn2[j])>WHITE) ? WHITE : imageIn1[i+3]*imageIn2[j];
				j++;
			}
			break;
		case 3:
		  for (i=0; i<4*width*height; i++) {
				imageOut[i]=((imageIn1[i]*imageIn2[i])>WHITE) ? WHITE : imageIn1[i]*imageIn2[i];
			}
			break;
		default:
			return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType LK_division(IP_Handle instance, Picture **in, Picture *out, float *params, void *result)
{
	int i, j, type=1, width, height;
	float r;
	BYTE *imageOut,*imageIn1,*imageIn2;
	Picture *p_imageIn1 = in[0];
	Picture *p_imageIn2 = in[1];

	if(p_imageIn2==NULL) return WRONGFORMAT;
	
	imageIn1=(BYTE *)p_imageIn1->data;
	imageIn2=(BYTE *)p_imageIn2->data;
	if(imageIn2==NULL) return WRONGFORMAT;
	imageOut=(BYTE *)out->data;

	width=p_imageIn1->width;
	height=p_imageIn1->height;
	
	// find the type of combination
	if((p_imageIn1->bytes_per_pixel==1)&&(p_imageIn2->bytes_per_pixel==1)) type=0;
	if((p_imageIn1->bytes_per_pixel==1)&&(p_imageIn2->bytes_per_pixel==4)) type=1;
	if((p_imageIn1->bytes_per_pixel==4)&&(p_imageIn2->bytes_per_pixel==1)) type=2;
	if((p_imageIn1->bytes_per_pixel==4)&&(p_imageIn2->bytes_per_pixel==4)) type=3;

	// Divide the images, using different cases depending upon their format.
	switch(type) {
		case 0:
		  for (i=0; i<width*height; i++) {
				if(imageIn2[i]!=0) {
					r = imageIn1[i]/imageIn2[i];
					if(r>WHITE) r = WHITE; if(r<BLACK) r = BLACK;
					imageOut[i]=(BYTE)r;
				}
				else imageOut[i]=BLACK;
			}
			break;
		case 1:
		  for (i=0,j=0; i<4*width*height; i+=4) {
				if(imageIn2[i  ]!=0) {
					r = imageIn1[j]/imageIn2[i  ];
					if(r>WHITE) r = WHITE; if(r<BLACK) r = BLACK;
					imageOut[i  ]=(BYTE)r;
				}
				else imageOut[i  ]=BLACK;
				if(imageIn2[i+1]!=0) {
					r = imageIn1[j]/imageIn2[i+1];
					if(r>WHITE) r = WHITE; if(r<BLACK) r = BLACK;
					imageOut[i+1]=(BYTE)r;
				}
				else imageOut[i+1]=BLACK;
				if(imageIn2[i+2]!=0) {
					r = imageIn1[j]/imageIn2[i+2];
					if(r>WHITE) r = WHITE; if(r<BLACK) r = BLACK;
					imageOut[i+2]=(BYTE)r;
				}
				else imageOut[i+2]=BLACK;
				if(imageIn2[i+3]!=0) {
					r = imageIn1[j]/imageIn2[i+3];
					if(r>WHITE) r = WHITE; if(r<BLACK) r = BLACK;
					imageOut[i+3]=(BYTE)r;
				}
				else imageOut[i+3]=BLACK;
				j++;
			}
			break;
		case 2:
		  for (i=0,j=0; i<4*width*height; i+=4) {
				if(imageIn2[j]!=0) {
					r = imageIn1[i  ]/imageIn2[j];
					if(r>WHITE) r = WHITE; if(r<BLACK) r = BLACK;
					imageOut[i  ]=(BYTE)r;
					r = imageIn1[i+1]/imageIn2[j];
					if(r>WHITE) r = WHITE; if(r<BLACK) r = BLACK;
					imageOut[i+1]=(BYTE)r;
					r = imageIn1[i+2]/imageIn2[j];
					if(r>WHITE) r = WHITE; if(r<BLACK) r = BLACK;
					imageOut[i+2]=(BYTE)r;
					r = imageIn1[i+3]/imageIn2[j];
					if(r>WHITE) r = WHITE; if(r<BLACK) r = BLACK;
					imageOut[i+3]=(BYTE)r;
				}
				else {
					imageOut[i  ]=BLACK;
					imageOut[i+1]=BLACK;
					imageOut[i+2]=BLACK;
					imageOut[i+3]=BLACK;
				}
				j++;
			}
			break;
		case 3:
		  for (i=0; i<4*width*height; i++) {
				if(imageIn2[i]!=0) {
					r = imageIn1[i]/imageIn2[i];
					if(r>WHITE) r = WHITE; if(r<BLACK) r = BLACK;
					imageOut[i]=(BYTE)r;
				}
				else imageIn2[i]=BLACK;
			}
			break;
		default:
			return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType LK_lighten(IP_Handle instance, Picture **in, Picture *out, float *params, void *result) {
	int i, j, type=1, width, height;
	BYTE *imageOut,*imageIn1,*imageIn2;
	Picture *p_imageIn1 = in[0];
	Picture *p_imageIn2 = in[1];

	if(p_imageIn2==NULL) return WRONGFORMAT;
	
	imageIn1=(BYTE *)p_imageIn1->data;
	imageIn2=(BYTE *)p_imageIn2->data;
	if(imageIn2==NULL) return WRONGFORMAT;
	imageOut=(BYTE *)out->data;

	width=p_imageIn1->width;
	height=p_imageIn1->height;
	
	// find the type of combination
	if((p_imageIn1->bytes_per_pixel==1)&&(p_imageIn2->bytes_per_pixel==1)) type=0;
	if((p_imageIn1->bytes_per_pixel==1)&&(p_imageIn2->bytes_per_pixel==4)) type=1;
	if((p_imageIn1->bytes_per_pixel==4)&&(p_imageIn2->bytes_per_pixel==1)) type=2;
	if((p_imageIn1->bytes_per_pixel==4)&&(p_imageIn2->bytes_per_pixel==4)) type=3;

	// Merge the images, using different cases depending upon their format.
	switch(type) {
		case 0:
		  for (i=0; i<width*height; i++) {
				imageOut[i]=(imageIn1[i]<imageIn2[i]) ? imageIn2[i] : imageIn1[i];
			}
			break;
		case 1:
		  for (i=0,j=0; i<4*width*height; i+=4) {
				imageOut[i  ]=(imageIn1[j]<imageIn2[i  ]) ? imageIn2[i  ] : imageIn1[j];
				imageOut[i+1]=(imageIn1[j]<imageIn2[i+1]) ? imageIn2[i+1] : imageIn1[j];
				imageOut[i+2]=(imageIn1[j]<imageIn2[i+2]) ? imageIn2[i+2] : imageIn1[j];
				imageOut[i+3]=(imageIn1[j]<imageIn2[i+3]) ? imageIn2[i+3] : imageIn1[j];
				j++;
			}
			break;
		case 2:
		  for (i=0,j=0; i<4*width*height; i+=4) {
				imageOut[i  ]=(imageIn1[i  ]<imageIn2[j]) ? imageIn2[j] : imageIn1[i  ];
				imageOut[i+1]=(imageIn1[i+1]<imageIn2[j]) ? imageIn2[j] : imageIn1[i+1];
				imageOut[i+2]=(imageIn1[i+2]<imageIn2[j]) ? imageIn2[j] : imageIn1[i+2];
				imageOut[i+3]=(imageIn1[i+3]<imageIn2[j]) ? imageIn2[j] : imageIn1[i+3];
				j++;
			}
			break;
		case 3:
		  for (i=0; i<4*width*height; i++) {
				imageOut[i]=(imageIn1[i]<imageIn2[i]) ? imageIn2[i] : imageIn1[i];
			}
			break;
		default:
			return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType LK_darken(IP_Handle instance, Picture **in, Picture *out, float *params, void *result) {
	int i, j, type=1, width, height;
	BYTE *imageOut,*imageIn1,*imageIn2;
	Picture *p_imageIn1 = in[0];
	Picture *p_imageIn2 = in[1];

	if(p_imageIn2==NULL) return WRONGFORMAT;
	
	imageIn1=(BYTE *)p_imageIn1->data;
	imageIn2=(BYTE *)p_imageIn2->data;
	if(imageIn2==NULL) return WRONGFORMAT;
	imageOut=(BYTE *)out->data;

	width=p_imageIn1->width;
	height=p_imageIn1->height;
	
	// find the type of combination
	if((p_imageIn1->bytes_per_pixel==1)&&(p_imageIn2->bytes_per_pixel==1)) type=0;
	if((p_imageIn1->bytes_per_pixel==1)&&(p_imageIn2->bytes_per_pixel==4)) type=1;
	if((p_imageIn1->bytes_per_pixel==4)&&(p_imageIn2->bytes_per_pixel==1)) type=2;
	if((p_imageIn1->bytes_per_pixel==4)&&(p_imageIn2->bytes_per_pixel==4)) type=3;

	// Merge the images, using different cases depending upon their format.
	switch(type) {
		case 0:
		  for (i=0; i<width*height; i++) {
				imageOut[i]=(imageIn1[i]>imageIn2[i]) ? imageIn2[i] : imageIn1[i];
			}
			break;
		case 1:
		  for (i=0,j=0; i<4*width*height; i+=4) {
				imageOut[i  ]=(imageIn1[j]>imageIn2[i  ]) ? imageIn2[i  ] : imageIn1[j];
				imageOut[i+1]=(imageIn1[j]>imageIn2[i+1]) ? imageIn2[i+1] : imageIn1[j];
				imageOut[i+2]=(imageIn1[j]>imageIn2[i+2]) ? imageIn2[i+2] : imageIn1[j];
				imageOut[i+3]=(imageIn1[j]>imageIn2[i+3]) ? imageIn2[i+3] : imageIn1[j];
				j++;
			}
			break;
		case 2:
		  for (i=0,j=0; i<4*width*height; i+=4) {
				imageOut[i  ]=(imageIn1[i  ]>imageIn2[j]) ? imageIn2[j] : imageIn1[i  ];
				imageOut[i+1]=(imageIn1[i+1]>imageIn2[j]) ? imageIn2[j] : imageIn1[i+1];
				imageOut[i+2]=(imageIn1[i+2]>imageIn2[j]) ? imageIn2[j] : imageIn1[i+2];
				imageOut[i+3]=(imageIn1[i+3]>imageIn2[j]) ? imageIn2[j] : imageIn1[i+3];
				j++;
			}
			break;
		case 3:
		  for (i=0; i<4*width*height; i++) {
				imageOut[i]=(imageIn1[i]>imageIn2[i]) ? imageIn2[i] : imageIn1[i];
			}
			break;
		default:
			return WRONGFORMAT;
	}
	return NOERROR;
}

/** Generic function to apply an arbitary (odd) sized Spatial mask to an image. */
/** @todo would be nice if a smaller sized mask were applied to
 * top and bottom lines... problem if mask is not symmetrical.
 * ie. would need non-generic implementation.
 */
void LKi_spatialmask(Picture *pIn, Picture *pOut, int xsize, int ysize, float *mask, int scale) {
	int xboundary, yboundary, width, height, i, x, y;
	float value;
	BYTE *imageOut,*imageIn;
	if((pIn->format==pix_grey)&&((xsize%2)==1)&&((ysize%2)==1)) {
  	xboundary = (int)floor(xsize/2);
		yboundary = (int)floor(ysize/2);
		width=pIn->width;
		height=pIn->height;
		imageIn=(BYTE *)pIn->data;
		imageOut=(BYTE *)pOut->data;
		// set the top edge that we cant process
		for(i=0;i<=yboundary*width;i++) {
			imageOut[i] = imageIn[i];
		}
		// apply the mask to the image
		for(i = yboundary*width+1; i<(height-yboundary)*width; i++) {
			value = (float)BLACK;
			for (y = -yboundary; y <= yboundary; y++) {
				for (x = -xboundary; x <= xboundary; x++) {
					value += imageIn[i+x+y*width] * mask[x+xboundary+(y+yboundary)*ysize];
				}
			}
			if(scale==1) value = value/(xsize*ysize);
			// clip the results
			if(value>WHITE) value = WHITE;
			if(value<BLACK) value = BLACK;
			imageOut[i] = (BYTE)value;
		}
		// set the bottom edge that we cant process
		for(;i<height*width;i++) {
			imageOut[i] = imageIn[i];
		}
	}
	else return;
}

pluginReturnType LK_highboost3(IP_Handle instance, Picture **in, Picture *out, float *params, void *result)
{
	float w = 9*params[0]*SCALING_HIGHBOOST - 1;
	float mask[] = {-1,-1,-1,
	                -1, w,-1,
                  -1,-1,-1};
	if(in[0]->format==pix_grey) {
		LKi_spatialmask(in[0],out,3,3,mask,1);
		return NOERROR;
	}
	else {
		return WRONGFORMAT;
	}
}

/* Also know as low pass filter. */
pluginReturnType LK_meanV(IP_Handle instance, Picture **in, Picture *out, float *params, void *result)
{
	int i;
	float *mask;
	int size = (int)floor(params[0]*SCALING_MEANV);
	if(in[0]->format==pix_grey) {
		if(size%2!=1) size+=1;
		mask = (float *)malloc(sizeof(float)*size*size);
		for(i=0;i<size*size;i++) mask[i] = 1;
		LKi_spatialmask(in[0],out,size,size,mask,1);
		sprintf(result,"Median Window Size = %d\n", size);
		return NOERROR;
	}
	else {
		return WRONGFORMAT;
	}
}

/** Generic function to apply an arbitary (odd) sized Derivative mask to an image. */
void LKi_derivativemask(Picture *pIn, Picture *pOut, int xsize, int ysize, float *xmask, float *ymask, int scale) {
	int xboundary, yboundary, width, height, i, x, y;
	float xvalue, yvalue, value;
	BYTE *imageOut,*imageIn;
	if((pIn->format==pix_grey)&&((xsize%2)==1)&&((ysize%2)==1)) {
  	xboundary = (int)floor(xsize/2);
		yboundary = (int)floor(ysize/2);
		width=pIn->width;
		height=pIn->height;
		imageIn=(BYTE *)pIn->data;
		imageOut=(BYTE *)pOut->data;
		// set the top edge that we cant process
		for(i=0;i<=yboundary*width;i++) {
			imageOut[i] = imageIn[i];
		}
		// apply the mask to the image
		for(i = yboundary*width+1; i<(height-yboundary)*width; i++) {
			xvalue = (float)BLACK;
			yvalue = (float)BLACK;
			for (y = -yboundary; y <= yboundary; y++) {
				for (x = -xboundary; x <= xboundary; x++) {
					xvalue += (int)(imageIn[i+x+y*width] * xmask[x+xboundary+(y+yboundary)*ysize]);
					yvalue += (int)(imageIn[i+x+y*width] * ymask[x+xboundary+(y+yboundary)*ysize]);
				}
			}
			if(scale==1) {
				xvalue = xvalue/(xsize*ysize);
				yvalue = yvalue/(xsize*ysize);
			}
			// clip the results
			value = (abs(xvalue) + abs(yvalue))/3;
			if(value>WHITE) xvalue = WHITE;
			if(value<BLACK) xvalue = BLACK;
			imageOut[i] = (BYTE)value;
		}
		// set the bottom edge that we cant process
		for(;i<height*width;i++) {
			imageOut[i] = imageIn[i];
		}
	}
	else return;
}

pluginReturnType LK_perwitt3(IP_Handle instance, Picture **in, Picture *out, float *params, void *result)
{
	float xmask[] = {-1,-1,-1,
	                  0, 0, 0,
                    1, 1, 1};
	float ymask[] = {-1, 0, 1,
	                 -1, 0, 1,
                   -1, 0, 1};
	LKi_derivativemask(in[0],out,3,3,xmask,ymask,0);
	return NOERROR;
}

pluginReturnType LK_sobel3(IP_Handle instance, Picture **in, Picture *out, float *params, void *result)
{
	float xmask[] = {-1,-2,-1,
	                  0, 0, 0,
	                  1, 2, 1};
	float ymask[] = {-1, 0, 1,
	                 -2, 0, 2,
                   -1, 0, 1};
	LKi_derivativemask(in[0],out,3,3,xmask,ymask,0);
	return NOERROR;
}

pluginReturnType LK_fft(IP_Handle instance, Picture **in, Picture *out, float *params, void *result)
{
	LKi_fft(in[0],out);
	return NOERROR;
}

int powerOfTwo(int x) {
	if(x < 2) return 0;
	if(x & (x-1)) return 0;
	return 1;
}

int nextHighestPowerOfTwo(int x) {
	int i, result=0x1;
	unsigned int mask = 0x80000000;
	for(i=31;i>=0;i--) {
		if(x & mask) break;
		mask >>= 1;
	}
	result <<= i;
	if((x>result)||(result<2)) result <<= 1;
	return result;
}

/** @todo fast fourier transform */
void LKi_fft(Picture *pIn, Picture *pOut) {
	int i, nn, newWidth=0, newHeight=0;
	Picture *resized, *resizedOut=NULL;
	BYTE *imageIn, *imageOut;

	if(powerOfTwo(pIn->width*pIn->height)!=1) {
		// resize the image to a power of 2
		nn = nextHighestPowerOfTwo(pIn->width*pIn->height);
		newHeight = sqrt((nn*pIn->height)/pIn->width);
		newWidth = nn/newHeight;
		resized = copyImage(pIn);
		resizeImage(resized,newWidth,newHeight);
		resizedOut = copyImage(resized);
		imageIn=(BYTE *)resized->data;
		imageOut=(BYTE *)resizedOut->data;
	}
	else {
		nn = pIn->width*pIn->height;
		imageIn=(BYTE *)pIn->data;
		imageOut=(BYTE *)pOut->data;
	}

	// fft
	
	// if neccessary, resize output image back
	if((newWidth>0)&&(newHeight>0)) {
		resizeImage(resizedOut,pIn->width,pIn->height);
		for(i=0;i<pOut->datasize;i++) pOut->data[i] = resizedOut->data[i];
	}
}

void LKi_xor(Picture *p_imageIn1, Picture *p_imageIn2, Picture *out)
{
  int i, j, type=1;
  int width, height;
  BYTE *imageOut,*imageIn1,*imageIn2;

  imageIn1=(BYTE *)p_imageIn1->data;
  imageIn2=(BYTE *)p_imageIn2->data;
  imageOut=(BYTE *)out->data;

  width=p_imageIn1->width;
  height=p_imageIn1->height;

	// find the type of combination
	if((p_imageIn1->bytes_per_pixel==1)&&(p_imageIn2->bytes_per_pixel==1)) type=0;
	if((p_imageIn1->bytes_per_pixel==4)&&(p_imageIn2->bytes_per_pixel==1)) {
		type=1;
	  imageIn1=(BYTE *)p_imageIn2->data;
		imageIn2=(BYTE *)p_imageIn1->data;
	}
	if((p_imageIn1->bytes_per_pixel==4)&&(p_imageIn2->bytes_per_pixel==4)) type=2;

	// XOR the images, using different cases depending upon their format.
	if(type==0) {
	  for (i=0; i<width*height; i++)
			imageOut[i] = imageIn1[i] ^ imageIn2[i];
	}
	if(type==1) {
	  for (i=0,j=0; i<4*width*height; i+=4) {
			imageOut[i  ] = imageIn1[j] ^ imageIn2[i  ];
			imageOut[i+1] = imageIn1[j] ^ imageIn2[i+1];
			imageOut[i+2] = imageIn1[j] ^ imageIn2[i+2];
			imageOut[i+3] = imageIn1[j] ^ imageIn2[i+3];
			j++;
		}
	}
	if(type==2) {
	  for (i=0; i<4*width*height; i++)
			imageOut[i] = imageIn1[i] ^ imageIn2[i];
	}	
}

pluginReturnType LK_xor(IP_Handle instance, Picture **in, Picture *out, float *params, void *result)
{
	int width, height;
	Picture *resized;

	Picture *p_image1 = in[0];
	Picture *p_image2 = in[1];

	width=p_image1->width;
	height=p_image1->height;

	if(p_image2==NULL) { return WRONGFORMAT; }
	else if(((BYTE *)p_image2->data)==NULL) { return WRONGFORMAT; }
	else {
		if((p_image1->width!=p_image2->width)||
			 (p_image1->height!=p_image2->height)) {
			resized = copyImage(p_image2);
			resizeImage(resized,p_image1->width,p_image1->height);
			LKi_xor(p_image1, resized, out);
		}
		else {
			LKi_xor(p_image1, p_image2, out);
		}
		return NOERROR;
	}
}

pluginReturnType LK_match_grey_range(IP_Handle instance, Picture **in, Picture *out, float *params, void *result)
{
  int t,i;
  int width, height;
  BYTE *imageOut,*imageIn;

	Picture *p_imageIn = *in;
  imageIn=(BYTE *)p_imageIn->data;
  imageOut=(BYTE *)out->data;

  width=p_imageIn->width;
  height=p_imageIn->height;

	t = (int)WHITE*params[0];

	if(p_imageIn->format==pix_grey) {
		for (i = 0; i<height*width; i++) {
			if(imageIn[i]==t) {
				imageOut[i] = WHITE;
			}
			else {
				imageOut[i] = BLACK;
			}
		}
		return NOERROR;
	}
	else {
		return WRONGFORMAT;
	}
}

pluginReturnType LK_intensityslice(IP_Handle instance, Picture **in, Picture *out, float *params, void *result) {
	int i, j, width, height;
	BYTE *imageOut,*imageIn, min=0, max=0, r;
	Picture *p_imageIn = *in;
	imageIn=(BYTE *)p_imageIn->data;
	imageOut=(BYTE *)out->data;

	width=p_imageIn->width;
	height=p_imageIn->height;

	if(p_imageIn->format==pix_grey) {
		if(out->format!=pix_bgr32) {
			out->format = pix_bgr32;
			resizeImage(out,width,height);
			imageOut=(BYTE *)out->data;
		}
		for(i=0; i<height*width; i++) {
    	min = (imageIn[i]<min) ? imageIn[i] : min;
    	max = (imageIn[i]>max) ? imageIn[i] : max;
		}
		r = max - min;
		for(i=0, j=0; i<height*width; i++,j+=4) {
			if(imageIn[i]<(BYTE)(r/8)) {
				// black
				imageOut[j  ] = (BYTE)0;
				imageOut[j+1] = (BYTE)0;
				imageOut[j+2] = (BYTE)0;
				imageOut[j+3] = (BYTE)0;
			}
			else if(imageIn[i]<(BYTE)(2*r/8)) {
				// purple
				imageOut[j  ] = (BYTE)255;
				imageOut[j+1] = (BYTE)0;
				imageOut[j+2] = (BYTE)255;
				imageOut[j+3] = (BYTE)0;
			}
			else if(imageIn[i]<(BYTE)(3*r/8)) {
				// blue
				imageOut[j  ] = (BYTE)255;
				imageOut[j+1] = (BYTE)0;
				imageOut[j+2] = (BYTE)0;
				imageOut[j+3] = (BYTE)0;
			}
			else if(imageIn[i]<(BYTE)(4*r/8)) {
				// orange
				imageOut[j  ] = (BYTE)0;
				imageOut[j+1] = (BYTE)128;
				imageOut[j+2] = (BYTE)255;
				imageOut[j+3] = (BYTE)0;
			}
			else if(imageIn[i]<(BYTE)(5*r/8)) {
				// green
				imageOut[j  ] = (BYTE)0;
				imageOut[j+1] = (BYTE)255;
				imageOut[j+2] = (BYTE)0;
				imageOut[j+3] = (BYTE)0;
			}
			else if(imageIn[i]<(BYTE)(6*r/8)) {
				// yellow
				imageOut[j  ] = (BYTE)0;
				imageOut[j+1] = (BYTE)255;
				imageOut[j+2] = (BYTE)255;
				imageOut[j+3] = (BYTE)0;
			}
			else if(imageIn[i]<(BYTE)(7*r/8)) {
				// red
				imageOut[j  ] = (BYTE)0;
				imageOut[j+1] = (BYTE)0;
				imageOut[j+2] = (BYTE)255;
				imageOut[j+3] = (BYTE)0;
			}
			else {
				// white
				imageOut[j  ] = (BYTE)255;
				imageOut[j+1] = (BYTE)255;
				imageOut[j+2] = (BYTE)255;
				imageOut[j+3] = (BYTE)0;
			}
		}
		return NOERROR;
	}
	else {
		return WRONGFORMAT;
	}
}

pluginReturnType LK_labelobjects(IP_Handle instance, Picture **in, Picture *out, float *params, void *result) {
	int objects;
	if(in[0]->format==pix_grey) {
		objects = LKi_labelobjects(in[0],out);
		sprintf(result,"Label Objects: Found %d objects",objects);
		return NOERROR;
	}
	else {
		return WRONGFORMAT;
	}
}

pluginReturnType LK_labelobjects_wr(IP_Handle instance, Picture **in, Picture *out, float *params, void *result) {
	int objects;
	if(in[0]->format==pix_grey) {
		objects = LKi_labelobjects(in[0],out);
		return NOERROR;
	}
	else {
		return WRONGFORMAT;
	}
}

/* The internal labelobjects function - the one that does the the work(tm)! */
int LKi_labelobjects(Picture *pIn, Picture *pOut) {
	int i, object=1;
  int width, height;
  BYTE *imageOut,*imageIn;

  imageIn=(BYTE *)pIn->data;
  imageOut=(BYTE *)pOut->data;
  width=pIn->width;
  height=pIn->height;

	clearImage(pOut);

	if(pIn->format==pix_grey) {
		for(i=0;i<height*width;i++) {
			if((imageIn[i]==WHITE)&&(imageOut[i]==0)) {
				LKi_followobject(i,object,pIn,pOut);
				object++;
				if(object>WHITE) break;
			}
		}
		return object;
	}
	else return 0;
}

/* recursion a-hoy-hoy  =)
 * This function 'follows' an object through an input image, setting
 * the corresponding pixels of the output image to the value 'object'.
 * This is used for LK_labelobjects.
 */
void LKi_followobject(int i, int object, Picture *pIn, Picture *pOut) {
	int width = pIn->width;
	int height = pIn->height;
	if(pOut->data[i]==0) {
		pOut->data[i] = (BYTE)object;
		if(i>=width) {
			if(pIn->data[i-1-width]==WHITE) LKi_followobject(i-1-width,object,pIn,pOut);
			if(pIn->data[i  -width]==WHITE) LKi_followobject(i  -width,object,pIn,pOut);
			if(pIn->data[i+1-width]==WHITE) LKi_followobject(i+1-width,object,pIn,pOut);
		}
		if(pIn->data[i-1]==WHITE) LKi_followobject(i-1,object,pIn,pOut);
		if(pIn->data[i+1]==WHITE) LKi_followobject(i+1,object,pIn,pOut);
		if(i<width*(height-1)) {
			if(pIn->data[i-1+width]==WHITE) LKi_followobject(i-1+width,object,pIn,pOut);
			if(pIn->data[i  +width]==WHITE) LKi_followobject(i  +width,object,pIn,pOut);
			if(pIn->data[i+1+width]==WHITE) LKi_followobject(i+1+width,object,pIn,pOut);
		}
	}	
}

pluginReturnType LK_object_centroid(IP_Handle instance, Picture **in, Picture *out, float *params, void *result) {
	Picture *pIn = in[0];
	int i, j, meanx=0, meany=0, area=0;
	if(pIn->format==pix_grey) {
		for(i=0;i<pIn->height;i++) {
			for(j=0;j<pIn->width;j++) {
				if(pIn->data[i*pIn->width+j]==WHITE) {
					area += 1;
					meanx += j;
					meany += i;
				}
			}
		}
		meanx = meanx/area;
		meany = meany/area;
		if(NOISY) fprintf(stderr,"LK_object_centroid: area= %d meanx= %d meany= %d\n", area, meanx, meany);
		memcpy(out->data, pIn->data,out->datasize);
		out = MONO2BGR32(out);
		// set the centeroid pixel to red
		out->data[4*(meany*pIn->width+meanx)] = 0;
		out->data[4*(meany*pIn->width+meanx)+1] = 0;
		out->data[4*(meany*pIn->width+meanx)+2] = WHITE;
		return NOERROR;
	}
	else
		return WRONGFORMAT;
}

pluginReturnType LK_minimum_inertia(IP_Handle instance, Picture **in, Picture *out, float *params, void *result) {
	Picture *pIn = in[0];
	double a=0.0, b=0.0, c=0.0, sin2theta, cos2theta, theta;
	int i, j, meanx=0, meany=0, area=0, startx=1, starty=1, endx=1, endy=1;
	if(pIn->format==pix_grey) {
		for(i=0;i<pIn->height;i++) {
			for(j=0;j<pIn->width;j++) {
				if(pIn->data[i*pIn->width+j]==WHITE) {
					area += 1;
					meanx += j;
					meany += i;
				}
			}
		}
		if(area<=0) return WRONGFORMAT;
		meanx = meanx/area;
		meany = meany/area;
		for(i=0;i<pIn->height;i++) {
			for(j=0;j<pIn->width;j++) {
				if(pIn->data[i*pIn->width+j]==WHITE) {
					a += pow(j-meanx,2);
					b += i*(j-meanx);
					c += pow(i-meany,2);
				}
			}
		}
		b = 2*b;
		if(b==0 && a==c) {
			// we have a perfectly round object
			theta = 0.0;
		}
		else {
			sin2theta = b / sqrt(pow(b,2) + pow((a-c),2));
			cos2theta = (a-c) / sqrt(pow(b,2) + pow((a-c),2));
			theta = atan2(sin2theta,cos2theta) / 2;
		}
		memcpy(out->data, pIn->data,out->datasize);
		out = MONO2BGR32(out);
		// set the centeroid pixel to red
		out->data[4*(meany*pIn->width+meanx)] = 0;
		out->data[4*(meany*pIn->width+meanx)+1] = 0;
		out->data[4*(meany*pIn->width+meanx)+2] = WHITE;
		// draw the two halves of the line of minimum inertia
		i=0;
		while(startx > 0 && starty > 0 && startx < pIn->width && starty < pIn->height) {
			startx = meanx + i * cos(theta);
			starty = meany + i * sin(theta);
			i++;
		}
		i=0;
		while(endx > 0 && endy > 0 && endx < pIn->width && endy < pIn->height) {
			endx = meanx + i * cos(theta);
			endy = meany + i * sin(theta);
			i--;
		}
		if(NOISY) fprintf(stderr,"LK_minimum_inertia: startx= %d starty= %d endx= %d endy= %d\n", startx, starty, endx, endy);
		setDrawColorImageR(0);
		setDrawColorImageG(0);
		setDrawColorImageB(WHITE);
		drawLineInImage(out, startx, starty, endx, endy, 2);
		return NOERROR;
	}
	else
		return WRONGFORMAT;
}

pluginReturnType LK_objectvolume(IP_Handle instance, Picture **in, Picture *out, float *params, void *result) {
	Picture *pIn = in[0];
	int i, j, area=0;
	if(pIn->format==pix_grey) {
		for(i=0;i<pIn->height;i++) {
			for(j=0;j<pIn->width;j++) {
				if(pIn->data[i*pIn->width+j]==WHITE) {
					area += 1;
				}
			}
		}
		memcpy(out->data, pIn->data,out->datasize);
		if(area<1000000) {
			sprintf(result,"Object Volume: %d pixels", area);
		}
		else {
			sprintf(result," ");
		}
		return NOERROR;
	}
	else
		return WRONGFORMAT;
}

pluginReturnType LK_red(IP_Handle instance, Picture **in, Picture *out, float *params, void *result) {
	int i, width, height;
  BYTE *imageOut,*imageIn;
	Picture *p_imageIn = *in;
  imageIn=(BYTE *)p_imageIn->data;
  imageOut=(BYTE *)out->data;

  width=p_imageIn->width;
  height=p_imageIn->height;

	switch(p_imageIn->format) {
		case pix_rgb24:
			for(i=0; i<3*height*width; i+=3) {
				imageOut[i  ] = (BYTE)imageIn[i];
				imageOut[i+1] = BLACK;
				imageOut[i+2] = BLACK;
			}
			break;		
		case pix_bgr24:
			for(i=0; i<3*height*width; i+=3) {
				imageOut[i  ] = BLACK;
				imageOut[i+1] = BLACK;
				imageOut[i+2] = (BYTE)imageIn[i+2];
			}
			break;		
		case pix_rgb32:
			for(i=0; i<4*height*width; i+=4) {
				imageOut[i  ] = (BYTE)imageIn[i];
				imageOut[i+1] = BLACK;
				imageOut[i+2] = BLACK;
				imageOut[i+3] = BLACK;
			}
			break;		
		case pix_bgr32:
			for(i=0; i<4*height*width; i+=4) {
				imageOut[i  ] = BLACK;
				imageOut[i+1] = BLACK;
				imageOut[i+2] = (BYTE)imageIn[i+2];
				imageOut[i+3] = BLACK;
			}
			break;		
		default:
			return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType LK_green(IP_Handle instance, Picture **in, Picture *out, float *params, void *result) {
	int i, width, height;
  BYTE *imageOut,*imageIn;
	Picture *p_imageIn = *in;
  imageIn=(BYTE *)p_imageIn->data;
  imageOut=(BYTE *)out->data;

  width=p_imageIn->width;
  height=p_imageIn->height;

	switch(p_imageIn->format) {
		case pix_rgb24:
		case pix_bgr24:
			for(i=0; i<3*height*width; i+=3) {
				imageOut[i  ] = BLACK;
				imageOut[i+1] = (BYTE)imageIn[i+1];
				imageOut[i+2] = BLACK;
			}
			break;		
		case pix_rgb32:
		case pix_bgr32:
			for(i=0; i<4*height*width; i+=4) {
				imageOut[i  ] = BLACK;
				imageOut[i+1] = (BYTE)imageIn[i+1];
				imageOut[i+2] = BLACK;
				imageOut[i+3] = BLACK;
			}
			break;		
		default:
			return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType LK_blue(IP_Handle instance, Picture **in, Picture *out, float *params, void *result) {
	int i, width, height;
  BYTE *imageOut,*imageIn;
	Picture *p_imageIn = *in;
  imageIn=(BYTE *)p_imageIn->data;
  imageOut=(BYTE *)out->data;

  width=p_imageIn->width;
  height=p_imageIn->height;

	switch(p_imageIn->format) {
		case pix_rgb24:
			for(i=0; i<3*height*width; i+=3) {
				imageOut[i  ] = BLACK;
				imageOut[i+1] = BLACK;
				imageOut[i+2] = (BYTE)imageIn[i+2];
			}
			break;		
		case pix_bgr24:
			for(i=0; i<3*height*width; i+=3) {
				imageOut[i  ] = (BYTE)imageIn[i];
				imageOut[i+1] = BLACK;
				imageOut[i+2] = BLACK;
			}
			break;		
		case pix_rgb32:
			for(i=0; i<4*height*width; i+=4) {
				imageOut[i  ] = BLACK;
				imageOut[i+1] = BLACK;
				imageOut[i+2] = (BYTE)imageIn[i+2];
				imageOut[i+3] = BLACK;
			}
			break;		
		case pix_bgr32:
			for(i=0; i<4*height*width; i+=4) {
				imageOut[i  ] = (BYTE)imageIn[i];
				imageOut[i+1] = BLACK;
				imageOut[i+2] = BLACK;
				imageOut[i+3] = BLACK;
			}
			break;		
		default:
			return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType LK_hue(IP_Handle instance, Picture **in, Picture *out, float *params, void *result) {
	int i, j, width, height;
  double a, b, c;
	BYTE *imageOut,*imageIn;
	Picture *p_imageIn = *in;
  imageIn=(BYTE *)p_imageIn->data;

  width=p_imageIn->width;
  height=p_imageIn->height;

	out->format = pix_grey;
	resizeImage(out, width, height);
  imageOut=(BYTE *)out->data;

	switch(p_imageIn->format) {
		case pix_rgb24:
		case pix_bgr24:
			for(i=0, j=0; i<height*width; i++,j+=3) {
				a = (double)imageIn[j  ]/255;
				b = (double)imageIn[j+1]/255;
				c = (double)imageIn[j+2]/255;
				imageOut[i] = (BYTE)(acos((0.5*((a-b)+(a-c)))/sqrt(pow(a-b,2)+(a-c)*(b-c)))*255);
			}
			break;		
		case pix_rgb32:
		case pix_bgr32:
			for(i=0,j=0; i<height*width; i++,j+=4) {
				a = (double)imageIn[j  ]/255;
				b = (double)imageIn[j+1]/255;
				c = (double)imageIn[j+2]/255;
				imageOut[i] = (BYTE)(acos((0.5*((a-b)+(a-c)))/sqrt(pow(a-b,2)+(a-c)*(b-c)))*255);
			}
			break;		
		default:
			return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType LK_saturation(IP_Handle instance, Picture **in, Picture *out, float *params, void *result) {
	int i, j, width, height;
  double a, b, c;
	BYTE *imageOut,*imageIn;
	Picture *p_imageIn = *in;
  imageIn=(BYTE *)p_imageIn->data;

  width=p_imageIn->width;
  height=p_imageIn->height;

	out->format = pix_grey;
	resizeImage(out, width, height);
  imageOut=(BYTE *)out->data;

	switch(p_imageIn->format) {
		case pix_rgb24:
		case pix_bgr24:
			for(i=0, j=0; i<height*width; i++,j+=3) {
				a = (double)imageIn[j  ]/255;
				b = (double)imageIn[j+1]/255;
				c = (double)imageIn[j+2]/255;
				imageOut[i] = (BYTE)((1 - (3*MIN(a,MIN(b,c)))/(a + b + c))*255);
			}
			break;		
		case pix_rgb32:
		case pix_bgr32:
			for(i=0,j=0; i<height*width; i++,j+=4) {
				a = (double)imageIn[j  ]/255;
				b = (double)imageIn[j+1]/255;
				c = (double)imageIn[j+2]/255;
				imageOut[i] = (BYTE)((1 - (3*MIN(a,MIN(b,c)))/(a + b + c))*255);
			}
			break;		
		default:
			return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType LK_intensity(IP_Handle instance, Picture **in, Picture *out, float *params, void *result) {
	int i, j, width, height;
  BYTE *imageOut,*imageIn;
	Picture *p_imageIn = *in;
  imageIn=(BYTE *)p_imageIn->data;

  width=p_imageIn->width;
  height=p_imageIn->height;

	out->format = pix_grey;
	resizeImage(out, width, height);
  imageOut=(BYTE *)out->data;

	switch(p_imageIn->format) {
		case pix_rgb24:
		case pix_bgr24:
			for(i=0, j=0; i<height*width; i++,j+=3) {
				imageOut[i] = (BYTE)((float)(imageIn[j] + imageIn[j+1] + imageIn[j+2])/3);
			}
			break;		
		case pix_rgb32:
		case pix_bgr32:
			for(i=0,j=0; i<height*width; i++,j+=4) {
				imageOut[i] = (BYTE)((float)(imageIn[j] + imageIn[j+1] + imageIn[j+2])/3);
			}
			break;		
		default:
			return WRONGFORMAT;
	}
	return NOERROR;
}

int factorial(x) {
	if (x <= 1) return (1);
	return (factorial(x-1) * x);
}

int sum(x) {
	if(x <= 1) return (1);
	return (sum(x-1) + x);
}

/** @todo unfinished */
pluginReturnType LK_hblur(IP_Handle instance, Picture **in, Picture *out, float *params, void *result) {
	int i, j, k, width, height;
	int s, t;
  BYTE *imageOut,*imageIn;
	Picture *p_imageIn = *in;
  imageIn=(BYTE *)p_imageIn->data;
  imageOut=(BYTE *)out->data;

  width=p_imageIn->width;
  height=p_imageIn->height;

	t = (int)params[0]*width;
	
	if(p_imageIn->format==pix_grey) {
		for(i=0; i<height; i++) {
			for(j=0; j<t-1; j++) {
				s = sum(j+1);
				imageOut[i*width+j] = (BYTE)0;
				for(k=0; k<j+1; k++) {
					imageOut[i*width+j] += (BYTE)(imageIn[i*width+j-k]*(k+1)/s);
				}
				//imageOut[i*width+j] = (BYTE)(imageOut[i*width+j]/(j+1));
			}

//			imageOut[i*width  ] = imageIn[i*width];
//			imageOut[i*width+1] = (BYTE)((imageIn[i*width] + imageIn[i*width+1])/2);

			s = sum(t);
			for(j=t-1; j<width; j++) {
				imageOut[i*width+j] = (BYTE)0;
				for(k=0; k<t; k++) {
					imageOut[i*width+j] += (BYTE)(imageIn[i*width+j-k]*(k+1)/s);
				}
				//imageOut[i*width+j] = (BYTE)(imageOut[i*width+j]/t);

				//imageOut[i*width+j] = (BYTE)((imageIn[i*width+j-2] + imageIn[i*width+j-1] + imageIn[i*width+j])/3);
				//imageOut[i*width+j] = (BYTE)(imageIn[i*width+j-2]/6 + imageIn[i*width+j-1]/3 + imageIn[i*width+j]/2);
			}
		}
		return NOERROR;
	}
	else {
			return WRONGFORMAT;
	}
}

/** Calculate the similarity of two points in different images,
 * using the absolute (centrally weighted) difference of all
 * greyscale values in a xsize * ysize window.
 * @todo don't compare across the start/end of different scan lines.
 */
int LKi_similarity(Picture *left, Picture *right, int leftPoint, int rightPoint, int xsize, int ysize) {
	int e=0, x, y, xboundary, yboundary;
	BYTE *imageLeft,*imageRight;
	imageLeft=(BYTE *)left->data;
  imageRight=(BYTE *)right->data;
	if((left->format==pix_grey)&&(right->format==pix_grey)) {
		xboundary = (int)floor(xsize/2);
		yboundary = (int)floor(ysize/2);
		for (y = -yboundary; y <= yboundary; y++) {
			for (x = -xboundary; x <= xboundary; x++) {
//				e += (int)abs(imageLeft[leftPoint+x+y*left->width] - imageRight[rightPoint+x+y*right->width]);
				if(x==0&&y==0) e += (int)abs(imageLeft[leftPoint+x+y*left->width] - imageRight[rightPoint+x+y*right->width])/2;
				else e += (int)abs(imageLeft[leftPoint+x+y*left->width] - imageRight[rightPoint+x+y*right->width])/(2*xsize*ysize-2);
			}
		}
		return e;
	}
	else return -1;
}

/* Calculate the correlation (sum of products) between two
 * images, using an xsize * ysize window.
 */
int LKi_correlation(Picture *left, Picture *right, int leftPoint, int rightPoint, int xsize, int ysize) {
	int c=0, x, y, xboundary, yboundary;
	BYTE *imageLeft,*imageRight;
	imageLeft=(BYTE *)left->data;
  imageRight=(BYTE *)right->data;
	if((left->format==pix_grey)&&(right->format==pix_grey)) {
		xboundary = (int)floor(xsize/2);
		yboundary = (int)floor(ysize/2);
		for (y = -yboundary; y <= yboundary; y++) {
			for (x = -xboundary; x <= xboundary; x++) {
				c += imageLeft[leftPoint+x+y*left->width] * imageRight[rightPoint+x+y*right->width];
			}
		}
		return c;
	}
	else return -1;
}

int LKi_correlationcoefficient(Picture *left, Picture *right, int leftPoint, int rightPoint, int xsize, int ysize, int wbar) {
	int x, y, xboundary, yboundary;
	float c1=0.0, c2=0.0, c3=0.0, fbar=0.0;
	BYTE *imageLeft,*imageRight;
	imageLeft=(BYTE *)left->data;
  imageRight=(BYTE *)right->data;
	if((left->format==pix_grey)&&(right->format==pix_grey)) {
		xboundary = (int)floor(xsize/2);
		yboundary = (int)floor(ysize/2);
		// find fbar of the right image (the one we are searching in).
		// fbar is the average pixel value in the window
		for (y = -yboundary; y <= yboundary; y++) {
			for (x = -xboundary; x <= xboundary; x++) {
				fbar += imageRight[rightPoint+x+y*right->width];
			}
		}
		fbar = (float)fbar/(xsize*ysize);
		// calculate the coefficient
		for (y = -yboundary; y <= yboundary; y++) {
			for (x = -xboundary; x <= xboundary; x++) {
				c1 += (imageRight[rightPoint+x+y*right->width] - fbar)*(imageLeft[leftPoint+x+y*left->width] - wbar);
				c2 += pow(imageRight[rightPoint+x+y*right->width] - fbar,2);
				c3 += pow(imageLeft[leftPoint+x+y*left->width] - wbar,2);
			}
		}
		return (int)(c1/sqrt(c2*c3));
	}
	else return -2;
}

/* Compute the disparity map of a 3D scene from a stereo
 * (left/right) image pair.
 *
 */
pluginReturnType LK_stereoLocal(IP_Handle instance, Picture **in, Picture *out, float *params, void *result) {
	int i, j, k, width, height, e, emin, d, dmax, xsize=3, ysize=3;
  BYTE *imageOut;
	Picture *pLeft = in[0];
	Picture *pRight = in[1];
  imageOut=(BYTE *)out->data;

	if((pLeft->format==pix_grey)&&(pRight->format==pix_grey)&&
	   (pLeft->width==pRight->width)&&(pLeft->height==pRight->height)) {
  	width=pLeft->width;
	  height=pLeft->height;
		dmax = params[0]*width;
		for(i=0; i<height; i++) {
			for(j=0; j<width; j++) {
				d = -1;
				// the maximum possible error
				emin = WHITE*xsize*ysize;
				for(k=j; k<j+dmax&&k<width; k++) {
					e = LKi_similarity(pLeft, pRight, j+i*width, k+i*width, xsize, ysize);
					if(e<emin) {
						emin = e;
						d = k-j;
					}
				}
				if(d>-1) imageOut[j+i*width] = (BYTE)d;
				else imageOut[j+i*width] = BLACK;
			}
		}
		return NOERROR;
	}
	else {
		return WRONGFORMAT;
	}
}

pluginReturnType LK_stereoLocal_bidirectional(IP_Handle instance, Picture **in, Picture *out, float *params, void *result) {
	int i, j, k, width, height, e, emin, d, d_r, dmax=20, xsize=3, ysize=3;
  BYTE *imageOut;
	Picture *pLeft = in[0];
	Picture *pRight = in[1];
  imageOut=(BYTE *)out->data;

	if((pLeft->format==pix_grey)&&(pRight->format==pix_grey)&&
	   (pLeft->width==pRight->width)&&(pLeft->height==pRight->height)) {
  	width=pLeft->width;
	  height=pLeft->height;
		for(i=0; i<height; i++) {
			for(j=0; j<width; j++) {
				d = -1;
				d_r = -1;
				// left to right (forward) matching
				// the maximum possible error
				emin = WHITE*xsize*ysize;
				for(k=j; k<j+dmax&&k<width; k++) {
					e = LKi_similarity(pLeft, pRight, j+i*width, k+i*width, xsize, ysize);
					if(e<emin) {
						emin = e;
						d = k-j;
					}
				}
				// right to left (reverse) matching
				// the maximum possible error
				emin = WHITE*xsize*ysize;
				for(k=d+j; k>=d+j-dmax; k--) {
					e = LKi_similarity(pRight, pLeft, d+j+i*width, k+i*width, xsize, ysize);
					if(e<emin) {
						emin = e;
						d_r = k;
					}
				}
				if((d>-1)&&(d_r>-1)&&(j==d_r)) imageOut[j+i*width] = (BYTE)d;
				else imageOut[j+i*width] = BLACK;
			}
		}
		return NOERROR;
	}
	else {
		return WRONGFORMAT;
	}
}

pluginReturnType LK_stereoCorrelation(IP_Handle instance, Picture **in, Picture *out, float *params, void *result) {
	int i, j, k, width, height, c, cmax, d, dmax, xsize=3, ysize=3, x, y, xboundary, yboundary, wbar=0;
  BYTE *imageOut;
	Picture *pLeft = in[0];
	Picture *pRight = in[1];
  imageOut=(BYTE *)out->data;

	if((pLeft->format==pix_grey)&&(pRight->format==pix_grey)&&
	   (pLeft->width==pRight->width)&&(pLeft->height==pRight->height)) {
		xboundary = (int)floor(xsize/2);
		yboundary = (int)floor(ysize/2);
  	width=pLeft->width;
	  height=pLeft->height;
		dmax = params[0]*width;
		for(i=0; i<height; i++) {
			for(j=0; j<width; j++) {
				d = -1;
				// find wbar of the left image (the one we are searching for).
				// wbar is the average pixel value in the window
				for (y = -yboundary; y <= yboundary; y++) {
					for (x = -xboundary; x <= xboundary; x++) {
						wbar += pRight->data[j+i*width+x+y*width];
					}
				}
				wbar = (float)wbar/(xsize*ysize);

				// the maximum possible correlation
				//cmax = WHITE*xsize*ysize;
				cmax = -1;
				for(k=j; k<j+dmax&&k<width-1; k++) {
					//c = LKi_correlation(pLeft, pRight, j+i*width, k+i*width, xsize, ysize);
					c = LKi_correlationcoefficient(pLeft, pRight, j+i*width, k+i*width, xsize, ysize, wbar);
					if(c>cmax) {
						cmax = c;
						d = k-j;
					}
				}
				if(d>-1) imageOut[j+i*width] = (BYTE)d;
				else imageOut[j+i*width] = BLACK;
			}
		}
		return NOERROR;
	}
	else {
		return WRONGFORMAT;
	}
}

pluginReturnType LK_test(IP_Handle instance, Picture **in, Picture *out, float *params, void *result) {
	int i;
	Picture *pIn = *in;
	if(pIn->format==pix_grey) {
		memcpy(out->data, pIn->data,out->datasize);
		for(i=0; i<pIn->height*pIn->width; i++) {
			if(LKi_similarity(pIn,pIn,i,i,3,3)!=0) exit(EXIT_FAILURE);
		}
		return NOERROR;
	}
	else return WRONGFORMAT;
}

