/* imageSeq.h
 * Last modified: 
 * Authors: Daniel Venkitachalam <venki-d@ee.uwa.edu.au>
 *          Leon Koch <leon@redfishsoftware.com.au>
 */
/** @class imageSeq
 * Class containing image sequence.
 * Images can be either Pixmaps or from a camera.
 */

#ifndef SEQUENCE_H
#define SEQUENCE_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <assert.h>
#include <string.h>
#include <stdlib.h>
#include <dirent.h>

#include "FW/labImage.h"
#include "Camera.h"
#ifdef HAVE_V4L
#include "V4LCamera.h"
#endif
#ifdef HAVE_V4L2
#include "V4L2Camera.h"
#endif
#ifdef HAVE_QUICKCAM
#include "QuickCam.h"
#include "qcam.h"
#endif

#define MAXSEQNAMELEN	256
#define MAXSOURCE		4096

/** @enum imageSeqType
 * The possible type of image sequences.
 */
typedef enum {
	OTHER,	/**< Enum value OTHER. */
#ifdef HAVE_V4L
	V4L,	/**< Enum value V4L. */
#endif
#ifdef HAVE_V4L2
	V4L2,	/**< Enum value V4L2. */
#endif
#ifdef HAVE_QUICKCAM
	QCAM,	/**< Enum value QCAM. */
#endif
	PIXMAP /**< Enum value PIXMAP. */
} imageSeqType;

/** @enum imageSeqLoopType
 * The possible types of sequence looping for a pixmap sequence. 
 */
typedef enum {
	LOOP,	/**< Enum value LOOP.   */
	BOUNCE,	/**< Enum value BOUNCE. */
	NOLOOP	/**< Enum value NOLOOP. */
} imageSeqLoopType;

class imageSeq {

	public:
		/** @enum directionType
		 * The possible directions that the sequence can be played in
		 */
		typedef enum {
			FORWARD,
			BACKWARD
		} directionType;

		/** Constructor. */
		imageSeq(imageSeqType type, const char *path);
		/** Destructor. Deletes the instance of the class. */
		~imageSeq();

		/** Callback function used when scanning directories for image
		 * files. This function returns 1 if the file is accepted, and
		 * 0 otherwise. Currently this function accepts all files except
		 * 'README', '.' and '..'. In future this should only return 1
		 * for those file types imageSeq can actually support.
		 * @param entry The file that is to be evaluated.
		 * @return Whether the file was accepted or not.
		 */
		static int file_select(const struct dirent *entry);
		
		/** Returns the number of frames in this image sequence. */
		int getNumFrames(void);

		/** Returns the next appropriate image.
		 * @param frames Number of frames to step through (default is 1)
		 * @return The request image.
		 */
		Picture* tick(int frames=1);

		/** Returns the current image. */
		Picture* getCurrent(void);

		/** Returns the width of the image (in pixels). */
		int getWidth(void);

		/** Returns the height of the image (in pixels). */
		int getHeight(void);

		/** Returns the current sequence type. @see imageSeqType */
		imageSeqType getSeqType(void);

		/** Indicates whether the camera is ready or not. */
		bool getCameraReady(void);

		/** Returns a pointer to the Camera object. @see Camera */
		Camera *getCamera(void);

		/** Returns the information string, loaded from the README file of the sequence. */
		char *getInfo(void);

	private:
		/** The absolute path to image sequence. */
		char *seqPath;
		/** The information string, loaded from the README file. */
		char seqInfo[MAXSOURCE];
		/** The list of image files that make up the sequence. */
		char **namelist;

		/** */
		imageSeqLoopType loopType;
		/** */
		imageSeqType seqType;
		/** */
		directionType direction;
		/** */
		Picture** pics;
		/** */
		Picture* current;
		/** */
		int currentIndex;
		
		/** */
		int numFrames;
		/** */
		int width;
		/** */
		int height;

		/** */
		void GetSFiles(const char *dirpath, char *sinfo);

		/** */
		Camera *camera;
		/** */
		bool cameraReady;
		
};

#endif

