/* improv_plugin.h
 * Authors: Daniel Venkitachalam <venki-d@ee.uwa.edu.au>
 *          Leon Koch <leon@redfishsoftware.com.au>
 */
/** 
 * @see ipPlugin
 */

#ifndef IMPROV_PLUGIN_H
#define IMPROV_PLUGIN_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "FW/labImage.h"

#define WHITE 255
#define BLACK 0

#ifndef MAX
/** @def MAX(a,b)
 * A macro that returns the maximum of \a a and \a b.
 */
#define MAX(a,b) (a>b?a:b)
#endif

#ifndef MIN
/** @def MIN(a,b)
 * A macro that returns the minimum of \a a and \a b.
 */
#define MIN(a,b) ((a)>=(b)?(b):(a))
#endif

#ifndef _BYTE
#define _BYTE
typedef unsigned char BYTE;
#endif

#ifdef __cplusplus
extern "C" {
#endif

/** @enum pluginResultType
 * These are the possible data types that can be returned
 * by a plugin, in addition to the processed image.
 */
typedef enum {
	TEXT,		/**< Enum value TEXT.     */
	INT,		/**< Enum value INT.      */
	FLOAT,		/**< Enum value FLOAT.    */
	NORESULT	/**< Enum value NORESULT. */
} pluginResultType;

/** @enum pluginReturnType
 * Enum for return code of image operations.
 */
typedef enum {
	WRONGFORMAT,	/**< Enum value WRONGFORMAT. */
	TOOLARGE,		/**< Enum value TOOLARGE.    */
	NOERROR			/**< Enum value NOERROR.     */
} pluginReturnType;

/** @struct Point
 * Structure used to represent a single point (or pixel) in an image.
 */
typedef struct {
	/** X (horizontal) position of the point. */
	short x;
	/** Y (vertical) position of the point. */
	short y;
} Point;

/** @struct Window
 * Structure used to represent a rectangluar window in an image.
 */
typedef struct {
	/** top left corner */
	short x1, y1;
  /** bottom right corner */
	short x2, y2;
} Window;

/** Instance data for the plugin is stored in an IP_Handle type */
typedef void* IP_Handle;

/** @struct IP_Op
 * This structure contains the metadata for each operation within 
 * the image processing plugin. Instances of this structure are 
 * created within the IP_Init() function of the plugin.
 */
typedef struct {
	/** Image processing category */
	const char* Category;
	/** Image processing operation name  */
	const char* Operation;
	/** The index of this operation within the plugin. */
	int Index;
	/** The maximum number of input images the operation */
	int	InputCount;
	/** The number of parameters the operation requires. */
	unsigned long ParamCount;
	/** The names of the operations parameters. */
	char** ParamNames;
	/** The enumerated type of result data this operation will return.
	 * @see pluginResultType
	 */
	pluginResultType resultType;
	/** The ammount of memory required by the operation for it's result data.
	 * This memory is allocated by the application.
	 */
	size_t resultSize;
	/** Pointer to the actual image processing function.
	 * Takes initialized IP_Handle, array of inputs, single (init'd) output and output parameters.
	 */
	pluginReturnType (*process)(IP_Handle instance, Picture **in, Picture *out, float *params, void *result);
	
} IP_Op;

/** @struct IP_Descriptor
 * A structure containing the meta information for the entire 
 * plugin and an array of meta information for each operation.
 * @see IP_Op
 */
typedef struct {
	/** Unique Id number. */
	unsigned long UniqueId;
	/** Plugin name. */
	const char* Name;
	/** Plugin maker. */
	const char* Maker;
	/** Plugin copyright. */
	const char* Copyright;
	/** Plugin comments. */
	const char* Comment;
	/** Number of operations provided by the plugin. */
	unsigned long OpCount;
	/** An array of operations, from 0 to OpCount-1. */
	IP_Op* Ops;
	
} IP_Descriptor;

/** The initialisation function for the plugin. This function 
 * instanciates the IP_Descriptor variable, including the array 
 * of IP_Op structs, containing the meta info on each operation.
 */
IP_Handle IP_init(void);
/** */
const IP_Descriptor* IP_Descriptor_get(unsigned long Index);
/** */
typedef const IP_Descriptor* (*IP_Descriptor_Function)(unsigned long Index);

#ifdef __cplusplus
}
#endif

#endif  // #ifndef IMPROV_PLUGIN_H

