/* QuickCam.cpp
 * Last modified:
 * Authors: Leon Koch <leon@redfishsoftware.com.au>
 */

#include "QuickCam.h"

#ifdef HAVE_QUICKCAM

int QInit ( int *colCam, int *camVer, int *conVer ) {
	/* Quickcam initialisation */
	if (_QCConfig.quickcam_port != 0) {
		if(_QCConfig.quickcam_port != 0x278 &&
		   _QCConfig.quickcam_port != 0x378 &&
		   _QCConfig.quickcam_port != 0x3bc) {
			fprintf(stderr, "Invalid port address\n");
			return (false);
		}
		ioperm(_QCConfig.quickcam_port,3,1);
	}
	else {
		_QCConfig.quickcam_port = QC_find();
		if (_QCConfig.quickcam_port < 0) {
			fprintf( stderr, "Can't get IO perms\n");
			return (false);
		}
		else if(_QCConfig.quickcam_port == 0) {
			fprintf( stderr, "Quickcam not connected\n");
			return (false);
		}
	}
	
	/* reset camera */
	QC_reset(_QCConfig.quickcam_port);

	/* get camera version (colour? b/w?) */
	QC_getversion(_QCConfig.quickcam_port, colCam, camVer, conVer);
	QC_reset(_QCConfig.quickcam_port);

	if(_QCConfig.zoom > 2 || _QCConfig.zoom < 0) {
		fprintf(stderr, "Zoom values are: 0(no zoom), 1(1.5x), 2(2x)\n");
		return (false);
	}

	return (true);
}

//bool QuickCam::open(const char* device) {
bool QuickCam::open(void) {
	int rc;
	int colCam, camVer, conVer;
  
	rc = QC_init();
  
	if (rc < 0) {
		if (rc == -1) fprintf(stderr, "Quickcam already in use ?! (remove /tmp/quickcam and try again)\n");
		return(false);
	}
  
	if (QInit(&colCam, &camVer, &conVer) != 1) {
		fprintf(stderr, "No camera found, exiting...\n");
		QC_exit();
		return(false);
	}                

	if (colCam) {
		_QCConfig.colour = 1;
		printf("Quickcam color camera\n");
	}
	else {
		_QCConfig.colour = 0;
		printf("Quickcam grayscale camera\n");
	}
  
	/* Check for invalid frame modes */
	/* Set decimation mode */
	switch (_QCConfig.zoom) {
		case 0: /* 4:1 decimation */
			_QCConfig.xfermode = 8;
			if((_QCConfig.xsize * 4 > 320) || (_QCConfig.ysize * 4 > 240)) {
				fprintf(stderr, "Image %d x %d too big for camera!\n", _QCConfig.xsize, _QCConfig.ysize);
				QC_exit();
				exit(1);
			}
			break;
		case 1: /* 2:1 decimation */
			_QCConfig.xfermode = 4;
			if((_QCConfig.xsize * 2 > 320) || (_QCConfig.ysize * 2 > 240)) {
				fprintf(stderr, "Image %d x %d too big for camera!\n", _QCConfig.xsize, _QCConfig.ysize);
				QC_exit();
				exit(1);
			}	
			break;
		case 2: /* No decimation */
			_QCConfig.xfermode = 0;
			if((_QCConfig.xsize > 320) || (_QCConfig.ysize > 240)) {
				fprintf(stderr, "Image %d x %d too big for camera!\n", _QCConfig.xsize, _QCConfig.ysize);
				QC_exit();
				exit(1);
			}
			break;
		default: /* Unknown state */
			fprintf(stderr, "Zoom values are: 0(no zoom), 1(1.5x), 2(2x)\n");
			QC_exit();
			exit(1);
	}
  
	if(_QCConfig.colour) {
		/* shift bits appropriately for colour config */
		_QCConfig.xfermode = _QCConfig.xfermode >> 1;
		_QCConfig.xfermode ^= 24;
	}
	else {
		if (_QCConfig.bpp6) _QCConfig.xfermode += 2;
	}
  
	_QCConfig.xfermode += _QCConfig.biMode;

	this->autobrightness = false;

	return(true);
}

void QuickCam::close() {
	QC_reset(_QCConfig.quickcam_port);
	QC_exit();
}

void QuickCam::read(Picture *p_frame) { 
	int expected_size;
	expected_size = p_frame->datasize;

	assert(expected_size>0);
	
	// allocate space for the image
	if (p_frame->data == NULL) {
		p_frame->data = (unsigned char *) malloc(sizeof(unsigned char) * expected_size);
		if(p_frame->data == NULL) {
			fprintf(stderr,"Error: No memory available for QCam picture\n");
			QC_exit();
			return;
		}
	}

	clearImage(p_frame);

	if(_QCConfig.colour) {
		QC_CGet4bppImageUni(&p_frame->data[0]);
	}
	else {
		QC_Get4bppImageUni(&p_frame->data[0]);
	}

	if(this->autobrightness==true) {
		int i;
		long sum = 0;
		double mean;
		static double tempz = 0.0;
		static double tempint = 0.0;
      
		for (i = 0; i < p_frame->datasize; i++) sum += p_frame->data[i];
      
		mean = (double)sum / p_frame->datasize;
      
		if (mean > 8) tempint = - MAX_AUTO_ADJUST * ((mean - 8) / 7);
		else if (mean < 7) tempint = MAX_AUTO_ADJUST * (1 - (mean / 7));
      
		if (mean > 8 || mean < 7) {
			tempz =_QCConfig.brightness;

			tempz += tempint;
	  
			if (tempz >= 255) tempz = 254;
	  
			if (tempz <= 0) tempz = 1;
	  
			//_QCConfig.brightness=(int) tempz;
			//printf("%d",(int) tempz);
			QC_set_brightness();
		}
	}
	else {
		QC_set_brightness();
		QC_set_contrast();
		QC_set_saturation();
	}
}

void QuickCam::get_info(CamInfo *info) {
	info->width  = _QCConfig.xsize; 
	info->height = _QCConfig.ysize;

	if (iscolor() == true) info->bpp = 24;
	else info->bpp = 8; 
}

bool QuickCam::iscolor(void) {
	if(_QCConfig.colour) return(true);
	else return(false);
}

void QuickCam::setBrightness(int value) {
	assert((value>=0)&&(value<=255));
	_QCConfig.brightness = value;
}

void QuickCam::setContrast(int value) {
	assert((value>=0)&&(value<=255));
	_QCConfig.contrast = value;
}

void QuickCam::setSaturation(int value) {
	assert((value>=0)&&(value<=255));
	_QCConfig.saturation = value;
}

void QuickCam::setAutoBrightness(bool on) {
	this->autobrightness = on;
}

int QuickCam::getBrightness(void) {
	return _QCConfig.brightness;
}

int QuickCam::getContrast(void) {
	return _QCConfig.contrast;
}

int QuickCam::getSaturation(void) {
	return _QCConfig.saturation;
}

bool QuickCam::getAutoBrightness(void) {
	return this->autobrightness;
}

#endif /* HAVE_QUICKCAM */
